/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.document;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.integration.cam.CapitalAssetManagementModuleService;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.krad.bo.DocumentHeader;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.maintenance.MaintenanceLock;
import org.kuali.kfs.krad.rules.MaintenanceDocumentRuleBase;
import org.kuali.kfs.krad.rules.rule.BusinessRule;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.KRADServiceLocatorInternal;
import org.kuali.kfs.krad.service.KRADServiceLocatorWeb;
import org.kuali.kfs.krad.service.NoteService;
import org.kuali.kfs.krad.util.ErrorMessage;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsKeyConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetLocationGlobal;
import org.kuali.kfs.module.cam.businessobject.AssetLocationGlobalDetail;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.module.cam.document.validation.impl.AssetLocationGlobalRule;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.FinancialSystemGlobalMaintainable;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.api.exception.WorkflowException;
import org.kuali.kfs.kim.api.identity.IdentityService;
import org.springframework.util.AutoPopulatingList;

import java.sql.Timestamp;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

public class AssetLocationGlobalMaintainableImpl extends FinancialSystemGlobalMaintainable {

    private static final Logger LOG = LogManager.getLogger();
    private AssetService assetService;
    private CapitalAssetManagementModuleService capitalAssetManagementModuleService;
    private ConfigurationService configurationService;
    private DocumentService documentService;
    private DateTimeService dateTimeService;
    private IdentityService identityService;
    private NoteService noteService;

    /**
     * Populates any empty fields from Asset primary key
     */
    @Override
    public void addNewLineToCollection(String collectionName) {
        AssetLocationGlobalDetail addAssetLine = (AssetLocationGlobalDetail) newCollectionLines.get(collectionName);
        Map<String, Object> map = new HashMap<>();
        map.put(CamsPropertyConstants.Asset.CAPITAL_ASSET_NUMBER, addAssetLine.getCapitalAssetNumber());
        Asset asset = getBusinessObjectService().findByPrimaryKey(Asset.class, map);

        if (ObjectUtils.isNotNull(asset) && ObjectUtils.isNotNull(asset.getCapitalAssetNumber())) {
            if (StringUtils.isBlank(addAssetLine.getCampusCode())) {
                addAssetLine.setCampusCode(asset.getCampusCode());
            }
            if (StringUtils.isBlank(addAssetLine.getBuildingCode())) {
                addAssetLine.setBuildingCode(asset.getBuildingCode());
            }
            if (StringUtils.isBlank(addAssetLine.getBuildingRoomNumber())) {
                addAssetLine.setBuildingRoomNumber(asset.getBuildingRoomNumber());
            }
            if (StringUtils.isBlank(addAssetLine.getBuildingSubRoomNumber())) {
                addAssetLine.setBuildingSubRoomNumber(asset.getBuildingSubRoomNumber());
            }
            if (StringUtils.isBlank(addAssetLine.getCampusTagNumber())) {
                addAssetLine.setCampusTagNumber(asset.getCampusTagNumber());
            }
            addAssetLine.setNewCollectionRecord(true);
        }
        super.addNewLineToCollection(collectionName);
    }

    @Override
    public void doRouteStatusChange(DocumentHeader documentHeader) {
        super.doRouteStatusChange(documentHeader);
        WorkflowDocument workflowDoc = documentHeader.getWorkflowDocument();
        // release the lock when document status changed as following...
        if (workflowDoc.isCanceled() || workflowDoc.isDisapproved() || workflowDoc.isProcessed()
                || workflowDoc.isFinal()) {
            getCapitalAssetManagementModuleService().deleteAssetLocks(getDocumentNumber(), null);
        }

        AssetLocationGlobal assetLocationGlobal = (AssetLocationGlobal) getBusinessObject();

        if (workflowDoc.isEnroute()) {
            addNoteIfAssetLocationHasNotChanged(assetLocationGlobal);
        }

        if (workflowDoc.isProcessed()) {
            updateLastInventoryDateIfNecessary(assetLocationGlobal);
        }
    }

    private void addNoteIfAssetLocationHasNotChanged(AssetLocationGlobal assetLocationGlobal) {
        Document document = KRADServiceLocatorInternal.getDocumentDao().findByDocumentHeaderId(
                getDataDictionaryService().getDocumentClassByTypeName(getDocumentTypeName()), getDocumentNumber());

        final String noteTextPattern = getConfigurationService().getPropertyValueAsString(
                CamsKeyConstants.AssetLocationGlobal.WARNING_ASSET_NOT_CHANGED);
        List<AssetLocationGlobalDetail> assetLocationGlobalDetails = assetLocationGlobal.getAssetLocationGlobalDetails();
        for (AssetLocationGlobalDetail detail : assetLocationGlobalDetails) {
            if (!getAssetService().hasCapitalAssetLocationDetailsChanged(detail) && !detail
                    .isUpdateLastInventoryDate()) {
                Object[] arguments = {detail.getCapitalAssetNumber().toString()};
                String noteText = MessageFormat.format(noteTextPattern, arguments);
                Note note = getDocumentService().createNoteFromDocument(document, noteText);
                note.setAuthorUniversalIdentifier(
                        getIdentityService().getPrincipalByPrincipalName(KFSConstants.SYSTEM_USER).getPrincipalId());
                document.addNote(note);
                getDocumentService().saveDocumentNotes(document);
            }
        }
    }

    private void updateLastInventoryDateIfNecessary(AssetLocationGlobal assetLocationGlobal) {
        for (AssetLocationGlobalDetail assetLocationGlobalDetail: assetLocationGlobal.getAssetLocationGlobalDetails()) {
            if (assetLocationGlobalDetail.isUpdateLastInventoryDate()) {
                Asset asset = getBusinessObjectService().findByPrimaryKey(Asset.class,
                        assetLocationGlobalDetail.getPrimaryKeys());
                asset.setLastInventoryDate(new Timestamp(getDateTimeService().getCurrentSqlDate().getTime()));
                getBusinessObjectService().save(asset);
            }
        }
    }

    /**
     * We are using a substitute mechanism for asset locking which can lock on assets when rule check passed.
     *
     * @return empty list from this method.
     */
    @Override
    public List<MaintenanceLock> generateMaintenanceLocks() {
        return new ArrayList<>();
    }

    @Override
    public Map<String, String> populateNewCollectionLines(Map<String, String> fieldValues,
            MaintenanceDocument maintenanceDocument, String methodToCall) {
        String capitalAssetNumber = fieldValues.get(
                CamsPropertyConstants.AssetLocationGlobalDetail.CAPITAL_ASSET_NUMBER);

        if (StringUtils.isNotBlank(capitalAssetNumber)) {
            fieldValues.remove(CamsPropertyConstants.AssetLocationGlobalDetail.CAPITAL_ASSET_NUMBER);
            fieldValues.put(CamsPropertyConstants.AssetLocationGlobalDetail.CAPITAL_ASSET_NUMBER,
                    capitalAssetNumber.trim());
        }
        return super.populateNewCollectionLines(fieldValues, maintenanceDocument, methodToCall);
    }

    @Override
    public Class<? extends PersistableBusinessObject> getPrimaryEditedBusinessObjectClass() {
        return Asset.class;
    }

    /**
     * Verify multiple value lookup entries are authorized by user to add
     */
    @Override
    public void addMultipleValueLookupResults(MaintenanceDocument document, String collectionName,
            Collection<PersistableBusinessObject> rawValues, boolean needsBlank, PersistableBusinessObject bo) {
        Collection<PersistableBusinessObject> allowedAssetsCollection = new ArrayList<>();
        final String maintDocTypeName = CamsConstants.DocumentTypeName.ASSET_EDIT;
        GlobalVariables.getMessageMap().clearErrorMessages();
        for (PersistableBusinessObject businessObject : rawValues) {
            Asset asset = (Asset) businessObject;
            if (StringUtils.isNotBlank(maintDocTypeName)) {
                boolean allowsEdit = getBusinessObjectAuthorizationService().canMaintain(asset,
                        GlobalVariables.getUserSession().getPerson(), maintDocTypeName);
                if (allowsEdit) {
                    allowedAssetsCollection.add(asset);
                } else {
                    GlobalVariables.getMessageMap().putErrorForSectionId(
                            CamsConstants.AssetLocationGlobal.SECTION_ID_EDIT_LIST_OF_ASSETS,
                            CamsKeyConstants.AssetLocationGlobal.ERROR_ASSET_AUTHORIZATION,
                            GlobalVariables.getUserSession().getPerson().getPrincipalName(),
                            asset.getCapitalAssetNumber().toString());
                }
            }
        }
        super.addMultipleValueLookupResults(document, collectionName, allowedAssetsCollection, needsBlank, bo);
    }

    @Override
    public void processAfterRetrieve() {
        super.processAfterRetrieve();
        Document document = KRADServiceLocatorInternal.getDocumentDao().findByDocumentHeaderId(
                getDataDictionaryService().getDocumentClassByTypeName(getDocumentTypeName()), getDocumentNumber());
        processWarningMessages(document);
    }

    private void processWarningMessages(Document document) {
        clearObsoleteWarningMessages();

        if (document != null) {
            WorkflowDocument workflowDoc = document.getDocumentHeader().getWorkflowDocument();

            if (workflowDoc != null && !workflowDoc.isProcessed() && !workflowDoc.isApproved()
                    && !workflowDoc.isFinal()) {
                Class<? extends BusinessRule> ruleClass = getDocumentDictionaryService().getDocumentEntry(
                        getDocumentTypeName()).getBusinessRulesClass();
                AssetLocationGlobalRule rule = (AssetLocationGlobalRule) KRADServiceLocatorWeb.getKualiRuleService()
                        .getBusinessRulesInstance(document, ruleClass);

                List<AssetLocationGlobalDetail> assetLocationGlobalDetails = ((AssetLocationGlobal) businessObject)
                        .getAssetLocationGlobalDetails();
                for (int i = 0; i < assetLocationGlobalDetails.size(); i++) {
                    AssetLocationGlobalDetail detail = assetLocationGlobalDetails.get(i);
                    if (shouldCheckForObsoleteNotes(detail)) {
                        findAndDeleteObsoleteNotes(document, detail);
                    }

                    String errorPath = MaintenanceDocumentRuleBase.MAINTAINABLE_ERROR_PREFIX +
                            CamsPropertyConstants.AssetLocationGlobal.ASSET_LOCATION_GLOBAL_DETAILS +
                            KFSConstants.SQUARE_BRACKET_LEFT + i + KFSConstants.SQUARE_BRACKET_RIGHT;
                    GlobalVariables.getMessageMap().addToErrorPath(errorPath);
                    rule.hasCapitalAssetChangedOrUpdateInventoryDateButtonClicked(detail);
                    GlobalVariables.getMessageMap().removeFromErrorPath(errorPath);
                }
            }
        }
    }

    /**
     * Sometimes processAfterRetrieve() is called without a session. Since checking for obsolete notes uses the
     * session to build the noteText to check, we don't want to do the check without a session.
     *
     * We also only need to check if the updateLastInventoryDate flag has been reset to false.
     *
     * @param detail
     * @return
     */
    private boolean shouldCheckForObsoleteNotes(AssetLocationGlobalDetail detail) {
        return !detail.isUpdateLastInventoryDate() && GlobalVariables.getUserSession() != null;
    }

    private void clearObsoleteWarningMessages() {
        Map<String, AutoPopulatingList<ErrorMessage>> warningMessages = GlobalVariables.getMessageMap()
                .getWarningMessages();
        if (ObjectUtils.isNotNull(warningMessages) && !warningMessages.isEmpty()) {
            warningMessages.entrySet().removeIf(nextWarningMessage -> nextWarningMessage.getValue().get(0)
                    .getErrorKey().equals(CamsKeyConstants.AssetLocationGlobal.WARNING_ASSET_NOT_CHANGED));
        }
    }

    @Override
    public void processAfterPost(MaintenanceDocument document, Map<String, String[]> parameters) {
        super.processAfterPost(document, parameters);

        String customAction = findCustomAction(parameters);

        if (isActionUpdateLastInventoryDate(customAction)) {
            processLastInventoryDateUpdate(document, customAction);
        } else if (wasDeleteLineAction(parameters)) {
            deleteNotesForDeletedAssetsIfNecessary(document);
        } else {
            deleteObsoleteNotes(document);
        }
    }

    private String findCustomAction(Map<String, String[]> parameters) {
        String customAction = null;
        String[] customActions = parameters.get(KRADConstants.CUSTOM_ACTION);
        if (customActions != null) {
            customAction = customActions[0];
        }
        return customAction;
    }

    private boolean isActionUpdateLastInventoryDate(String customAction) {
        if (StringUtils.startsWith(customAction, CamsPropertyConstants.AssetLocationGlobal.ASSET_LOCATION_GLOBAL_DETAILS +
                KFSConstants.SQUARE_BRACKET_LEFT)) {
            String actualAction = customAction.substring(customAction.indexOf(KFSConstants.SQUARE_BRACKET_RIGHT) + 2);
            return StringUtils.equals(CamsPropertyConstants.Asset.LAST_INVENTORY_DATE_UPDATE_BUTTON, actualAction);
        }

        return false;
    }

    private void processLastInventoryDateUpdate(MaintenanceDocument document, String customAction) {
        WorkflowDocument workflowDoc = document.getDocumentHeader().getWorkflowDocument();
        if (workflowDoc != null && workflowDoc.isInitiated() || workflowDoc.isSaved()) {
            int detailNumber = Character.getNumericValue(customAction.charAt(customAction.indexOf(
                    KFSConstants.SQUARE_BRACKET_LEFT) + 1));
            AssetLocationGlobal assetLocationGlobal = (AssetLocationGlobal) document.getDocumentBusinessObject();
            AssetLocationGlobalDetail detail = assetLocationGlobal.getAssetLocationGlobalDetails().get(detailNumber);

            if (getAssetService().hasCapitalAssetLocationDetailsChanged(detail)) {
                String errorPath = MaintenanceDocumentRuleBase.MAINTAINABLE_ERROR_PREFIX +
                        CamsPropertyConstants.AssetLocationGlobal.ASSET_LOCATION_GLOBAL_DETAILS +
                        KFSConstants.SQUARE_BRACKET_LEFT + detailNumber + KFSConstants.SQUARE_BRACKET_RIGHT;
                GlobalVariables.getMessageMap().addToErrorPath(errorPath);
                GlobalVariables.getMessageMap().putWarning(
                        CamsPropertyConstants.AssetLocationGlobal.CAPITAL_ASSET_NUMBER,
                        CamsKeyConstants.AssetLocationGlobal.WARNING_LOCATION_UPDATE_UNNECESSARY,
                        detail.getCapitalAssetNumber().toString());
                GlobalVariables.getMessageMap().removeFromErrorPath(errorPath);
            } else {
                detail.setUpdateLastInventoryDate(true);

                Asset asset = getBusinessObjectService().findByPrimaryKey(Asset.class,
                        assetLocationGlobal.getAssetLocationGlobalDetails().get(detailNumber).getPrimaryKeys());

                String noteText = buildNoteTextForInventoryDateUpdate(asset.getCapitalAssetNumber().toString());
                Note lastInventoryDateUpdatedNote = getDocumentService().createNoteFromDocument(document, noteText);

                if (shouldAddNote(document, noteText)) {
                    lastInventoryDateUpdatedNote.setAuthorUniversalIdentifier(
                        getIdentityService().getPrincipalByPrincipalName(KFSConstants.SYSTEM_USER).getPrincipalId());
                    document.addNote(lastInventoryDateUpdatedNote);
                    getDocumentService().saveDocumentNotes(document);
                }

                processWarningMessages(document);
            }
        }
    }

    private String buildNoteTextForInventoryDateUpdate(String capitalAssetNumber) {
        String userPrincipalName = GlobalVariables.getUserSession().getPrincipalName();
        final String noteTextPattern = getConfigurationService().getPropertyValueAsString(
                CamsKeyConstants.Asset.LAST_INVENTORY_DATE_UPDATE_NOTE_TEXT);
        Object[] arguments = {userPrincipalName, capitalAssetNumber};
        return MessageFormat.format(noteTextPattern, arguments);
    }

    private boolean shouldAddNote(MaintenanceDocument document, String noteText) {
        boolean shouldAddNote = true;

        for (Note note : document.getNotes()) {
            if (note.getNoteText().equals(noteText)) {
                shouldAddNote = false;
                break;
            }
        }

        return shouldAddNote;
    }

    private boolean wasDeleteLineAction(Map<String, String[]> parameters) {
        String parameterName = getMethodToCall(parameters);
        return StringUtils.isNotBlank(parameterName) && parameterName.contains(KRADConstants.DELETE_LINE_METHOD);
    }

    private String getMethodToCall(Map<String, String[]> parameters) {
        for (String key: parameters.keySet()) {
            if (key.startsWith(KRADConstants.DISPATCH_REQUEST_PARAMETER)) {
                return key;
            }
        }
        return null;
    }

    private void deleteNotesForDeletedAssetsIfNecessary(MaintenanceDocument document) {
        for (Iterator<Note> noteIterator = document.getNotes().iterator(); noteIterator.hasNext(); ) {
            Note note = noteIterator.next();
            AssetLocationGlobal assetLocationGlobal = (AssetLocationGlobal) document.getDocumentBusinessObject();
            List<AssetLocationGlobalDetail> assetLocationGlobalDetails =
                    assetLocationGlobal.getAssetLocationGlobalDetails();
            boolean noteMatches = false;
            for (AssetLocationGlobalDetail assetLocationGlobalDetail: assetLocationGlobalDetails) {
                if (note.getNoteText().contains(assetLocationGlobalDetail.getCapitalAssetNumber().toString())) {
                    noteMatches = true;
                    break;
                }
            }
            if (!noteMatches) {
                noteIterator.remove();
                getDocumentService().saveDocumentNotes(document);
                getNoteService().deleteNote(note);
            }
        }
    }

    /**
     * If location details have been updated, the last inventory date will be updated from that. Notes from when the
     * user had previously clicked the update button are no longer needed and should be deleted.
     *
     * @param document
     */
    private void deleteObsoleteNotes(MaintenanceDocument document) {
        List<AssetLocationGlobalDetail> assetLocationGlobalDetails = ((AssetLocationGlobal) businessObject)
                .getAssetLocationGlobalDetails();
        for (AssetLocationGlobalDetail detail: assetLocationGlobalDetails) {
            if (getAssetService().hasCapitalAssetLocationDetailsChanged(detail)) {
                if (detail.isUpdateLastInventoryDate()) {
                    detail.setUpdateLastInventoryDate(false);
                    findAndDeleteObsoleteNotes(document, detail);
                }
            }
        }
    }

    private void findAndDeleteObsoleteNotes(Document document, AssetLocationGlobalDetail detail) {
        String noteText = buildNoteTextForInventoryDateUpdate(detail.getCapitalAssetNumber().toString());

        for (Iterator<Note> noteIterator = document.getNotes().iterator(); noteIterator.hasNext(); ) {
            Note note = noteIterator.next();
            if (note.getNoteText().equals(noteText)) {
                noteIterator.remove();
                getDocumentService().saveDocumentNotes(document);
                getNoteService().deleteNote(note);
            }
        }
    }

    @Override
    protected boolean answerSplitNodeQuestion(String nodeName) throws UnsupportedOperationException {
        if (CamsConstants.RouteLevelNames.ORGANIZATION_INACTIVE.equals(nodeName)) {
            return isRequiresOrganizationInactiveRouteNode();
        }

        throw new UnsupportedOperationException("Cannot answer split question for this node you call \"" + nodeName +
                "\"");
    }

    /**
     * @return if the organization inactive route node needs to be stopped at
     */
    protected boolean isRequiresOrganizationInactiveRouteNode() {
        List<AssetLocationGlobalDetail> assetLocationGlobalDetails = ((AssetLocationGlobal) businessObject)
                .getAssetLocationGlobalDetails();

        List<Long> assets = new ArrayList<>();
        for (AssetLocationGlobalDetail assetLocationGlobalDetail : assetLocationGlobalDetails) {
            Asset asset = assetLocationGlobalDetail.getAsset();

            if (!asset.getOrganizationOwnerAccount().getOrganization().isActive()) {
                assets.add(asset.getCapitalAssetNumber());
            }
        }

        if (assets.isEmpty()) {
            return false;
        } else {
            try {
                this.getDocumentService().getByDocumentHeaderIdSessionless(getDocumentNumber());
            } catch (WorkflowException we) {
                LOG.error("Failed to answerSplitNodeQuestion for following routeNode: " +
                        CamsConstants.RouteLevelNames.ORGANIZATION_INACTIVE, we);
                return false;
            }

            return true;
        }
    }

    public AssetService getAssetService() {
        if (this.assetService == null) {
            this.assetService = SpringContext.getBean(AssetService.class);
        }

        return this.assetService;
    }

    public CapitalAssetManagementModuleService getCapitalAssetManagementModuleService() {
        if (this.capitalAssetManagementModuleService == null) {
            this.capitalAssetManagementModuleService = SpringContext.getBean(CapitalAssetManagementModuleService.class);
        }

        return this.capitalAssetManagementModuleService;
    }

    public ConfigurationService getConfigurationService() {
        if (this.configurationService == null) {
            this.configurationService = SpringContext.getBean(ConfigurationService.class);
        }

        return this.configurationService;
    }

    public DocumentService getDocumentService() {
        if (this.documentService == null) {
            this.documentService = SpringContext.getBean(DocumentService.class);
        }

        return this.documentService;
    }

    public DateTimeService getDateTimeService() {
        if (this.dateTimeService == null) {
            this.dateTimeService = SpringContext.getBean(DateTimeService.class);
        }

        return this.dateTimeService;
    }

    public IdentityService getIdentityService() {
        if (identityService == null) {
            identityService = SpringContext.getBean(IdentityService.class);
        }
        return identityService;
    }

    public NoteService getNoteService() {
        if (noteService == null) {
            noteService = SpringContext.getBean(NoteService.class);
        }
        return noteService;
    }

}
