/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cam.businessobject.lookup;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.datadictionary.legacy.DocumentDictionaryService;
import org.kuali.kfs.integration.cam.CapitalAssetManagementAsset;
import org.kuali.kfs.kim.api.services.KimApiServiceLocator;
import org.kuali.kfs.kim.impl.identity.principal.Principal;
import org.kuali.kfs.kns.document.authorization.MaintenanceDocumentAuthorizer;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.lookup.HtmlData.AnchorHtmlData;
import org.kuali.kfs.kns.lookup.KualiLookupableHelperServiceImpl;
import org.kuali.kfs.kns.web.struts.form.LookupForm;
import org.kuali.kfs.kns.web.ui.Field;
import org.kuali.kfs.kns.web.ui.Row;
import org.kuali.kfs.krad.bo.BusinessObject;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.cam.CamsConstants;
import org.kuali.kfs.module.cam.CamsParameterConstants;
import org.kuali.kfs.module.cam.CamsPropertyConstants;
import org.kuali.kfs.module.cam.batch.AssetDepreciationStep;
import org.kuali.kfs.module.cam.businessobject.Asset;
import org.kuali.kfs.module.cam.businessobject.AssetGlobal;
import org.kuali.kfs.module.cam.businessobject.AssetRetirementGlobal;
import org.kuali.kfs.module.cam.document.service.AssetService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.document.authorization.FinancialSystemTransactionalDocumentAuthorizerBase;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class overrides the base getActionUrls method
 */
public class AssetLookupableHelperServiceImpl extends KualiLookupableHelperServiceImpl {

    protected AssetService assetService;
    protected DocumentDictionaryService documentDictionaryService;

    /**
     * Custom action urls for Asset.
     */
    @Override
    public List<HtmlData> getCustomActionUrls(BusinessObject bo, List pkNames) {
        List<HtmlData> anchorHtmlDataList = super.getCustomActionUrls(bo, pkNames);
        Asset asset = (Asset) bo;

        // For retired asset, all action link will be hidden.
        if (assetService.isAssetRetired(asset)) {
            // clear 'edit' link
            anchorHtmlDataList.clear();
            // add 'view' link instead
            anchorHtmlDataList.add(getViewAssetUrl(asset));
        } else {
            anchorHtmlDataList.add(getLoanUrl(asset));
            anchorHtmlDataList.add(getMergeUrl(asset));
            anchorHtmlDataList.add(getSeparateUrl(asset));
            anchorHtmlDataList.add(getTransferUrl(asset));
        }

        return anchorHtmlDataList;
    }

    protected HtmlData getViewAssetUrl(Asset asset) {
        Map<String, String> parameters = new HashMap<>();
        parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.START_METHOD);
        parameters.put(CamsPropertyConstants.Asset.CAPITAL_ASSET_NUMBER, asset.getCapitalAssetNumber().toString());
        parameters.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, CapitalAssetManagementAsset.class.getName());

        String href = UrlFactory.parameterizeUrl(CamsConstants.INQUIRY_URL, parameters);

        AnchorHtmlData anchorHtmlData = new AnchorHtmlData(href, KFSConstants.START_METHOD,
                CamsConstants.AssetActions.VIEW);
        anchorHtmlData.setTarget("blank");
        return anchorHtmlData;
    }

    protected HtmlData getMergeUrl(Asset asset) {
        MaintenanceDocumentAuthorizer documentAuthorizer = getDocumentAuthorizer(
                CamsConstants.DocumentTypeName.ASSET_RETIREMENT_GLOBAL);
        boolean isAuthorized = documentAuthorizer.isAuthorized(asset, CamsConstants.CAM_MODULE_CODE,
                CamsConstants.PermissionNames.MERGE, determinePrincipalIdForCurrentUser());

        if (isAuthorized) {
            Map<String, String> parameters = new HashMap<>();
            parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.MAINTENANCE_NEW_WITH_EXISTING_ACTION);
            parameters.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, AssetRetirementGlobal.class.getName());
            parameters.put(CamsPropertyConstants.AssetRetirementGlobal.MERGED_TARGET_CAPITAL_ASSET_NUMBER,
                    asset.getCapitalAssetNumber().toString());
            parameters.put(KFSConstants.OVERRIDE_KEYS,
                    CamsPropertyConstants.AssetRetirementGlobal.RETIREMENT_REASON_CODE +
                            KFSConstants.FIELD_CONVERSIONS_SEPERATOR +
                            CamsPropertyConstants.AssetRetirementGlobal.MERGED_TARGET_CAPITAL_ASSET_NUMBER);
            parameters.put(CamsPropertyConstants.AssetRetirementGlobal.RETIREMENT_REASON_CODE,
                    CamsConstants.AssetRetirementReasonCode.MERGED);
            parameters.put(KFSConstants.REFRESH_CALLER,
                    CamsPropertyConstants.AssetRetirementGlobal.RETIREMENT_REASON_CODE + "::" +
                            CamsConstants.AssetRetirementReasonCode.MERGED);

            String href = UrlFactory.parameterizeUrl(KFSConstants.MAINTENANCE_ACTION, parameters);

            return new AnchorHtmlData(href, CamsConstants.AssetActions.MERGE, CamsConstants.AssetActions.MERGE);
        } else {
            return new AnchorHtmlData("", "", "");
        }
    }

    @Override
    protected String getReturnHref(Map<String, String> parameters, LookupForm lookupForm, List returnKeys) {
        String href = super.getReturnHref(parameters, lookupForm, returnKeys);
        href += "&referencesToRefresh=mergedTargetCapitalAsset";
        return href;

    }

    protected HtmlData getLoanUrl(Asset asset) {
        AnchorHtmlData anchorHtmlData;
        List<HtmlData> childURLDataList = new ArrayList<>();

        Map<String, String> parameters = new HashMap<>();
        parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KRADConstants.DOC_HANDLER_METHOD);
        parameters.put(CamsPropertyConstants.AssetTransferDocument.CAPITAL_ASSET_NUMBER,
                asset.getCapitalAssetNumber().toString());
        parameters.put(KFSConstants.PARAMETER_COMMAND, "initiate");
        parameters.put(KFSConstants.DOCUMENT_TYPE_NAME, CamsConstants.DocumentTypeName.ASSET_EQUIPMENT_LOAN_OR_RETURN);

        final String baseUrl = configurationService.getPropertyValueAsString(KFSConstants.APPLICATION_URL_KEY) + "/";
        if (getAssetService().isAssetLoaned(asset)) {
            anchorHtmlData = new AnchorHtmlData("", "", "");

            AnchorHtmlData childURLData = new AnchorHtmlData("", "", CamsConstants.AssetActions.LOAN);
            childURLDataList.add(childURLData);

            parameters.put(CamsConstants.AssetActions.LOAN_TYPE, CamsConstants.AssetActions.LOAN_RENEW);
            String childHref = UrlFactory.parameterizeUrl(baseUrl + CamsConstants.StrutsActions.EQUIPMENT_LOAN_OR_RETURN,
                    parameters);
            childURLData = new AnchorHtmlData(childHref, KRADConstants.DOC_HANDLER_METHOD,
                    CamsConstants.AssetActions.LOAN_RENEW);
            childURLDataList.add(childURLData);

            parameters.remove(CamsConstants.AssetActions.LOAN_TYPE);
            parameters.put(CamsConstants.AssetActions.LOAN_TYPE, CamsConstants.AssetActions.LOAN_RETURN);
            childHref = UrlFactory.parameterizeUrl(baseUrl + CamsConstants.StrutsActions.EQUIPMENT_LOAN_OR_RETURN,
                    parameters);
            childURLData = new AnchorHtmlData(childHref, KRADConstants.DOC_HANDLER_METHOD,
                    CamsConstants.AssetActions.LOAN_RETURN);
            childURLDataList.add(childURLData);

            anchorHtmlData.setChildUrlDataList(childURLDataList);
        } else {
            anchorHtmlData = new AnchorHtmlData("", "", "");
            AnchorHtmlData childURLData;
            if (asset.getCampusTagNumber() == null) {
                childURLData = new AnchorHtmlData("", "", CamsConstants.AssetActions.LOAN);
                childURLDataList.add(childURLData);
            } else {
                parameters.put(CamsConstants.AssetActions.LOAN_TYPE, CamsConstants.AssetActions.LOAN);
                String childHref = UrlFactory.parameterizeUrl(baseUrl +
                        CamsConstants.StrutsActions.EQUIPMENT_LOAN_OR_RETURN, parameters);
                childURLData = new AnchorHtmlData(childHref, KRADConstants.DOC_HANDLER_METHOD,
                        CamsConstants.AssetActions.LOAN);
                childURLDataList.add(childURLData);
            }

            childURLData = new AnchorHtmlData("", "", CamsConstants.AssetActions.LOAN_RENEW);
            childURLDataList.add(childURLData);

            childURLData = new AnchorHtmlData("", "", CamsConstants.AssetActions.LOAN_RETURN);
            childURLDataList.add(childURLData);

            anchorHtmlData.setChildUrlDataList(childURLDataList);
        }

        return anchorHtmlData;
    }

    protected HtmlData getSeparateUrl(Asset asset) {
        MaintenanceDocumentAuthorizer documentAuthorizer = getDocumentAuthorizer(
                CamsConstants.DocumentTypeName.ASSET_ADD_GLOBAL);
        boolean isAuthorized = documentAuthorizer.isAuthorized(asset, CamsConstants.CAM_MODULE_CODE,
                CamsConstants.PermissionNames.SEPARATE, determinePrincipalIdForCurrentUser());

        if (isAuthorized) {
            String href = UrlFactory.parameterizeUrl(KFSConstants.MAINTENANCE_ACTION, getSeparateParameters(asset));
            return new AnchorHtmlData(href, KFSConstants.MAINTENANCE_NEW_METHOD_TO_CALL,
                    CamsConstants.AssetActions.SEPARATE);
        } else {
            return new AnchorHtmlData("", "", "");
        }
    }

    protected Map<String, String> getSeparateParameters(Asset asset) {
        Map<String, String> parameters = new HashMap<>();
        parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.MAINTENANCE_NEW_METHOD_TO_CALL);
        parameters.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, AssetGlobal.class.getName());
        parameters.put(CamsPropertyConstants.AssetGlobal.SEPARATE_SOURCE_CAPITAL_ASSET_NUMBER,
                asset.getCapitalAssetNumber().toString());
        // parameter that tells us this is a separate action. We read this in AssetMaintenanbleImpl.processAfterNew
        parameters.put(KFSPropertyConstants.FINANCIAL_DOCUMENT_TYPE_CODE,
                CamsConstants.PaymentDocumentTypeCodes.ASSET_GLOBAL_SEPARATE);
        return parameters;
    }

    protected HtmlData getTransferUrl(Asset asset) {
        if (isNonCapitalAsset(asset)) {
            return getTransferUrlForNonCapitalAsset(asset);
        } else {
            return getTransferUrlForCapitalAsset(asset);
        }
    }

    private boolean isNonCapitalAsset(Asset asset) {
        Collection<String> nonDepreciableNonCapitalAssetStatusCodes = parameterService.getParameterValuesAsString(
                AssetDepreciationStep.class, CamsParameterConstants.NON_DEPRECIABLE_NON_CAPITAL_ASSET_STATUS_CODES);
        return nonDepreciableNonCapitalAssetStatusCodes.contains(asset.getInventoryStatusCode());
    }

    private HtmlData getTransferUrlForNonCapitalAsset(Asset asset) {
        if (showTransferLinkForNonCapitalAsset(asset)) {
            return buildTransferUrl(asset);
        } else {
            return new AnchorHtmlData("", "", "");
        }
    }

    private boolean showTransferLinkForNonCapitalAsset(Asset asset) {
        String financialObjectSubTypeCode = assetService.determineFinancialObjectSubTypeCode(asset);

        return showTransferLinkForNonCapitalAssetByPayment(financialObjectSubTypeCode);
    }

    private boolean showTransferLinkForNonCapitalAssetByPayment(String financialObjectSubTypeCode) {
        if (ObjectUtils.isNull(financialObjectSubTypeCode)) {
            return true;
        }

        return parameterService.getParameterValuesAsString(Asset.class,
                CamsParameterConstants.NON_CAPITAL_EQUIPMENT_OBJECT_SUB_TYPES).contains(financialObjectSubTypeCode);
    }

    private HtmlData buildTransferUrl(Asset asset) {
        Map<String, String> parameters = new HashMap<>();
        parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KRADConstants.DOC_HANDLER_METHOD);
        parameters.put(CamsPropertyConstants.AssetTransferDocument.CAPITAL_ASSET_NUMBER,
                asset.getCapitalAssetNumber().toString());
        parameters.put(KFSConstants.PARAMETER_COMMAND, "initiate");
        parameters.put(KFSConstants.DOCUMENT_TYPE_NAME, CamsConstants.DocumentTypeName.ASSET_TRANSFER);

        String href = UrlFactory.parameterizeUrl(configurationService
                .getPropertyValueAsString(KFSConstants.APPLICATION_URL_KEY) + "/" +
                CamsConstants.StrutsActions.TRANSFER, parameters);

        return new AnchorHtmlData(href, KRADConstants.DOC_HANDLER_METHOD, CamsConstants.AssetActions.TRANSFER);
    }

    private HtmlData getTransferUrlForCapitalAsset(Asset asset) {
        boolean isAuthorized = true;
        boolean assetMovable = false;
        try {
            assetMovable = getAssetService().isAssetMovableCheckByPayment(asset);
        } catch (ValidationException ve) {
            isAuthorized = false;
        }
        if (!assetMovable) {
            FinancialSystemTransactionalDocumentAuthorizerBase documentAuthorizer =
                    (FinancialSystemTransactionalDocumentAuthorizerBase) getDocumentDictionaryService()
                            .getDocumentAuthorizer(CamsConstants.DocumentTypeName.ASSET_TRANSFER);
            isAuthorized = documentAuthorizer.isAuthorized(asset, CamsConstants.CAM_MODULE_CODE,
                    CamsConstants.PermissionNames.SEPARATE, determinePrincipalIdForCurrentUser());
        }

        if (isAuthorized) {
            return buildTransferUrl(asset);
        } else {
            return new AnchorHtmlData("", "", "");
        }
    }

    /**
     * This method has been extracted primarily so it can be overridden for unit testing purposes.
     *
     * @return
     */
    protected String determinePrincipalIdForCurrentUser() {
        return GlobalVariables.getUserSession().getPerson().getPrincipalId();
    }

    /**
     * Overridden to fix a field conversion
     * <p>
     * KRAD Conversion: Performs customization of the  Row objects to be used to generate the search query screen.
     */
    @Override
    public List<Row> getRows() {
        convertOrganizationOwnerAccountField();
        return super.getRows();
    }

    /**
     * Overridden to fix a field conversion
     */
    @Override
    protected void setRows() {
        super.setRows();
        convertOrganizationOwnerAccountField();
    }

    /**
     * Goes through all the rows, making sure that problematic field conversions are fixed
     * <p>
     * KRAD Conversion: Performs customization of the fields.
     */
    protected void convertOrganizationOwnerAccountField() {
        boolean foundField = false;
        int i = 0;
        while (!foundField && i < super.getRows().size()) {
            final Row r = super.getRows().get(i);
            int j = 0;
            while (!foundField && j < r.getFields().size()) {
                Field f = r.getField(j);
                if (f.getPropertyName().equals(CamsPropertyConstants.Asset.ORGANIZATION_CODE)) {
                    f.setFieldConversions(fixProblematicField(f.getFieldConversions()));
                    f.setLookupParameters(fixProblematicField(f.getLookupParameters()));
                    foundField = true;
                }
                j += 1;
            }
            i += 1;
        }
    }

    /**
     * Fixes the field conversions - replaces "organizationOwnerAccount.chartOfAccountsCode" with
     * "organizationOwnerChartOfAccountsCode"
     *
     * @param problemChildField the original field conversions
     * @return the fixed field conversions
     */
    protected String fixProblematicField(String problemChildField) {
        if (StringUtils.isBlank(problemChildField)) {
            return problemChildField;
        }
        return problemChildField.replace(CamsPropertyConstants.Asset.ORGANIZATION_OWNER_ACCOUNTS_COA_CODE,
                CamsPropertyConstants.Asset.ORGANIZATION_OWNER_CHART_OF_ACCOUNTS_CODE);
    }

    @Override
    protected List<? extends BusinessObject> getSearchResultsHelper(Map<String, String> fieldValues, boolean unbounded) {
        // perform the lookup on the asset representative first
        String principalName = fieldValues.get(CamsPropertyConstants.Asset.REP_USER_AUTH_ID);
        if (StringUtils.isNotBlank(principalName)) {
            Principal principal = KimApiServiceLocator.getIdentityService().getPrincipalByPrincipalName(principalName);

            if (principal == null) {
                return Collections.EMPTY_LIST;
            }
            // place the universal ID into the fieldValues map and remove the dummy attribute
            fieldValues.put(CamsPropertyConstants.Asset.REPRESENTATIVE_UNIVERSAL_IDENTIFIER, principal.getPrincipalId());
            fieldValues.remove(CamsPropertyConstants.Asset.REP_USER_AUTH_ID);
        }

        return super.getSearchResultsHelper(fieldValues, unbounded);
    }

    private MaintenanceDocumentAuthorizer getDocumentAuthorizer(String assetAddGlobal) {
        return (MaintenanceDocumentAuthorizer) getDocumentDictionaryService().getDocumentAuthorizer(
                assetAddGlobal);
    }

    public AssetService getAssetService() {
        return assetService;
    }

    public void setAssetService(AssetService assetService) {
        this.assetService = assetService;
    }

    public DocumentDictionaryService getDocumentDictionaryService() {
        return documentDictionaryService;
    }

    public void setDocumentDictionaryService(DocumentDictionaryService documentDictionaryService) {
        this.documentDictionaryService = documentDictionaryService;
    }

}
