/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2017 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.kuali.kfs.module.cg.businessobject;

import org.apache.commons.lang.StringUtils;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.service.ModuleService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.rice.core.api.mo.common.active.MutableInactivatable;
import org.kuali.rice.location.api.LocationConstants;
import org.kuali.rice.location.framework.country.CountryEbo;
import org.kuali.rice.location.framework.state.StateEbo;

import java.util.HashMap;
import java.util.Map;

/**
 * Subcontractors are vendors involved with an awarded {@link Proposal}.
 */
public class SubContractor extends PersistableBusinessObjectBase implements MutableInactivatable {
    protected String subcontractorNumber;
    protected String subcontractorName;
    protected String subcontractorAddressLine1;
    protected String subcontractorAddressLine2;
    protected String subcontractorCity;
    protected String subcontractorStateCode;
    protected String subcontractorZipCode;
    protected String subcontractorCountryCode;
    protected boolean active;

    protected StateEbo subcontractorState;
    protected CountryEbo subcontractorCountry;

    public static final String CACHE_NAME = KFSConstants.APPLICATION_NAMESPACE_CODE + "/" + "SubContractor";

    /**
     * Gets the subcontractorNumber attribute.
     *
     * @return Returns the subcontractorNumber
     */
    public String getSubcontractorNumber() {
        return subcontractorNumber;
    }

    /**
     * Sets the subcontractorNumber attribute.
     *
     * @param subcontractorNumber The subcontractorNumber to set.
     */
    public void setSubcontractorNumber(String subcontractorNumber) {
        this.subcontractorNumber = subcontractorNumber;
    }

    /**
     * Gets the subcontractorName attribute.
     *
     * @return Returns the subcontractorName
     */
    public String getSubcontractorName() {
        return subcontractorName;
    }

    /**
     * Sets the subcontractorName attribute.
     *
     * @param subcontractorName The subcontractorName to set.
     */
    public void setSubcontractorName(String subcontractorName) {
        this.subcontractorName = subcontractorName;
    }

    /**
     * Gets the subcontractorAddressLine1 attribute.
     *
     * @return Returns the subcontractorAddressLine1
     */
    public String getSubcontractorAddressLine1() {
        return subcontractorAddressLine1;
    }

    /**
     * Sets the subcontractorAddressLine1 attribute.
     *
     * @param subcontractorAddressLine1 The subcontractorAddressLine1 to set.
     */
    public void setSubcontractorAddressLine1(String subcontractorAddressLine1) {
        this.subcontractorAddressLine1 = subcontractorAddressLine1;
    }

    /**
     * Gets the subcontractorAddressLine2 attribute.
     *
     * @return Returns the subcontractorAddressLine2
     */
    public String getSubcontractorAddressLine2() {
        return subcontractorAddressLine2;
    }

    /**
     * Sets the subcontractorAddressLine2 attribute.
     *
     * @param subcontractorAddressLine2 The subcontractorAddressLine2 to set.
     */
    public void setSubcontractorAddressLine2(String subcontractorAddressLine2) {
        this.subcontractorAddressLine2 = subcontractorAddressLine2;
    }

    /**
     * Gets the subcontractorCity attribute.
     *
     * @return Returns the subcontractorCity
     */
    public String getSubcontractorCity() {
        return subcontractorCity;
    }

    /**
     * Sets the subcontractorCity attribute.
     *
     * @param subcontractorCity The subcontractorCity to set.
     */
    public void setSubcontractorCity(String subcontractorCity) {
        this.subcontractorCity = subcontractorCity;
    }

    /**
     * Gets the subcontractorStateCode attribute.
     *
     * @return Returns the subcontractorStateCode.
     */
    public String getSubcontractorStateCode() {
        return subcontractorStateCode;
    }

    /**
     * Sets the subcontractorStateCode attribute value.
     *
     * @param subcontractorStateCode The subcontractorStateCode to set.
     */
    public void setSubcontractorStateCode(String subcontractorStateCode) {
        this.subcontractorStateCode = subcontractorStateCode;
    }

    /**
     * Gets the subcontractorZipCode attribute.
     *
     * @return Returns the subcontractorZipCode
     */
    public String getSubcontractorZipCode() {
        return subcontractorZipCode;
    }

    /**
     * Sets the subcontractorZipCode attribute.
     *
     * @param subcontractorZipCode The subcontractorZipCode to set.
     */
    public void setSubcontractorZipCode(String subcontractorZipCode) {
        this.subcontractorZipCode = subcontractorZipCode;
    }

    /**
     * Gets the subcontractorCountryCode attribute.
     *
     * @return Returns the subcontractorCountryCode
     */
    public String getSubcontractorCountryCode() {
        return subcontractorCountryCode;
    }

    /**
     * Sets the subcontractorCountryCode attribute.
     *
     * @param subcontractorCountryCode The subcontractorCountryCode to set.
     */
    public void setSubcontractorCountryCode(String subcontractorCountryCode) {
        this.subcontractorCountryCode = subcontractorCountryCode;
    }

    /**
     * Gets the active attribute.
     *
     * @return Returns the active.
     */
    @Override
    public boolean isActive() {
        return active;
    }

    /**
     * Sets the active attribute value.
     *
     * @param active The active to set.
     */
    @Override
    public void setActive(boolean active) {
        this.active = active;
    }

    /**
     * Gets the {@link Country} in which the subcontractor is located.
     *
     * @return the {@link Country} in which the subcontractor is located.
     */
    public CountryEbo getSubcontractorCountry() {
        if (StringUtils.isBlank(subcontractorCountryCode)) {
            subcontractorCountry = null;
        } else {
            if (subcontractorCountry == null || !StringUtils.equals(subcontractorCountry.getCode(), subcontractorCountryCode)) {
                ModuleService moduleService = SpringContext.getBean(KualiModuleService.class).getResponsibleModuleService(CountryEbo.class);
                if (moduleService != null) {
                    Map<String, Object> keys = new HashMap<String, Object>(1);
                    keys.put(LocationConstants.PrimaryKeyConstants.CODE, subcontractorCountryCode);
                    subcontractorCountry = moduleService.getExternalizableBusinessObject(CountryEbo.class, keys);
                } else {
                    throw new RuntimeException("CONFIGURATION ERROR: No responsible module found for EBO class.  Unable to proceed.");
                }
            }
        }
        return subcontractorCountry;
    }

    /**
     * Sets the {@link Country} in which the subcontractor is located.
     *
     * @param country the {@link Country} in which the subcontractor is located.
     */
    public void setSubcontractorCountry(CountryEbo country) {
        this.subcontractorCountry = country;
    }

    /**
     * Gets the {@link State} in which the subcontractor is located.
     *
     * @return the {@link State} in which the subcontractor is located.
     */
    public StateEbo getSubcontractorState() {
        if (StringUtils.isBlank(subcontractorCountryCode) || StringUtils.isBlank(KFSConstants.COUNTRY_CODE_UNITED_STATES)) {
            subcontractorState = null;
        } else {
            if (subcontractorState == null || !StringUtils.equals(subcontractorState.getCode(), subcontractorCountryCode) || !StringUtils.equals(subcontractorState.getCountryCode(), KFSConstants.COUNTRY_CODE_UNITED_STATES)) {
                ModuleService moduleService = SpringContext.getBean(KualiModuleService.class).getResponsibleModuleService(StateEbo.class);
                if (moduleService != null) {
                    Map<String, Object> keys = new HashMap<String, Object>(2);
                    keys.put(LocationConstants.PrimaryKeyConstants.COUNTRY_CODE, KFSConstants.COUNTRY_CODE_UNITED_STATES);/*RICE20_REFACTORME*/
                    keys.put(LocationConstants.PrimaryKeyConstants.CODE, subcontractorCountryCode);
                    subcontractorState = moduleService.getExternalizableBusinessObject(StateEbo.class, keys);
                } else {
                    throw new RuntimeException("CONFIGURATION ERROR: No responsible module found for EBO class.  Unable to proceed.");
                }
            }
        }

        return subcontractorState;
    }


    /**
     * Sets the {@link State} in which the subcontractor is located.
     *
     * @param state the {@link State} in which the subcontractor is located.
     */
    public void setSubcontractorState(StateEbo state) {
        this.subcontractorState = state;
    }

}
