/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cg.businessobject;

import org.kuali.kfs.core.api.mo.common.active.MutableInactivatable;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.integration.cg.ContractAndGrantsProposal;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;

import java.sql.Date;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public class Proposal extends PersistableBusinessObjectBase implements MutableInactivatable, ContractAndGrantsProposal {

    private String proposalNumber;
    private Date proposalBeginningDate;
    private Date proposalEndingDate;

    /*
     * This field is for write-only to the database via OJB, not the corresponding property of this BO. OJB uses
     * reflection to read it, so the compiler warns because it doesn't know.
     *
     * @see #getProposalTotalAmount
     * @see #setProposalTotalAmount
     */
    private KualiDecimal proposalTotalAmount;

    private KualiDecimal proposalDirectCostAmount;
    private KualiDecimal proposalIndirectCostAmount;
    private Date proposalRejectedDate;
    private Timestamp proposalLastUpdateDate;
    private Date proposalDueDate;
    private KualiDecimal proposalTotalProjectAmount;
    private Date proposalSubmissionDate;
    private boolean proposalFederalPassThroughIndicator;
    private String oldProposalNumber;
    private String grantNumber;
    private Date proposalClosingDate;
    private String proposalAwardTypeCode;
    private String agencyNumber;
    private String proposalStatusCode;
    private String federalPassThroughAgencyNumber;
    private String cfdaNumber;
    private String proposalFellowName;
    private String proposalPurposeCode;
    private String proposalProjectTitle;
    private boolean active;
    private List<ProposalSubcontractor> proposalSubcontractors;
    private List<ProposalOrganization> proposalOrganizations;
    private List<ProposalProjectDirector> proposalProjectDirectors;
    private List<ProposalResearchRisk> proposalResearchRisks;

    private ProposalAwardType proposalAwardType;
    private Agency agency;
    private ProposalStatus proposalStatus;
    private Agency federalPassThroughAgency;
    private ProposalPurpose proposalPurpose;
    private CFDA cfda;
    private ProposalOrganization primaryProposalOrganization;
    private String routingOrg;
    private String routingChart;
    private Award award;

    /**
     * Dummy value used to facilitate lookups
     */
    private transient String lookupProjectDirectorUniversalIdentifier;
    private transient Person lookupProjectDirector;

    public Proposal() {
        // Must use ArrayList because its get() method automatically grows the array for Struts.
        proposalSubcontractors = new ArrayList<>();
        proposalOrganizations = new ArrayList<>();
        proposalProjectDirectors = new ArrayList<>();
        proposalResearchRisks = new ArrayList<>();
    }

    @Override
    public Award getAward() {
        return award;
    }

    public void setAward(Award award) {
        this.award = award;
    }

    @Override
    public List<Collection<PersistableBusinessObject>> buildListOfDeletionAwareLists() {
        List<Collection<PersistableBusinessObject>> managedLists = super.buildListOfDeletionAwareLists();
        managedLists.add((List) getProposalSubcontractors());
        managedLists.add((List) getProposalOrganizations());
        managedLists.add((List) getProposalProjectDirectors());
        // research risks cannot be deleted (nor added)
        return managedLists;
    }

    @Override
    public String getProposalNumber() {
        return proposalNumber;
    }

    public void setProposalNumber(String proposalNumber) {
        this.proposalNumber = proposalNumber;
    }

    @Override
    public Date getProposalBeginningDate() {
        return proposalBeginningDate;
    }

    public void setProposalBeginningDate(Date proposalBeginningDate) {
        this.proposalBeginningDate = proposalBeginningDate;
    }

    @Override
    public Date getProposalEndingDate() {
        return proposalEndingDate;
    }

    public void setProposalEndingDate(Date proposalEndingDate) {
        this.proposalEndingDate = proposalEndingDate;
    }

    @Override
    public KualiDecimal getProposalTotalAmount() {
        KualiDecimal direct = getProposalDirectCostAmount();
        KualiDecimal indirect = getProposalIndirectCostAmount();
        return ObjectUtils.isNull(direct) || ObjectUtils.isNull(indirect) ? null : direct.add(indirect);
    }

    /**
     * Does nothing. This property is determined by the direct and indirect cost amounts. This setter is here only
     * because without it, specifying search criteria in this field on the lookup (e.g. >100000) doesn't return
     * correctly filtered results.
     *
     * @param proposalTotalAmount The proposalTotalAmount to set.
     */
    public void setProposalTotalAmount(KualiDecimal proposalTotalAmount) {
        // do nothing
    }

    /**
     * OJB calls this method as the first operation before this BO is inserted into the database. The database
     * contains CGPRPSL_TOT_AMT, a denormalized column that Kuali does not use but needs to maintain with this method
     * because OJB bypasses the getter.
     */
    @Override
    protected void beforeInsert() {
        super.beforeInsert();
        proposalTotalAmount = getProposalTotalAmount();
    }

    /**
     * OJB calls this method as the first operation before this BO is updated to the database. The database contains
     * CGPRPSL_TOT_AMT, a denormalized column that Kuali does not use but needs to maintain with this method because
     * OJB bypasses the getter.
     */
    @Override
    protected void beforeUpdate() {
        super.beforeUpdate();
        proposalTotalAmount = getProposalTotalAmount();
    }

    @Override
    public KualiDecimal getProposalDirectCostAmount() {
        return proposalDirectCostAmount;
    }

    public void setProposalDirectCostAmount(KualiDecimal proposalDirectCostAmount) {
        this.proposalDirectCostAmount = proposalDirectCostAmount;
    }

    @Override
    public KualiDecimal getProposalIndirectCostAmount() {
        return proposalIndirectCostAmount;
    }

    public void setProposalIndirectCostAmount(KualiDecimal proposalIndirectCostAmount) {
        this.proposalIndirectCostAmount = proposalIndirectCostAmount;
    }

    @Override
    public Date getProposalRejectedDate() {
        return proposalRejectedDate;
    }

    public void setProposalRejectedDate(Date proposalRejectedDate) {
        this.proposalRejectedDate = proposalRejectedDate;
    }

    @Override
    public Timestamp getProposalLastUpdateDate() {
        return proposalLastUpdateDate;
    }

    public void setProposalLastUpdateDate(Timestamp proposalLastUpdateDate) {
        this.proposalLastUpdateDate = proposalLastUpdateDate;
    }

    @Override
    public Date getProposalDueDate() {
        return proposalDueDate;
    }

    public void setProposalDueDate(Date proposalDueDate) {
        this.proposalDueDate = proposalDueDate;
    }

    @Override
    public KualiDecimal getProposalTotalProjectAmount() {
        return proposalTotalProjectAmount;
    }

    public void setProposalTotalProjectAmount(KualiDecimal proposalTotalProjectAmount) {
        this.proposalTotalProjectAmount = proposalTotalProjectAmount;
    }

    @Override
    public Date getProposalSubmissionDate() {
        return proposalSubmissionDate;
    }

    public void setProposalSubmissionDate(Date proposalSubmissionDate) {
        this.proposalSubmissionDate = proposalSubmissionDate;
    }

    @Override
    public boolean getProposalFederalPassThroughIndicator() {
        return proposalFederalPassThroughIndicator;
    }

    public void setProposalFederalPassThroughIndicator(boolean proposalFederalPassThroughIndicator) {
        this.proposalFederalPassThroughIndicator = proposalFederalPassThroughIndicator;
    }

    @Override
    public String getOldProposalNumber() {
        return oldProposalNumber;
    }

    public void setOldProposalNumber(String oldProposalNumber) {
        this.oldProposalNumber = oldProposalNumber;
    }

    @Override
    public String getGrantNumber() {
        return grantNumber;
    }

    public void setGrantNumber(String grantNumber) {
        this.grantNumber = grantNumber;
    }

    @Override
    public Date getProposalClosingDate() {
        return proposalClosingDate;
    }

    public void setProposalClosingDate(Date proposalClosingDate) {
        this.proposalClosingDate = proposalClosingDate;
    }

    @Override
    public String getProposalAwardTypeCode() {
        return proposalAwardTypeCode;
    }

    public void setProposalAwardTypeCode(String proposalAwardTypeCode) {
        this.proposalAwardTypeCode = proposalAwardTypeCode;
    }

    @Override
    public String getAgencyNumber() {
        return agencyNumber;
    }

    public void setAgencyNumber(String agencyNumber) {
        this.agencyNumber = agencyNumber;
    }

    @Override
    public String getProposalStatusCode() {
        return proposalStatusCode;
    }

    public void setProposalStatusCode(String proposalStatusCode) {
        this.proposalStatusCode = proposalStatusCode;
    }

    @Override
    public String getFederalPassThroughAgencyNumber() {
        return federalPassThroughAgencyNumber;
    }

    public void setFederalPassThroughAgencyNumber(String federalPassThroughAgencyNumber) {
        this.federalPassThroughAgencyNumber = federalPassThroughAgencyNumber;
    }

    @Override
    public String getCfdaNumber() {
        return cfdaNumber;
    }

    public void setCfdaNumber(String cfdaNumber) {
        this.cfdaNumber = cfdaNumber;
    }

    @Override
    public String getProposalFellowName() {
        return proposalFellowName;
    }

    public void setProposalFellowName(String proposalFellowName) {
        this.proposalFellowName = proposalFellowName;
    }

    @Override
    public String getProposalPurposeCode() {
        return proposalPurposeCode;
    }

    public void setProposalPurposeCode(String proposalPurposeCode) {
        this.proposalPurposeCode = proposalPurposeCode;
    }

    @Override
    public String getProposalProjectTitle() {
        return proposalProjectTitle;
    }

    public void setProposalProjectTitle(String proposalProjectTitle) {
        this.proposalProjectTitle = proposalProjectTitle;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public void setActive(boolean active) {
        this.active = active;
    }

    public ProposalAwardType getProposalAwardType() {
        return proposalAwardType;
    }

    @Deprecated
    public void setProposalAwardType(ProposalAwardType proposalAwardType) {
        this.proposalAwardType = proposalAwardType;
    }

    public Agency getAgency() {
        return agency;
    }

    @Deprecated
    public void setAgency(Agency agency) {
        this.agency = agency;
    }

    public ProposalStatus getProposalStatus() {
        return proposalStatus;
    }

    @Deprecated
    public void setProposalStatus(ProposalStatus proposalStatus) {
        this.proposalStatus = proposalStatus;
    }

    public Agency getFederalPassThroughAgency() {
        return federalPassThroughAgency;
    }

    @Deprecated
    public void setFederalPassThroughAgency(Agency federalPassThroughAgency) {
        this.federalPassThroughAgency = federalPassThroughAgency;
    }

    public ProposalPurpose getProposalPurpose() {
        return proposalPurpose;
    }

    @Deprecated
    public void setProposalPurpose(ProposalPurpose proposalPurpose) {
        this.proposalPurpose = proposalPurpose;
    }

    public CFDA getCfda() {
        return cfda;
    }

    @Deprecated
    public void setCfda(CFDA cfda) {
        this.cfda = cfda;
    }

    public List<ProposalSubcontractor> getProposalSubcontractors() {
        return proposalSubcontractors;
    }

    public void setProposalSubcontractors(List<ProposalSubcontractor> proposalSubcontractors) {
        this.proposalSubcontractors = proposalSubcontractors;
    }

    public List<ProposalOrganization> getProposalOrganizations() {
        return proposalOrganizations;
    }

    public void setProposalOrganizations(List<ProposalOrganization> proposalOrganizations) {
        this.proposalOrganizations = proposalOrganizations;
    }

    public List<ProposalProjectDirector> getProposalProjectDirectors() {
        return proposalProjectDirectors;
    }

    public void setProposalProjectDirectors(List<ProposalProjectDirector> proposalProjectDirectors) {
        this.proposalProjectDirectors = proposalProjectDirectors;
    }

    public List<ProposalResearchRisk> getProposalResearchRisks() {
        return proposalResearchRisks;
    }

    public List<ProposalResearchRisk> getActiveProposalResearchRisks() {
        List<ProposalResearchRisk> activeRisks = new ArrayList<>();
        for (ProposalResearchRisk risk : proposalResearchRisks) {
            if (risk.isActive()) {
                activeRisks.add(risk);
            }
        }
        return activeRisks;
    }

    public void setProposalResearchRisks(List<ProposalResearchRisk> proposalResearchRisks) {
        this.proposalResearchRisks = proposalResearchRisks;
    }

    public Person getLookupProjectDirector() {
        return lookupProjectDirector;
    }

    public void setLookupProjectDirector(Person lookupProjectDirector) {
        this.lookupProjectDirector = lookupProjectDirector;
    }

    public String getLookupProjectDirectorUniversalIdentifier() {
        lookupProjectDirector = (Person) SpringContext.getBean(PersonService.class)
                .updatePersonIfNecessary(lookupProjectDirectorUniversalIdentifier, lookupProjectDirector);
        return lookupProjectDirectorUniversalIdentifier;
    }

    public void setLookupProjectDirectorUniversalIdentifier(String lookupPersonId) {
        this.lookupProjectDirectorUniversalIdentifier = lookupPersonId;
    }

    public String getRoutingChart() {
        return routingChart;
    }

    public void setRoutingChart(String routingChart) {
        this.routingChart = routingChart;
    }

    public String getRoutingOrg() {
        return routingOrg;
    }

    public void setRoutingOrg(String routingOrg) {
        this.routingOrg = routingOrg;
    }

    public ProposalOrganization getPrimaryProposalOrganization() {
        for (ProposalOrganization po : proposalOrganizations) {
            if (po != null && po.isProposalPrimaryOrganizationIndicator()) {
                setPrimaryProposalOrganization(po);
                break;
            }
        }

        return primaryProposalOrganization;
    }

    public void setPrimaryProposalOrganization(ProposalOrganization primaryProposalOrganization) {
        this.primaryProposalOrganization = primaryProposalOrganization;
        this.routingChart = primaryProposalOrganization.getChartOfAccountsCode();
        this.routingOrg = primaryProposalOrganization.getOrganizationCode();
    }

    public String getUserLookupRoleNamespaceCode() {
        return KFSConstants.CoreModuleNamespaces.KFS;
    }

    public String getUserLookupRoleName() {
        return KFSConstants.SysKimApiConstants.CONTRACTS_AND_GRANTS_PROJECT_DIRECTOR;
    }
}
