/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cg.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.core.api.mo.common.active.MutableInactivatable;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;

/**
 * Represents a relationship between a {@link Proposal} and a {@link ProposalProjectDirector}.
 */
public class ProposalProjectDirector extends PersistableBusinessObjectBase implements Primaryable, CGProjectDirector,
        MutableInactivatable {

    private String principalId;
    private String proposalNumber;
    private boolean proposalPrimaryProjectDirectorIndicator;
    private String proposalProjectDirectorProjectTitle;
    private boolean active = true;

    private Person projectDirector;

    private final String userLookupRoleNamespaceCode = KFSConstants.CoreModuleNamespaces.KFS;
    private final String userLookupRoleName = KFSConstants.SysKimApiConstants.CONTRACTS_AND_GRANTS_PROJECT_DIRECTOR;

    @Override
    public String getPrincipalId() {
        return principalId;
    }

    @Override
    public void setPrincipalId(final String principalId) {
        this.principalId = principalId;
        // without this, adding a new project director fails the first time because it says principal name is a required
        // field (it's getting it from the project director reference - which isn't populated at first)
        if (ObjectUtils.isNull(projectDirector) || !StringUtils.equals(principalId, projectDirector.getPrincipalId())) {
            refreshReferenceObject(KFSPropertyConstants.PROJECT_DIRECTOR);
        }
    }

    @Override
    public String getProposalNumber() {
        return proposalNumber;
    }

    @Override
    public void setProposalNumber(final String proposalNumber) {
        this.proposalNumber = proposalNumber;
    }

    public boolean isProposalPrimaryProjectDirectorIndicator() {
        return proposalPrimaryProjectDirectorIndicator;
    }

    @Override
    public boolean isPrimary() {
        return isProposalPrimaryProjectDirectorIndicator();
    }

    public void setProposalPrimaryProjectDirectorIndicator(final boolean proposalPrimaryProjectDirectorIndicator) {
        this.proposalPrimaryProjectDirectorIndicator = proposalPrimaryProjectDirectorIndicator;
    }

    public String getProposalProjectDirectorProjectTitle() {
        return proposalProjectDirectorProjectTitle;
    }

    public void setProposalProjectDirectorProjectTitle(final String proposalProjectDirectorProjectTitle) {
        this.proposalProjectDirectorProjectTitle = proposalProjectDirectorProjectTitle;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public void setActive(final boolean active) {
        this.active = active;
    }

    @Override
    public Person getProjectDirector() {
        if (principalId != null) {
            // It would be nice to get rid of this and use OJB mappings instead, unfortunately without this, the old
            // lookup on the Proposal doc doesn't work.
            projectDirector = SpringContext.getBean(PersonService.class)
                    .updatePersonIfNecessary(principalId, projectDirector);
        }
        return projectDirector;
    }

    @Override
    public void setProjectDirector(final Person projectDirector) {
        this.projectDirector = projectDirector;
    }

    /**
     * This can be displayed by Proposal.xml lookup results.
     */
    @Override
    public String toString() {
        // todo: get "nonexistent", "primary", and "secondary" from ApplicationResources.properties via KFSKeyConstants?
        final String name = ObjectUtils.isNull(getProjectDirector()) ? "nonexistent" : getProjectDirector().getName();
        final String title = getProposalProjectDirectorProjectTitle() == null ? "" : " " + getProposalProjectDirectorProjectTitle();
        return name + " " + (isProposalPrimaryProjectDirectorIndicator() ? "primary" : "secondary") + title;
    }

    public String getUserLookupRoleNamespaceCode() {
        return userLookupRoleNamespaceCode;
    }

    public String getUserLookupRoleName() {
        return userLookupRoleName;
    }
}
