/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.cg.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.COAConstants;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.cg.CGConstants;
import org.kuali.kfs.module.cg.businessobject.AwardAccount;
import org.kuali.kfs.module.cg.service.AgencyService;
import org.kuali.kfs.module.cg.service.CfdaService;
import org.kuali.kfs.sys.KFSPropertyConstants;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This Class provides implementation to the services required for inter module communication.
 */
public class ContractsAndGrantsModuleServiceImpl implements ContractsAndGrantsModuleService {

    protected ParameterService parameterService;
    protected AgencyService agencyService;
    protected CfdaService cfdaService;
    protected BusinessObjectService businessObjectService;

    @Override
    public Person getProjectDirectorForAccount(final String chartOfAccountsCode, final String accountNumber) {
        final Map<String, Object> awardAccountMap = new HashMap<>();
        awardAccountMap.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        awardAccountMap.put(KFSPropertyConstants.ACCOUNT_NUMBER, accountNumber);

        final Collection<AwardAccount> proposals = getBusinessObjectService().findMatchingOrderBy(AwardAccount.class,
                awardAccountMap, KFSPropertyConstants.PROPOSAL_NUMBER, false);
        if (proposals != null && !proposals.isEmpty()) {
            final AwardAccount proposalWithMaxProposalNumber = proposals.iterator().next();

            return proposalWithMaxProposalNumber.getProjectDirector();
        }

        return null;
    }

    @Override
    public Person getProjectDirectorForAccount(final Account account) {
        if (ObjectUtils.isNotNull(account)) {
            account.refreshNonUpdateableReferences();
            final String chartOfAccountsCode = account.getChartOfAccountsCode();
            final String accountNumber = account.getAccountNumber();
            return getProjectDirectorForAccount(chartOfAccountsCode, accountNumber);
        }
        return null;
    }

    @Override
    public boolean isAwardedByFederalAgency(
            final String chartOfAccountsCode, final String accountNumber,
            final Collection<String> federalAgencyTypeCodes) {
        final AwardAccount primaryAward = getPrimaryAwardAccount(chartOfAccountsCode, accountNumber);
        if (primaryAward == null) {
            return false;
        }

        final String agencyTypeCode = primaryAward.getAward().getAgency().getAgencyTypeCode();
        return federalAgencyTypeCodes.contains(agencyTypeCode)
                || primaryAward.getAward().getFederalPassThroughIndicator();

    }

    /**
     * get the primary award account for the given account
     *
     * @param chartOfAccountsCode the chart of accounts code for the given account
     * @return the primary award account for the given account
     */
    protected AwardAccount getPrimaryAwardAccount(final String chartOfAccountsCode, final String accountNumber) {
        AwardAccount primaryAwardAccount = null;
        long highestProposalNumber = 0;

        final Map<String, Object> accountKeyValues = new HashMap<>();
        accountKeyValues.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        accountKeyValues.put(KFSPropertyConstants.ACCOUNT_NUMBER, accountNumber);

        for (final Object awardAccountAsObject : getBusinessObjectService().findMatching(AwardAccount.class,
                accountKeyValues)) {
            final AwardAccount awardAccount = (AwardAccount) awardAccountAsObject;
            final Long proposalNumber = Long.parseLong(awardAccount.getProposalNumber());

            if (proposalNumber >= highestProposalNumber) {
                highestProposalNumber = proposalNumber;
                primaryAwardAccount = awardAccount;
            }
        }

        return primaryAwardAccount;
    }

    @Override
    public List<Integer> getAllAccountReponsiblityIds() {
        final int maxResponsibilityId = getMaxiumAccountResponsibilityId();

        final List<Integer> contractsAndGrantsResponsiblityIds = new ArrayList<>();
        for (int id = 1; id <= maxResponsibilityId; id++) {
            contractsAndGrantsResponsiblityIds.add(id);
        }

        return contractsAndGrantsResponsiblityIds;
    }

    @Override
    public boolean hasValidAccountReponsiblityIdIfNotNull(final Account account) {
        final Integer accountResponsibilityId = account.getContractsAndGrantsAccountResponsibilityId();

        if (accountResponsibilityId == null) {
            return true;
        }

        return accountResponsibilityId >= 1 && accountResponsibilityId <= getMaxiumAccountResponsibilityId();
    }

    protected int getMaxiumAccountResponsibilityId() {
        final String maxResponsibilityId = getParameterService().getParameterValueAsString(
                COAConstants.COA_NAMESPACE_CODE,
                CGConstants.Components.ACCOUNT_CMPNT,
                CGConstants.CG_RESPONSIBILITY_ID);
        return Integer.parseInt(maxResponsibilityId);
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public AgencyService getAgencyService() {
        return agencyService;
    }

    public CfdaService getCfdaService() {
        return cfdaService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    @Override
    public String getProposalNumberForAccountAndProjectDirector(
            final String chartOfAccountsCode, final String accountNumber,
            final String projectDirectorId) {
        String proposalNumber = null;

        final Map<String, Object> awardAccountMap = new HashMap<>();
        awardAccountMap.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        awardAccountMap.put(KFSPropertyConstants.ACCOUNT_NUMBER, accountNumber);

        final Collection<AwardAccount> proposals = getBusinessObjectService().findMatchingOrderBy(AwardAccount.class,
                awardAccountMap, KFSPropertyConstants.PROPOSAL_NUMBER, false);
        if (proposals != null && !proposals.isEmpty()) {
            final AwardAccount proposalWithMaxProposalNumber = proposals.iterator().next();

            if (StringUtils.equalsIgnoreCase(proposalWithMaxProposalNumber.getProjectDirector().getPrincipalId(),
                    projectDirectorId)) {
                proposalNumber = proposalWithMaxProposalNumber.getProposalNumber();
            }
        }

        return proposalNumber;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setAgencyService(final AgencyService agencyService) {
        this.agencyService = agencyService;
    }

    public void setCfdaService(final CfdaService cfdaService) {
        this.cfdaService = cfdaService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }
}
