/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.document.validation.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.maintenance.rules.MaintenanceDocumentRuleBase;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ec.EffortKeyConstants;
import org.kuali.kfs.module.ec.EffortPropertyConstants;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationReportDefinition;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationReportPosition;

import java.util.Collection;

/**
 * Contains Business Rules for the Effort Certification Report Maintenance Document.
 */
public class EffortCertificationReportDefinitionRule extends MaintenanceDocumentRuleBase {

    private static final Logger LOG = LogManager.getLogger(EffortCertificationReportDefinitionRule.class);

    @Override
    protected boolean processCustomRouteDocumentBusinessRules(MaintenanceDocument document) {
        LOG.debug("processCustomRouteDocumentBusinessRules() start");

        if (GlobalVariables.getMessageMap().getErrorCount() > 0) {
            return false;
        }

        boolean isValid = true;
        EffortCertificationReportDefinition reportDefinition = (EffortCertificationReportDefinition) document
                .getNewMaintainableObject().getBusinessObject();

        if (reportDefinition.getEffortCertificationReportBeginFiscalYear() >
                reportDefinition.getEffortCertificationReportEndFiscalYear()
                || (reportDefinition.getEffortCertificationReportBeginFiscalYear().equals(
                        reportDefinition.getEffortCertificationReportEndFiscalYear())
                && Integer.parseInt(reportDefinition.getEffortCertificationReportBeginPeriodCode()) >=
                        Integer.parseInt(reportDefinition.getEffortCertificationReportEndPeriodCode()))) {
            putFieldError(EffortPropertyConstants.EFFORT_CERTIFICATION_REPORT_END_FISCAL_YEAR,
                    EffortKeyConstants.ERROR_END_FISCAL_YEAR);
            isValid = false;
        }

        // at least one position object group code defined for report
        Collection<EffortCertificationReportPosition> reportPositions =
                reportDefinition.getEffortCertificationReportPositions();
        if (reportPositions == null || reportPositions.isEmpty()) {
            putFieldError(EffortPropertyConstants.EFFORT_CERTIFICATION_REPORT_POSITIONS,
                    EffortKeyConstants.ERROR_NOT_HAVE_POSITION_GROUP);
            isValid = false;
        }

        // add custom validation for periods but dont check active flag
        if (ObjectUtils.isNull(reportDefinition.getReportBeginPeriod())) {
            putFieldError(EffortPropertyConstants.EFFORT_CERTIFICATION_REPORT_BEGIN_FISCAL_YEAR,
                    EffortKeyConstants.INVALID_REPORT_BEGIN_PERIOD);
            isValid = false;
        }
        if (ObjectUtils.isNull(reportDefinition.getReportEndPeriod())) {
            putFieldError(EffortPropertyConstants.EFFORT_CERTIFICATION_REPORT_END_FISCAL_YEAR,
                    EffortKeyConstants.INVALID_REPORT_END_PERIOD);
            isValid = false;
        }
        if (ObjectUtils.isNull(reportDefinition.getExpenseTransferFiscalPeriod())) {
            putFieldError(EffortPropertyConstants.EXPENSE_TRANSFER_FISCAL_YEAR,
                    EffortKeyConstants.INVALID_EXPENSE_TRANSFER_PERIOD);
            isValid = false;
        }

        return isValid;
    }

}
