/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.businessobject.SubAccount;
import org.kuali.kfs.coa.service.AccountService;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleService;
import org.kuali.kfs.integration.ld.LaborModuleService;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ec.document.EffortCertificationDocument;
import org.kuali.kfs.module.ec.util.EffortCertificationParameterFinder;
import org.kuali.kfs.module.ec.util.PayrollAmountHolder;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLineOverride;
import org.kuali.kfs.sys.businessobject.SystemOptions;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.util.Collection;
import java.util.List;

/**
 * Business Object for the Effort Certification Detail Table.
 */
public class EffortCertificationDetail extends PersistableBusinessObjectBase {

    private String documentNumber;
    private String chartOfAccountsCode;
    private String accountNumber;
    private String subAccountNumber;
    private String positionNumber;
    private String financialObjectCode;
    private String sourceChartOfAccountsCode;
    private String sourceAccountNumber;


    private KualiDecimal effortCertificationPayrollAmount;
    private KualiDecimal effortCertificationOriginalPayrollAmount;
    private Integer effortCertificationCalculatedOverallPercent;
    private Integer effortCertificationUpdatedOverallPercent;
    private String costShareSourceSubAccountNumber;

    private Integer universityFiscalYear;

    private KualiDecimal originalFringeBenefitAmount;

    private boolean accountExpiredOverride;
    private boolean accountExpiredOverrideNeeded;
    private String overrideCode = AccountingLineOverride.CODE.NONE;
    // to indicate if this detail line has been persisted or not
    private boolean newLineIndicator;

    // holds last saved updated payroll amount so business rule can check if it has been updated at the route level
    private KualiDecimal persistedPayrollAmount;
    private Integer persistedEffortPercent;
    private String groupId;

    private EffortCertificationDocument effortCertificationDocument;
    private ObjectCode financialObject;
    private Chart chartOfAccounts;
    private Account account;
    private Chart sourceChartOfAccounts;
    private Account sourceAccount;
    private SubAccount subAccount;
    private SystemOptions options;
    private Boolean federalOrFederalPassThroughIndicator = null;

    protected String effectiveDate;

    public EffortCertificationDetail() {
        super();

        if (SpringContext.isInitialized()) {
            subAccountNumber = KFSConstants.getDashSubAccountNumber();
        }

        effortCertificationPayrollAmount = KualiDecimal.ZERO;
        effortCertificationOriginalPayrollAmount = KualiDecimal.ZERO;
        effortCertificationCalculatedOverallPercent = 0;
        effortCertificationUpdatedOverallPercent = 0;
        originalFringeBenefitAmount = KualiDecimal.ZERO;
        effectiveDate = KFSConstants.EMPTY_STRING;
    }

    public EffortCertificationDetail(EffortCertificationDetail effortCertificationDetail) {
        super();
        if (effortCertificationDetail != null) {
            this.chartOfAccountsCode = effortCertificationDetail.getChartOfAccountsCode();
            this.accountNumber = effortCertificationDetail.getAccountNumber();
            this.subAccountNumber = effortCertificationDetail.getSubAccountNumber();
            this.positionNumber = effortCertificationDetail.getPositionNumber();
            this.financialObjectCode = effortCertificationDetail.getFinancialObjectCode();
            this.sourceChartOfAccountsCode = effortCertificationDetail.getSourceChartOfAccountsCode();
            this.sourceAccountNumber = effortCertificationDetail.getSourceAccountNumber();
            this.effortCertificationPayrollAmount = effortCertificationDetail.getEffortCertificationPayrollAmount();
            this.effortCertificationCalculatedOverallPercent =
                    effortCertificationDetail.getEffortCertificationCalculatedOverallPercent();
            this.effortCertificationUpdatedOverallPercent =
                    effortCertificationDetail.getEffortCertificationUpdatedOverallPercent();
            this.universityFiscalYear = effortCertificationDetail.getUniversityFiscalYear();
            this.costShareSourceSubAccountNumber = effortCertificationDetail.getCostShareSourceSubAccountNumber();
            this.effortCertificationOriginalPayrollAmount =
                    effortCertificationDetail.getEffortCertificationOriginalPayrollAmount();
            this.originalFringeBenefitAmount = effortCertificationDetail.getOriginalFringeBenefitAmount();
            this.effectiveDate = effortCertificationDetail.getEffectiveDate();
        }
    }

    public String getDocumentNumber() {
        return documentNumber;
    }

    public void setDocumentNumber(String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public String getChartOfAccountsCode() {
        return chartOfAccountsCode;
    }

    public void setChartOfAccountsCode(String chartOfAccountsCode) {
        this.chartOfAccountsCode = chartOfAccountsCode;
    }

    public String getAccountNumber() {
        return accountNumber;
    }

    public void setAccountNumber(String accountNumber) {
        this.accountNumber = accountNumber;
        // if accounts can't cross charts, set chart code whenever account number is set
        AccountService accountService = SpringContext.getBean(AccountService.class);
        if (!accountService.accountsCanCrossCharts()) {
            Account account = accountService.getUniqueAccountForAccountNumber(accountNumber);
            if (ObjectUtils.isNotNull(account)) {
                setChartOfAccountsCode(account.getChartOfAccountsCode());
                setChartOfAccounts(account.getChartOfAccounts());

            }
        }
    }

    public String getSubAccountNumber() {
        return subAccountNumber;
    }

    public void setSubAccountNumber(String subAccountNumber) {
        this.subAccountNumber = subAccountNumber;
    }

    public String getPositionNumber() {
        return positionNumber;
    }

    public void setPositionNumber(String positionNumber) {
        this.positionNumber = positionNumber;
    }

    public String getEffectiveDate() {
        return effectiveDate;
    }

    public void setEffectiveDate(String effectiveDate) {
        this.effectiveDate = effectiveDate;
    }

    public String getFinancialObjectCode() {
        return financialObjectCode;
    }

    public void setFinancialObjectCode(String financialObjectCode) {
        this.financialObjectCode = financialObjectCode;
    }

    public String getSourceChartOfAccountsCode() {
        return sourceChartOfAccountsCode;
    }

    public void setSourceChartOfAccountsCode(String sourceChartOfAccountsCode) {
        this.sourceChartOfAccountsCode = sourceChartOfAccountsCode;
    }

    public String getSourceAccountNumber() {
        return sourceAccountNumber;
    }

    public void setSourceAccountNumber(String sourceAccountNumber) {
        this.sourceAccountNumber = sourceAccountNumber;
    }

    public KualiDecimal getEffortCertificationPayrollAmount() {
        return effortCertificationPayrollAmount;
    }

    public void setEffortCertificationPayrollAmount(KualiDecimal effortCertificationPayrollAmount) {
        this.effortCertificationPayrollAmount = effortCertificationPayrollAmount;
    }

    public Integer getEffortCertificationCalculatedOverallPercent() {
        return effortCertificationCalculatedOverallPercent;
    }

    public void setEffortCertificationCalculatedOverallPercent(Integer effortCertificationCalculatedOverallPercent) {
        this.effortCertificationCalculatedOverallPercent = effortCertificationCalculatedOverallPercent;
    }

    public Integer getEffortCertificationUpdatedOverallPercent() {
        return effortCertificationUpdatedOverallPercent;
    }

    public void setEffortCertificationUpdatedOverallPercent(Integer effortCertificationUpdatedOverallPercent) {
        this.effortCertificationUpdatedOverallPercent = effortCertificationUpdatedOverallPercent;
    }

    public String getCostShareSourceSubAccountNumber() {
        return costShareSourceSubAccountNumber;
    }

    public void setCostShareSourceSubAccountNumber(String costShareSourceSubAccountNumber) {
        this.costShareSourceSubAccountNumber = costShareSourceSubAccountNumber;
    }

    public KualiDecimal getEffortCertificationOriginalPayrollAmount() {
        return effortCertificationOriginalPayrollAmount;
    }

    public void setEffortCertificationOriginalPayrollAmount(KualiDecimal effortCertificationOriginalPayrollAmount) {
        this.effortCertificationOriginalPayrollAmount = effortCertificationOriginalPayrollAmount;
    }

    public ObjectCode getFinancialObject() {
        return financialObject;
    }

    @Deprecated
    public void setFinancialObject(ObjectCode financialObject) {
        this.financialObject = financialObject;
    }

    public Chart getChartOfAccounts() {
        return chartOfAccounts;
    }

    @Deprecated
    public void setChartOfAccounts(Chart chartOfAccounts) {
        this.chartOfAccounts = chartOfAccounts;
    }

    public EffortCertificationDocument getEffortCertificationDocument() {
        return effortCertificationDocument;
    }

    @Deprecated
    public void setEffortCertificationDocument(EffortCertificationDocument effortCertificationDocument) {
        this.effortCertificationDocument = effortCertificationDocument;
    }

    public Account getAccount() {
        if (account == null && StringUtils.isNotBlank(this.getChartOfAccountsCode())
                && StringUtils.isNotBlank(this.getAccountNumber())) {
            this.refreshReferenceObject(KFSPropertyConstants.ACCOUNT);
        }

        return account;
    }

    @Deprecated
    public void setAccount(Account account) {
        this.account = account;
    }

    public Chart getSourceChartOfAccounts() {
        return sourceChartOfAccounts;
    }

    @Deprecated
    public void setSourceChartOfAccounts(Chart sourceChartOfAccounts) {
        this.sourceChartOfAccounts = sourceChartOfAccounts;
    }

    public Account getSourceAccount() {
        return sourceAccount;
    }

    @Deprecated
    public void setSourceAccount(Account sourceAccount) {
        this.sourceAccount = sourceAccount;
    }

    public SubAccount getSubAccount() {
        return subAccount;
    }

    @Deprecated
    public void setSubAccount(SubAccount subAccount) {
        this.subAccount = subAccount;
    }

    public SystemOptions getOptions() {
        return options;
    }

    @Deprecated
    public void setOptions(SystemOptions options) {
        this.options = options;
    }

    public boolean isNewLineIndicator() {
        return newLineIndicator;
    }

    public void setNewLineIndicator(boolean newLineIndicator) {
        this.newLineIndicator = newLineIndicator;
    }

    /**
     * If the account of this detail line is closed, the line cannot be edited.
     *
     * @return Returns true if line can be edited, false otherwise
     */
    public boolean isEditable() {
        return this.getAccount() == null || this.getAccount().isActive();
    }

    /**
     * Gets the federalOrFederalPassThroughIndicator attribute. If this line is associated with a valid account, the
     * indicator will be retrieved and updated.
     *
     * @return Returns the federalOrFederalPassThroughIndicator.
     */
    public boolean isFederalOrFederalPassThroughIndicator() {
        if (this.federalOrFederalPassThroughIndicator != null) {
            return this.federalOrFederalPassThroughIndicator;
        }
        if (this.getAccount() != null) {
            Collection<String> federalAgencyTypeCodes = EffortCertificationParameterFinder.getFederalAgencyTypeCodes();
            this.federalOrFederalPassThroughIndicator = SpringContext.getBean(ContractsAndGrantsModuleService.class)
                    .isAwardedByFederalAgency(
                getAccount().getChartOfAccountsCode(), getAccount().getAccountNumber(), federalAgencyTypeCodes);
            return this.federalOrFederalPassThroughIndicator;
        }

        return false;
    }

    /**
     * This is a marker method, which does nothing.
     */
    public void setFederalOrFederalPassThroughIndicator(boolean federalOrFederalPassThroughIndicator) {
    }

    public String getOverrideCode() {
        return overrideCode;
    }

    public void setOverrideCode(String overrideCode) {
        this.overrideCode = overrideCode;
    }

    public KualiDecimal getFringeBenefitAmount() {
        KualiDecimal payrollAmount = this.getEffortCertificationPayrollAmount();

        return EffortCertificationDetail.calculateFringeBenefit(this, payrollAmount);
    }

    /**
     * This is a marker method, which does nothing.
     */
    public void setFringeBenefitAmount(KualiDecimal fringeBenefitAmount) {
    }

    public KualiDecimal getOriginalFringeBenefitAmount() {
        if (this.originalFringeBenefitAmount == null || originalFringeBenefitAmount.isZero()) {
            this.recalculateOriginalFringeBenefit();
        }
        return originalFringeBenefitAmount;
    }

    public void setOriginalFringeBenefitAmount(KualiDecimal originalFringeBenefitAmount) {
        this.originalFringeBenefitAmount = originalFringeBenefitAmount;
    }

    public Integer getUniversityFiscalYear() {
        return this.universityFiscalYear;
    }

    public void setUniversityFiscalYear(Integer universityFiscalYear) {
        this.universityFiscalYear = universityFiscalYear;
    }

    public KualiDecimal getPersistedPayrollAmount() {
        return persistedPayrollAmount;
    }

    public void setPersistedPayrollAmount(KualiDecimal persistedPayrollAmount) {
        this.persistedPayrollAmount = persistedPayrollAmount;
    }

    public boolean isAccountExpiredOverride() {
        return accountExpiredOverride;
    }

    public void setAccountExpiredOverride(boolean accountExpiredOverride) {
        this.accountExpiredOverride = accountExpiredOverride;
    }

    public boolean isAccountExpiredOverrideNeeded() {
        return accountExpiredOverrideNeeded;
    }

    public void setAccountExpiredOverrideNeeded(boolean accountExpiredOverrideNeeded) {
        this.accountExpiredOverrideNeeded = accountExpiredOverrideNeeded;
    }

    /**
     * calculate the total effort percent of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total effort percent
     */
    public static Integer getTotalEffortPercent(List<EffortCertificationDetail> effortCertificationDetailLines) {
        Integer totalEffortPercent = 0;

        for (EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalEffortPercent += detailLine.getEffortCertificationUpdatedOverallPercent();
        }

        return totalEffortPercent;
    }

    /**
     * calculate the total persisted effort percent of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total persisted effort percent
     */
    public static Integer getTotalPersistedEffortPercent(List<EffortCertificationDetail> effortCertificationDetailLines) {
        Integer totalEffortPercent = 0;

        for (EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalEffortPercent += detailLine.getPersistedEffortPercent();
        }

        return totalEffortPercent;
    }

    /**
     * calculate the total original effort percent of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total original effort percent
     */
    public static Integer getTotalOriginalEffortPercent(
            List<EffortCertificationDetail> effortCertificationDetailLines) {
        Integer totalOriginalEffortPercent = 0;

        for (EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalOriginalEffortPercent += detailLine.getEffortCertificationCalculatedOverallPercent();
        }

        return totalOriginalEffortPercent;
    }

    /**
     * calculate the total payroll amount of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total original payroll amount
     */
    public static KualiDecimal getTotalPayrollAmount(List<EffortCertificationDetail> effortCertificationDetailLines) {
        KualiDecimal totalPayrollAmount = KualiDecimal.ZERO;

        for (EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalPayrollAmount = totalPayrollAmount.add(detailLine.getEffortCertificationPayrollAmount());
        }

        return totalPayrollAmount;
    }

    /**
     * calculate the total payroll amount of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total original payroll amount
     */
    public static KualiDecimal getTotalPersistedPayrollAmount(
            List<EffortCertificationDetail> effortCertificationDetailLines) {
        KualiDecimal totalPayrollAmount = KualiDecimal.ZERO;

        for (EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalPayrollAmount = totalPayrollAmount.add(detailLine.getPersistedPayrollAmount());
        }

        return totalPayrollAmount;
    }

    /**
     * calculate the total original payroll amount of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total original payroll amount
     */
    public static KualiDecimal getTotalOriginalPayrollAmount(
            List<EffortCertificationDetail> effortCertificationDetailLines) {
        KualiDecimal totalOriginalPayrollAmount = KualiDecimal.ZERO;

        for (EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalOriginalPayrollAmount = totalOriginalPayrollAmount.add(detailLine
                    .getEffortCertificationOriginalPayrollAmount());
        }

        return totalOriginalPayrollAmount;
    }

    public static KualiDecimal getTotalFringeBenefit(List<EffortCertificationDetail> effortCertificationDetailLines) {
        KualiDecimal totalFringeBenefit = KualiDecimal.ZERO;

        for (EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalFringeBenefit = totalFringeBenefit.add(detailLine.getFringeBenefitAmount());
        }

        return totalFringeBenefit;
    }

    public static KualiDecimal getTotalOriginalFringeBenefit(
            List<EffortCertificationDetail> effortCertificationDetailLines) {
        KualiDecimal totalOriginalFringeBenefit = KualiDecimal.ZERO;

        for (EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalOriginalFringeBenefit = totalOriginalFringeBenefit.add(detailLine.getOriginalFringeBenefitAmount());
        }

        return totalOriginalFringeBenefit;
    }

    /**
     * recalculate the payroll amount of the current detail line
     *
     * @param totalPayrollAmount the total payroll amount of the hosting document
     */
    public void recalculatePayrollAmount(KualiDecimal totalPayrollAmount) {
        Integer effortPercent = this.getEffortCertificationUpdatedOverallPercent();
        KualiDecimal payrollAmount = PayrollAmountHolder.recalculatePayrollAmount(totalPayrollAmount, effortPercent);
        this.setEffortCertificationPayrollAmount(payrollAmount);
    }

    /**
     * recalculate the original fringe benefit of the current detail line
     */
    public void recalculateOriginalFringeBenefit() {
        KualiDecimal originalPayrollAmount = this.getEffortCertificationOriginalPayrollAmount();
        KualiDecimal fringeBenefit = EffortCertificationDetail.calculateFringeBenefit(this, originalPayrollAmount);
        this.setOriginalFringeBenefitAmount(fringeBenefit);
    }

    /**
     * recalculate the original fringe benefit of the current detail line
     */
    public static KualiDecimal calculateFringeBenefit(EffortCertificationDetail detailLine,
            KualiDecimal payrollAmount) {
        LaborModuleService laborModuleService = SpringContext.getBean(LaborModuleService.class);
        Integer fiscalYear = EffortCertificationParameterFinder.getCreateReportFiscalYear();
        String chartOfAccountsCode = detailLine.getChartOfAccountsCode();
        String objectCode = detailLine.getFinancialObjectCode();
        String accountNumber = detailLine.getAccountNumber();
        String subAccountNumber = detailLine.getSubAccountNumber();

        return laborModuleService.calculateFringeBenefit(fiscalYear, chartOfAccountsCode, objectCode, payrollAmount,
                accountNumber, subAccountNumber);
    }

    public Integer getPersistedEffortPercent() {
        return persistedEffortPercent;
    }

    public void setPersistedEffortPercent(Integer persistedEffortPercent) {
        this.persistedEffortPercent = persistedEffortPercent;
    }

    public String getGroupId() {
        return groupId;
    }

    public void setGroupId(String groupId) {
        this.groupId = groupId;
    }

}
