/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants.COMPONENT;
import org.kuali.kfs.kew.actionitem.ActionItemActionListExtension;
import org.kuali.kfs.kew.api.KewApiConstants;
import org.kuali.kfs.kew.service.KEWServiceLocator;
import org.kuali.kfs.kim.api.identity.Person;
import org.kuali.kfs.kim.api.services.KimApiServiceLocator;
import org.kuali.kfs.module.ec.document.EffortCertificationDocument;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.FinancialSystemUserService;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

/**
 * Business object for the outstanding documents by organization report
 */
@COMPONENT(component = "OutstandingCertificationsByOrganization")
public class OutstandingCertificationsByOrganization extends EffortCertificationDocument {

    /**
     * gets a string representation of the document's chart and organization.
     *
     * @return comma separated String
     */
    public String getCertificationOrganizations() {
        String certificationOrganizations = "";
        List<String> certificationOrganizationList = new ArrayList<>();
        List<EffortCertificationDetail> details = getEffortCertificationDetailLines();

        for (EffortCertificationDetail detailLine : details) {
            String accountOrg = detailLine.getAccount().getChartOfAccountsCode() + "-" +
                    detailLine.getAccount().getOrganizationCode();
            if (!certificationOrganizationList.contains(accountOrg)) {
                certificationOrganizationList.add(accountOrg);
                if ("".equals(certificationOrganizations)) {
                    certificationOrganizations = accountOrg;
                } else {
                    certificationOrganizations = certificationOrganizations + ", " + accountOrg;
                }
            }
        }

        return certificationOrganizations;
    }

    /**
     * Queries workflow to get users who have an approval request for this effort certification.
     *
     * @return String - names of users (separated by comma) who have an approval request
     */
    public String getNextApprovers() {
        String nextApprovers = "";

        Collection<ActionItemActionListExtension> actionList = KEWServiceLocator.getActionListService()
                .getActionListForSingleDocument(getDocumentHeader().getDocumentNumber());
        for (ActionItemActionListExtension actionItem : actionList) {
            if (actionItem.getActionRequestCd().equals(KewApiConstants.ACTION_REQUEST_APPROVE_REQ)) {
                String principalId = actionItem.getPrincipalId();
                if (principalId != null) {
                    Person person = KimApiServiceLocator.getPersonService().getPerson(actionItem.getPrincipalId());
                    if (StringUtils.isBlank(nextApprovers)) {
                        nextApprovers = person.getName();
                    } else {
                        nextApprovers += "; " + person.getName();
                    }
                }
            }
        }

        return nextApprovers;
    }

    public String getEmployeeName() {
        return SpringContext.getBean(FinancialSystemUserService.class).getPersonNameByEmployeeId(emplid);
    }

    @Override
    public boolean equals(Object arg0) {
        if (arg0 instanceof OutstandingCertificationsByOrganization) {
            List<String> keyFields = new ArrayList<>();
            keyFields.add(KFSPropertyConstants.DOCUMENT_NUMBER);
            ObjectUtil.equals(this, arg0, keyFields);
        }

        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(documentNumber);
    }
}
