/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.batch.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.integration.ld.LaborLedgerBalance;
import org.kuali.kfs.integration.ld.LaborLedgerBalanceForEffortCertification;
import org.kuali.kfs.integration.ld.LaborModuleService;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.module.ec.EffortConstants;
import org.kuali.kfs.module.ec.EffortConstants.ExtractProcess;
import org.kuali.kfs.module.ec.EffortKeyConstants;
import org.kuali.kfs.module.ec.EffortParameterConstants;
import org.kuali.kfs.module.ec.EffortPropertyConstants;
import org.kuali.kfs.module.ec.batch.service.EffortCertificationExtractService;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationDocumentBuild;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationReportDefinition;
import org.kuali.kfs.module.ec.document.EffortCertificationDocument;
import org.kuali.kfs.module.ec.document.validation.impl.LedgerBalanceFieldValidator;
import org.kuali.kfs.module.ec.service.EffortCertificationDocumentBuildService;
import org.kuali.kfs.module.ec.service.EffortCertificationReportDefinitionService;
import org.kuali.kfs.module.ec.service.EffortCertificationReportService;
import org.kuali.kfs.module.ec.util.EffortCertificationParameterFinder;
import org.kuali.kfs.module.ec.util.ExtractProcessReportDataHolder;
import org.kuali.kfs.module.ec.util.LedgerBalanceConsolidationHelper;
import org.kuali.kfs.module.ec.util.LedgerBalanceWithMessage;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.Message;
import org.kuali.kfs.sys.MessageBuilder;
import org.kuali.kfs.sys.service.OptionsService;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This is an implementation of Effort Certification Extract process, which extracts Labor Ledger records of the
 * employees who were paid on a grant or cost shared during the selected reporting period, and generates effort
 * certification build/temporary document. Its major tasks include:
 * <p>
 * - Identify employees who were paid on a grant or cost shared;
 * - Select qualified Labor Ledger records for each identified employee;
 * - Generate effort certification build document from the selected Labor Ledger records for each employee.
 */
@Transactional
public class EffortCertificationExtractServiceImpl implements EffortCertificationExtractService {

    private static final Logger LOG = LogManager.getLogger();

    protected BusinessObjectService businessObjectService;
    private OptionsService optionsService;
    private DateTimeService dateTimeService;
    private UniversityDateService universityDateService;

    private LaborModuleService laborModuleService;
    private KualiModuleService kualiModuleService;

    private EffortCertificationDocumentBuildService effortCertificationDocumentBuildService;
    private EffortCertificationReportService effortCertificationReportService;
    private EffortCertificationReportDefinitionService effortCertificationReportDefinitionService;

    @Override
    public void extract() {
        final Integer fiscalYear = EffortCertificationParameterFinder.getExtractReportFiscalYear();
        final String reportNumber = EffortCertificationParameterFinder.getExtractReportNumber();

        extract(fiscalYear, reportNumber);
    }

    @Override
    public void extract(final Integer fiscalYear, final String reportNumber) {
        final Map<String, String> fieldValues =
                EffortCertificationReportDefinition.buildKeyMap(fiscalYear, reportNumber);

        // check if a report has been defined and its documents have not been generated.
        LOG.info("Validating and retrieving report definition ...");
        String errorMessage = validateReportDefinition(fiscalYear, reportNumber);
        errorMessage = StringUtils.isNotEmpty(errorMessage) ? errorMessage :
                existEffortCertificationDocument(fieldValues);
        if (StringUtils.isNotEmpty(errorMessage)) {
            LOG.fatal(errorMessage);
            throw new IllegalArgumentException(errorMessage);
        }

        final Map<String, Collection<String>> parameters = getSystemParameters();
        parameters.put(ExtractProcess.EXPENSE_OBJECT_TYPE, getExpenseObjectTypeCodes(fiscalYear));

        final EffortCertificationReportDefinition reportDefinition = effortCertificationReportDefinitionService
                .findReportDefinitionByPrimaryKey(fieldValues);
        final ExtractProcessReportDataHolder reportDataHolder = initializeReportData(reportDefinition);

        LOG.info("Finding employees eligible for effort certification ...");
        final List<String> employees = findEmployeesEligibleForEffortCertification(reportDefinition);

        LOG.info("Generating document build for the eligible employees ...");
        effortCertificationDocumentBuildService.removeExistingDocumentBuild(fieldValues);
        generateDocumentBuild(reportDefinition, employees, reportDataHolder, parameters);

        LOG.info("Generating report for extract process ...");
        final Date runDate = dateTimeService.getCurrentSqlDate();
        effortCertificationReportService.generateReportForExtractProcess(reportDataHolder, runDate);
    }

    @Override
    public EffortCertificationDocumentBuild extract(
            final String emplid,
            final EffortCertificationReportDefinition effortCertificationReportDefinition
    ) {
        final Map<String, Collection<String>> parameters = getSystemParameters();
        parameters.put(ExtractProcess.EXPENSE_OBJECT_TYPE,
                getExpenseObjectTypeCodes(effortCertificationReportDefinition.getUniversityFiscalYear()));

        final List<String> positionGroupCodes = effortCertificationReportDefinitionService
                .findPositionObjectGroupCodes(effortCertificationReportDefinition);
        final Integer postingYear = universityDateService.getCurrentFiscalYear();
        final ExtractProcessReportDataHolder reportDataHolder = initializeReportData(
                effortCertificationReportDefinition);

        return generateDocumentBuildByEmployee(postingYear, emplid, positionGroupCodes,
                effortCertificationReportDefinition,
                reportDataHolder, parameters);
    }

    @Override
    public boolean isEmployeeEligibleForEffortCertification(
            final String emplid,
            final EffortCertificationReportDefinition reportDefinition
    ) {
        final Map<String, Set<String>> earnCodePayGroups = effortCertificationReportDefinitionService
                .findReportEarnCodePayGroups(reportDefinition);
        final List<String> balanceTypeList = EffortConstants.ELIGIBLE_BALANCE_TYPES_FOR_EFFORT_REPORT;
        final Map<Integer, Set<String>> reportPeriods = reportDefinition.getReportPeriods();

        return laborModuleService.isEmployeeWithPayType(emplid, reportPeriods, balanceTypeList, earnCodePayGroups);
    }

    @Override
    public List<String> findEmployeesEligibleForEffortCertification(
            final EffortCertificationReportDefinition reportDefinition
    ) {
        final Map<String, Set<String>> earnCodePayGroups = effortCertificationReportDefinitionService
                .findReportEarnCodePayGroups(reportDefinition);
        final List<String> balanceTypeList = EffortConstants.ELIGIBLE_BALANCE_TYPES_FOR_EFFORT_REPORT;
        final Map<Integer, Set<String>> reportPeriods = reportDefinition.getReportPeriods();

        return laborModuleService.findEmployeesWithPayType(reportPeriods, balanceTypeList, earnCodePayGroups);
    }

    /**
     * check if a report has been defined. The combination of fiscal year and report number can determine a report
     * definition.
     *
     * @param fiscalYear   the the given fiscalYear
     * @param reportNumber the the given report number
     * @return a message if a report has not been defined or its documents have been generated; otherwise, return null
     */
    protected String validateReportDefinition(final Integer fiscalYear, final String reportNumber) {
        final EffortCertificationReportDefinition reportDefinition = new EffortCertificationReportDefinition();
        reportDefinition.setUniversityFiscalYear(fiscalYear);
        reportDefinition.setEffortCertificationReportNumber(reportNumber);

        final String errorMessage = effortCertificationReportDefinitionService
                .validateEffortCertificationReportDefinition(reportDefinition);
        return StringUtils.isNotEmpty(errorMessage) ? errorMessage : null;
    }

    /**
     * check if the documents for the given report definition have not been generated. The combination of fiscal year
     * and report number can determine a report definition.
     *
     * @param fieldValues the map containing fiscalYear and report number
     * @return a message if the documents have been generated; otherwise, return null
     */
    protected String existEffortCertificationDocument(final Map<String, String> fieldValues) {
        final String fiscalYear = fieldValues.get(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR);
        final String reportNumber = fieldValues.get(EffortPropertyConstants.EFFORT_CERTIFICATION_REPORT_NUMBER);

        // check if any document has been generated for the selected report definition
        final int countOfDocuments =
                businessObjectService.countMatching(EffortCertificationDocument.class, fieldValues);
        if (countOfDocuments > 0) {
            return MessageBuilder.buildMessageWithPlaceHolder(EffortKeyConstants.ERROR_REPORT_DOCUMENT_EXIST,
                    reportNumber, fiscalYear).getMessage();
        }

        return null;
    }

    /**
     * generate a document (build) as well as their detail lines for the given employees
     *
     * @param reportDefinition the given report definition
     * @param employees        the given employees
     * @param reportDataHolder the holder of report data
     * @param parameters       the given system parameters
     */
    protected void generateDocumentBuild(
            final EffortCertificationReportDefinition reportDefinition,
            final List<String> employees,
            final ExtractProcessReportDataHolder reportDataHolder,
            final Map<String, Collection<String>> parameters
    ) {
        final List<String> positionGroupCodes = effortCertificationReportDefinitionService
                .findPositionObjectGroupCodes(reportDefinition);
        final Integer postingYear = universityDateService.getCurrentFiscalYear();

        for (final String emplid : employees) {
            final EffortCertificationDocumentBuild document = generateDocumentBuildByEmployee(postingYear, emplid,
                    positionGroupCodes, reportDefinition, reportDataHolder, parameters);

            if (document != null) {
                final List<EffortCertificationDocumentBuild> documents = new ArrayList<
                        >();
                documents.add(document);

                businessObjectService.save(documents);

                reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_DETAIL_LINES_WRITTEN,
                        getCountOfDetailLines(documents));
                reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_CERTIFICATIONS_WRITTEN, documents.size());
            }
        }
        reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_EMPLOYEES_SELECTED, employees.size());
        reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_ERRORS_FOUND,
                reportDataHolder.getLedgerBalancesWithMessage().size());
    }

    /**
     * extract the qualified labor ledger balance records of the given employee with the given report periods.
     *
     * @param emplid             the given employee id
     * @param positionGroupCodes the specified position group codes
     * @param reportDefinition   the specified report definition
     * @param parameters         the system parameters setup in front
     * @param reportDataHolder   the given holder that contains any information to be written into the working report
     * @return the qualified labor ledger balance records of the given employee
     */
    protected List<LaborLedgerBalance> getQualifiedLedgerBalances(
            final String emplid,
            final List<String> positionGroupCodes,
            final EffortCertificationReportDefinition reportDefinition,
            final ExtractProcessReportDataHolder reportDataHolder,
            final Map<String, Collection<String>> parameters
    ) {
        // this getter has a side potential effect
        reportDefinition.getReportPeriods();

        // select ledger balances by the given employee and other criteria
        final Collection<LaborLedgerBalance> ledgerBalances = selectLedgerBalanceForEmployee(emplid, positionGroupCodes,
                reportDefinition, parameters);
        reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_BALANCES_READ, ledgerBalances.size());

        // clear up the ledger balance collection
        final List<LaborLedgerBalance> validLedgerBalances = removeUnqualifiedLedgerBalances(ledgerBalances,
                reportDefinition, reportDataHolder);
        reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_BALANCES_SELECTED, validLedgerBalances.size());

        // consolidate the pre-qualified ledger balances
        final List<LaborLedgerBalance> consolidatedLedgerBalances = consolidateLedgerBalances(validLedgerBalances,
                reportDefinition);

        // check the employee according to the pre-qualified ledger balances
        final boolean isQualifiedEmployee = checkEmployeeBasedOnLedgerBalances(emplid, consolidatedLedgerBalances,
                reportDefinition, reportDataHolder, parameters);

        // abort all ledger balances if the employee is not qualified; otherwise, adopt the consolidated balances
        return isQualifiedEmployee ? consolidatedLedgerBalances : null;
    }

    /**
     * remove the ledger balances without valid account, and nonzero total amount
     *
     * @param ledgerBalances   the given ledger balances
     * @param reportDefinition the given report definition
     * @param reportDataHolder the given holder that contains any information to be written into the working report
     */
    protected static List<LaborLedgerBalance> removeUnqualifiedLedgerBalances(
            final Collection<LaborLedgerBalance> ledgerBalances,
            final EffortCertificationReportDefinition reportDefinition,
            final ExtractProcessReportDataHolder reportDataHolder
    ) {
        final Map<Integer, Set<String>> reportPeriods = reportDefinition.getReportPeriods();
        final List<LedgerBalanceWithMessage> ledgerBalancesWithMessage =
                reportDataHolder.getLedgerBalancesWithMessage();

        final List<LaborLedgerBalance> validLedgerBalances = new ArrayList<>();

        for (final LaborLedgerBalance balance : ledgerBalances) {
            // within the given periods, the total amount of a single balance cannot be ZERO
            final Message errorAmountMessage =
                    LedgerBalanceFieldValidator.isNonZeroAmountBalanceWithinReportPeriod(balance, reportPeriods);

            // every balance record must be associated with a valid account
            final Message invalidAccountMessage = LedgerBalanceFieldValidator.hasValidAccount(balance);
            if (invalidAccountMessage != null) {
                reportInvalidLedgerBalance(ledgerBalancesWithMessage, balance, invalidAccountMessage);
            }

            if (errorAmountMessage == null && invalidAccountMessage == null) {
                validLedgerBalances.add(balance);
            }
        }

        return validLedgerBalances;
    }

    /**
     * check all ledger balances of the given employee and see if they can meet certain requirements. If not, the
     * employee would be unqualified for effort reporting
     *
     * @param emplid           the given employee id
     * @param ledgerBalances   the all pre-qualified ledger balances of the employee
     * @param reportDefinition the specified report definition
     * @param reportDataHolder the given holder that contains any information to be written into the working report
     * @param parameters       the system parameters setup in front
     * @return true if all ledger balances as whole meet requirements; otherwise, return false
     */
    protected boolean checkEmployeeBasedOnLedgerBalances(
            final String emplid,
            final List<LaborLedgerBalance> ledgerBalances,
            final EffortCertificationReportDefinition reportDefinition,
            final ExtractProcessReportDataHolder reportDataHolder,
            final Map<String, Collection<String>> parameters
    ) {
        if (ledgerBalances == null || ledgerBalances.isEmpty()) {
            return false;
        }

        final Map<Integer, Set<String>> reportPeriods = reportDefinition.getReportPeriods();
        final List<LedgerBalanceWithMessage> ledgerBalancesWithMessage =
                reportDataHolder.getLedgerBalancesWithMessage();

        // the total amount of all balances must be positive; otherwise, not to generate effort report for the employee
        final Message nonPositiveTotalError = LedgerBalanceFieldValidator.isTotalAmountPositive(ledgerBalances,
                reportPeriods);
        if (nonPositiveTotalError != null) {
            reportEmployeeWithoutValidBalances(ledgerBalancesWithMessage, nonPositiveTotalError, emplid);
            return false;
        }

        // the specified employee must have at least one grant account
        final Message grantAccountNotFoundError = LedgerBalanceFieldValidator.hasGrantAccount(ledgerBalances);
        if (grantAccountNotFoundError != null) {
            return false;
        }

        // check if there is at least one account funded by federal grants when an effort report can only be generated
        // for an employee with pay by federal grant
        final boolean isFederalFundsOnly =
                Boolean.parseBoolean(parameters.get(EffortParameterConstants.FEDERAL_ACCOUNT_IND).iterator().next());
        if (isFederalFundsOnly) {
            final Collection<String> federalAgencyTypeCodes = parameters.get(KfsParameterConstants.FEDERAL_AGENCY_TYPE);

            final Message federalFundsNotFoundError = LedgerBalanceFieldValidator.hasFederalFunds(ledgerBalances,
                    federalAgencyTypeCodes);
            if (federalFundsNotFoundError != null) {
                reportEmployeeWithoutValidBalances(ledgerBalancesWithMessage, federalFundsNotFoundError, emplid);
                return false;
            }
        }

        return true;
    }

    /**
     * select the labor ledger balances for the specified employee
     *
     * @param emplid                   the given employee id
     * @param positionObjectGroupCodes the specified position object group codes
     * @param reportDefinition         the specified report definition
     * @return the labor ledger balances for the specified employee
     */
    protected Collection<LaborLedgerBalance> selectLedgerBalanceForEmployee(
            final String emplid,
            final List<String> positionObjectGroupCodes,
            final EffortCertificationReportDefinition reportDefinition,
            final Map<String, Collection<String>> parameters
    ) {
        final Collection<String> expenseObjectTypeCodes = parameters.get(ExtractProcess.EXPENSE_OBJECT_TYPE);
        final Collection<String> excludedAccountTypeCode = parameters.get(EffortParameterConstants.ACCOUNT_TYPES);
        final List<String> emplids = Arrays.asList(emplid);
        final List<String> laborObjectCodes = Arrays.asList(EffortConstants.LABOR_OBJECT_SALARY_CODE);

        final Map<String, Collection<String>> fieldValues = new HashMap<>();
        fieldValues.put(KFSPropertyConstants.EMPLID, emplids);
        fieldValues.put(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE, expenseObjectTypeCodes);
        fieldValues.put(EffortPropertyConstants.LABOR_OBJECT_FRINGE_OR_SALARY_CODE, laborObjectCodes);

        final Map<String, Collection<String>> excludedFieldValues = new HashMap<>();
        excludedFieldValues.put(EffortPropertyConstants.ACCOUNT_ACCOUNT_TYPE_CODE, excludedAccountTypeCode);

        final Set<Integer> fiscalYears = reportDefinition.getReportPeriods().keySet();
        final List<String> balanceTypes = EffortConstants.ELIGIBLE_BALANCE_TYPES_FOR_EFFORT_REPORT;

        return laborModuleService.findLedgerBalances(fieldValues, excludedFieldValues, fiscalYears, balanceTypes,
                positionObjectGroupCodes);
    }

    /**
     * consolidate the given labor ledger balances and determine whether they are qualified for effort reporting
     *
     * @param ledgerBalances   the given labor ledger balances
     * @param reportDefinition the specified report definition
     * @return a collection of ledger balances if they are qualified; otherwise, return null
     */
    protected static List<LaborLedgerBalance> consolidateLedgerBalances(
            final List<LaborLedgerBalance> ledgerBalances,
            final EffortCertificationReportDefinition reportDefinition
    ) {
        final List<LaborLedgerBalance> consolidatedLedgerBalances = new ArrayList<>();

        final Map<Integer, Set<String>> reportPeriods = reportDefinition.getReportPeriods();
        final Map<String, LaborLedgerBalance> ledgerBalanceMap = new HashMap<>();
        LedgerBalanceConsolidationHelper.consolidateLedgerBalances(ledgerBalanceMap, ledgerBalances,
                getConsolidationKeys());

        for (final LaborLedgerBalance ledgerBalance : ledgerBalanceMap.values()) {
            final KualiDecimal totalAmount = LedgerBalanceConsolidationHelper.calculateTotalAmountWithinReportPeriod(
                    ledgerBalance, reportPeriods);
            if (totalAmount.isNonZero()) {
                consolidatedLedgerBalances.add(ledgerBalance);
            }
        }

        return consolidatedLedgerBalances;
    }

    // generate the effort certification document build for the given employee
    protected EffortCertificationDocumentBuild generateDocumentBuildByEmployee(
            final Integer postingYear,
            final String emplid,
            final List<String> positionGroupCodes,
            final EffortCertificationReportDefinition reportDefinition,
            final ExtractProcessReportDataHolder reportDataHolder,
            final Map<String, Collection<String>> parameters
    ) {
        final List<LaborLedgerBalance> qualifiedLedgerBalance = getQualifiedLedgerBalances(emplid, positionGroupCodes,
                reportDefinition, reportDataHolder, parameters);

        if (qualifiedLedgerBalance == null || qualifiedLedgerBalance.isEmpty()) {
            return null;
        }

        return effortCertificationDocumentBuildService.generateDocumentBuild(postingYear, reportDefinition,
                qualifiedLedgerBalance);
    }

    // add an error entry into error map
    protected static void reportInvalidLedgerBalance(
            final List<LedgerBalanceWithMessage> ledgerBalancesWithMessage,
            final LaborLedgerBalance ledgerBalance,
            final Message message
    ) {
        ledgerBalancesWithMessage.add(new LedgerBalanceWithMessage(ledgerBalance, message.toString()));
    }

    // add an error entry into error map
    protected void reportEmployeeWithoutValidBalances(
            final List<LedgerBalanceWithMessage> ledgerBalancesWithMessage,
            final Message message, final String emplid
    ) {
        final LaborLedgerBalance ledgerBalance = kualiModuleService.getResponsibleModuleService(
                LaborLedgerBalanceForEffortCertification.class).createNewObjectFromExternalizableClass(
                        LaborLedgerBalanceForEffortCertification.class);
        ledgerBalance.setEmplid(emplid);
        reportInvalidLedgerBalance(ledgerBalancesWithMessage, ledgerBalance, message);
    }

    // store and cache relating system parameters in a Map for the future use
    protected static Map<String, Collection<String>> getSystemParameters() {
        final Map<String, Collection<String>> parameters = new HashMap<>();

        parameters.put(EffortParameterConstants.ACCOUNT_TYPES,
                EffortCertificationParameterFinder.getAccountTypeCodes());
        parameters.put(EffortParameterConstants.FEDERAL_ACCOUNT_IND,
                EffortCertificationParameterFinder.getFederalOnlyBalanceIndicatorAsString());
        parameters.put(KfsParameterConstants.FEDERAL_AGENCY_TYPE,
                EffortCertificationParameterFinder.getFederalAgencyTypeCodes());

        return parameters;
    }

    // get the expense object code setup in System Options
    protected List<String> getExpenseObjectTypeCodes(final Integer fiscalYear) {
        final List<String> expenseObjectTypeCodes = new ArrayList<>();
        expenseObjectTypeCodes.add(optionsService.getOptions(fiscalYear).getFinObjTypeExpenditureexpCd());
        return expenseObjectTypeCodes;
    }

    // get the count of detail lines associated with the given documents
    protected static int getCountOfDetailLines(final List<EffortCertificationDocumentBuild> documents) {
        int numOfDetailLines = 0;
        for (final EffortCertificationDocumentBuild document : documents) {
            numOfDetailLines += document.getEffortCertificationDetailLinesBuild().size();
        }
        return numOfDetailLines;
    }

    // get the field names used to build the keys for record consolidation
    protected static List<String> getConsolidationKeys() {
        final List<String> consolidationKeys = new ArrayList<>();

        consolidationKeys.add(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR);
        consolidationKeys.add(KFSPropertyConstants.EMPLID);
        consolidationKeys.add(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        consolidationKeys.add(KFSPropertyConstants.ACCOUNT_NUMBER);
        consolidationKeys.add(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
        consolidationKeys.add(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
        consolidationKeys.add(KFSPropertyConstants.POSITION_NUMBER);

        return consolidationKeys;
    }

    // initialize the report data hold with default values
    protected static ExtractProcessReportDataHolder initializeReportData(
            final EffortCertificationReportDefinition reportDefinition
    ) {
        final ExtractProcessReportDataHolder reportDataHolder = new ExtractProcessReportDataHolder(reportDefinition);

        reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_BALANCES_READ, 0);
        reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_BALANCES_SELECTED, 0);
        reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_DETAIL_LINES_WRITTEN, 0);
        reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_CERTIFICATIONS_WRITTEN, 0);
        reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_EMPLOYEES_SELECTED, 0);
        reportDataHolder.updateBasicStatistics(ExtractProcess.NUM_ERRORS_FOUND, 0);

        return reportDataHolder;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setOptionsService(final OptionsService optionsService) {
        this.optionsService = optionsService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setLaborModuleService(final LaborModuleService laborModuleService) {
        this.laborModuleService = laborModuleService;
    }

    public void setEffortCertificationDocumentBuildService(
            final EffortCertificationDocumentBuildService effortCertificationDocumentBuildService
    ) {
        this.effortCertificationDocumentBuildService = effortCertificationDocumentBuildService;
    }

    public void setEffortCertificationReportService(
            final EffortCertificationReportService effortCertificationReportService
    ) {
        this.effortCertificationReportService = effortCertificationReportService;
    }

    public void setUniversityDateService(final UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }

    public void setEffortCertificationReportDefinitionService(
            final EffortCertificationReportDefinitionService effortCertificationReportDefinitionService
    ) {
        this.effortCertificationReportDefinitionService = effortCertificationReportDefinitionService;
    }

    public void setKualiModuleService(final KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }
}
