/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleService;
import org.kuali.kfs.integration.ld.LaborLedgerExpenseTransferAccountingLine;
import org.kuali.kfs.integration.ld.LaborLedgerExpenseTransferSourceAccountingLine;
import org.kuali.kfs.integration.ld.LaborLedgerExpenseTransferTargetAccountingLine;
import org.kuali.kfs.integration.ld.LaborModuleService;
import org.kuali.kfs.kew.actiontaken.ActionTaken;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.api.action.ActionRequestType;
import org.kuali.kfs.kew.api.action.WorkflowAction;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.UserSession;
import org.kuali.kfs.krad.bo.AdHocRoutePerson;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.ec.EffortConstants;
import org.kuali.kfs.module.ec.EffortKeyConstants;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationDetail;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationDetailBuild;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationDocumentBuild;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationReportDefinition;
import org.kuali.kfs.module.ec.document.EffortCertificationDocument;
import org.kuali.kfs.module.ec.document.validation.impl.EffortCertificationDocumentRuleUtil;
import org.kuali.kfs.module.ec.service.EffortCertificationDocumentService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.MessageBuilder;
import org.kuali.kfs.sys.businessobject.AccountingLineOverride;
import org.kuali.kfs.sys.businessobject.AccountingLineOverride.COMPONENT;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * To implement the services related to the effort certification document
 */
@Transactional
public class EffortCertificationDocumentServiceImpl implements EffortCertificationDocumentService {

    private static final Logger LOG = LogManager.getLogger();

    private LaborModuleService laborModuleService;
    private KualiModuleService kualiModuleService;
    private ContractsAndGrantsModuleService contractsAndGrantsModuleService;

    private DocumentService documentService;
    private BusinessObjectService businessObjectService;
    private PersonService personService;

    @Override
    public void processApprovedEffortCertificationDocument(final EffortCertificationDocument effortCertificationDocument) {
        final WorkflowDocument workflowDocument = effortCertificationDocument.getDocumentHeader().getWorkflowDocument();

        if (workflowDocument.isProcessed()) {
            GlobalVariables.setUserSession(new UserSession(KFSConstants.SYSTEM_USER));
            generateSalaryExpenseTransferDocument(effortCertificationDocument);
        }
    }

    @Override
    public boolean createAndRouteEffortCertificationDocument(
            final EffortCertificationDocumentBuild effortCertificationDocumentBuild) {
        final EffortCertificationDocument effortCertificationDocument = (EffortCertificationDocument) documentService
                .getNewDocument(EffortConstants.EffortDocumentTypes.EFFORT_CERTIFICATION_DOCUMENT);
        populateEffortCertificationDocument(effortCertificationDocument, effortCertificationDocumentBuild);
        documentService.routeDocument(effortCertificationDocument, KFSConstants.EMPTY_STRING, null);

        return true;
    }

    @Override
    public boolean populateEffortCertificationDocument(
            final EffortCertificationDocument effortCertificationDocument,
            final EffortCertificationDocumentBuild effortCertificationDocumentBuild) {
        // populate the fields of the document
        effortCertificationDocument.setUniversityFiscalYear(effortCertificationDocumentBuild.getUniversityFiscalYear());
        effortCertificationDocument.setEmplid(effortCertificationDocumentBuild.getEmplid());
        effortCertificationDocument.setEffortCertificationReportNumber(
                effortCertificationDocumentBuild.getEffortCertificationReportNumber());
        effortCertificationDocument.setEffortCertificationDocumentCode(
                effortCertificationDocumentBuild.getEffortCertificationDocumentCode());

        // populate the detail line of the document
        final List<EffortCertificationDetail> detailLines = effortCertificationDocument.getEffortCertificationDetailLines();
        detailLines.clear();

        final List<EffortCertificationDetailBuild> detailLinesBuild =
                effortCertificationDocumentBuild.getEffortCertificationDetailLinesBuild();
        for (final EffortCertificationDetailBuild detailLineBuild : detailLinesBuild) {
            detailLines.add(new EffortCertificationDetail(detailLineBuild));
        }

        // populate the document header of the document
        final DocumentHeader documentHeader = effortCertificationDocument.getDocumentHeader();
        documentHeader.setDocumentDescription(effortCertificationDocumentBuild.getEmplid());
        documentHeader.setFinancialDocumentTotalAmount(
                EffortCertificationDocument.getDocumentTotalAmount(effortCertificationDocument));

        return true;
    }

    @Override
    public void removeEffortCertificationDetailLines(final EffortCertificationDocument effortCertificationDocument) {
        final Map<String, String> fieldValues = new HashMap<>();
        fieldValues.put(KFSPropertyConstants.DOCUMENT_NUMBER, effortCertificationDocument.getDocumentNumber());

        businessObjectService.deleteMatching(EffortCertificationDetail.class, fieldValues);
    }

    @Override
    public boolean generateSalaryExpenseTransferDocument(final EffortCertificationDocument effortCertificationDocument) {
        final List<LaborLedgerExpenseTransferAccountingLine> sourceAccountingLines =
                buildSourceAccountingLines(effortCertificationDocument);
        final List<LaborLedgerExpenseTransferAccountingLine> targetAccountingLines =
                buildTargetAccountingLines(effortCertificationDocument);

        if (sourceAccountingLines.isEmpty() || targetAccountingLines.isEmpty()) {
            return true;
        }

        final String description = effortCertificationDocument.getEmplid();
        final String explanation = MessageBuilder.buildMessageWithPlaceHolder(
                EffortKeyConstants.MESSAGE_CREATE_SET_DOCUMENT_DESCRIPTION,
                effortCertificationDocument.getDocumentNumber()).toString();

        final String annotation = KFSConstants.EMPTY_STRING;
        final List<String> adHocRecipients = new ArrayList<>(
                getFiscalOfficersIfAmountChanged(effortCertificationDocument));

        laborModuleService.createAndBlankApproveSalaryExpenseTransferDocument(description, explanation,
                annotation, adHocRecipients, sourceAccountingLines, targetAccountingLines);
        return true;
    }

    @Override
    public void addRouteLooping(final EffortCertificationDocument effortCertificationDocument) {
        final WorkflowDocument workflowDocument = effortCertificationDocument.getDocumentHeader().getWorkflowDocument();
        final Set<Person> priorApprovers = getPriorApprovers(workflowDocument);

        for (final EffortCertificationDetail detailLine : effortCertificationDocument.getEffortCertificationDetailLines()) {
            final boolean hasBeenChanged = EffortCertificationDocumentRuleUtil.isPayrollAmountChangedFromPersisted(detailLine);
            if (!hasBeenChanged) {
                continue;
            }
            final boolean isNewLine = detailLine.isNewLineIndicator();
            LOG.info("EC Detail Line has been changed: {}", detailLine);

            final Account account = detailLine.getAccount();
            final Person fiscalOfficer = account.getAccountFiscalOfficerUser();
            if (fiscalOfficer != null && StringUtils.isNotBlank(fiscalOfficer.getPrincipalName())) {
                final AdHocRoutePerson adHocRoutePerson = buildAdHocRouteRecipient(fiscalOfficer.getPrincipalName(),
                        ActionRequestType.APPROVE);
                addAdHocRoutePerson(effortCertificationDocument.getAdHocRoutePersons(), priorApprovers,
                        adHocRoutePerson, isNewLine);
            } else {
                LOG.warn(
                        "Unable to obtain a fiscal officer for the detail line's account: {}-{}",
                        account::getChartOfAccountsCode,
                        account::getAccountNumber
                );
            }

            final Person projectDirector = contractsAndGrantsModuleService.getProjectDirectorForAccount(account);
            if (projectDirector != null) {
                final String accountProjectDirectorPersonUserId = projectDirector.getPrincipalName();
                final AdHocRoutePerson adHocRoutePerson = buildAdHocRouteRecipient(accountProjectDirectorPersonUserId,
                        ActionRequestType.APPROVE);
                addAdHocRoutePerson(effortCertificationDocument.getAdHocRoutePersons(), priorApprovers,
                        adHocRoutePerson, isNewLine);
            }
        }
    }

    /**
     * add the given ad hoc route person in the list if the person is one of prior approvers and is not in the list
     *
     * @param adHocRoutePersonList Collection of adhoc route persons
     * @param priorApprovers       Set of prior approvers
     * @param adHocRoutePerson     person to adhoc route to
     */
    protected void addAdHocRoutePerson(
            final Collection<AdHocRoutePerson> adHocRoutePersonList, final Set<Person> priorApprovers,
            final AdHocRoutePerson adHocRoutePerson) {
        addAdHocRoutePerson(adHocRoutePersonList, priorApprovers, adHocRoutePerson, false);
    }

    /**
     * add the given ad hoc route person in the list if the person is one of prior approvers, or the change was a new
     * line, and the person is not in the list
     *
     * @param adHocRoutePersonList Collection of adhoc route persons
     * @param priorApprovers       Set of prior approvers
     * @param adHocRoutePerson     person to adhoc route to
     * @param isNewLine            whether the change was a new line
     */
    protected void addAdHocRoutePerson(
            final Collection<AdHocRoutePerson> adHocRoutePersonList, final Set<Person> priorApprovers,
            final AdHocRoutePerson adHocRoutePerson, final boolean isNewLine) {
        boolean canBeAdded = false;

        // if it's a new line, person can be added even if they weren't a prior approver
        if (priorApprovers == null || isNewLine) {
            canBeAdded = true;
        } else {
            // we only want to ad-hoc if the user previously approved this document
            for (final Person approver : priorApprovers) {
                if (StringUtils.equals(approver.getPrincipalName(), adHocRoutePerson.getId())) {
                    canBeAdded = true;
                    break;
                }
            }
        }

        if (canBeAdded) {
            // check that we have not already added them for the same action
            for (final AdHocRoutePerson person : adHocRoutePersonList) {
                if (isSameAdHocRoutePerson(person, adHocRoutePerson)) {
                    canBeAdded = false;
                    break;
                }
            }
        }

        if (canBeAdded) {
            adHocRoutePersonList.add(adHocRoutePerson);
        }
    }

    protected boolean isSameAdHocRoutePerson(final AdHocRoutePerson person1, final AdHocRoutePerson person2) {
        if (person1 == null || person2 == null) {
            return false;
        }

        boolean isSameAdHocRoutePerson = StringUtils.equals(person1.getId(), person2.getId());
        isSameAdHocRoutePerson &= person1.getType().equals(person2.getType());
        isSameAdHocRoutePerson &= StringUtils.equals(person1.getActionRequested(), person2.getActionRequested());

        return isSameAdHocRoutePerson;
    }

    protected Set<Person> getPriorApprovers(final WorkflowDocument workflowDocument) {
        final List<ActionTaken> actionsTaken = workflowDocument.getActionsTaken();
        final Set<String> principalIds = new HashSet<>();
        final Set<Person> persons = new HashSet<>();

        for (final ActionTaken actionTaken : actionsTaken) {
            if (WorkflowAction.APPROVE.getCode().equals(actionTaken.getActionTaken())) {
                final String principalId = actionTaken.getPrincipalId();
                if (!principalIds.contains(principalId)) {
                    principalIds.add(principalId);
                    persons.add(personService.getPerson(principalId));
                }
            }
        }
        return persons;
    }

    /**
     * build an adhoc route recipient from the given person user id and action request
     *
     * @param personUserId  the given person user id
     * @param actionRequest the given action request
     * @return an adhoc route recipient built from the given information
     */
    protected AdHocRoutePerson buildAdHocRouteRecipient(final String personUserId, final ActionRequestType actionRequest) {
        final AdHocRoutePerson adHocRoutePerson = new AdHocRoutePerson();
        adHocRoutePerson.setActionRequested(actionRequest.getCode());
        adHocRoutePerson.setId(personUserId);
        return adHocRoutePerson;
    }

    /**
     * build the source accounting lines for a salary expense transfer document from the given effort certification
     * document. In the holder, the first item is source accounting line list and the second the target accounting
     * line list.
     *
     * @param effortCertificationDocument the given effort certification document
     * @return the source accounting lines for a salary expense transfer document built from the given effort
     *         certification document
     */
    protected List<LaborLedgerExpenseTransferAccountingLine> buildSourceAccountingLines(
            final EffortCertificationDocument effortCertificationDocument) {
        final List<LaborLedgerExpenseTransferAccountingLine> sourceAccountingLines = new ArrayList<>();

        final List<EffortCertificationDetail> effortCertificationDetailLines =
                effortCertificationDocument.getEffortCertificationDetailLines();
        for (final EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            if (getDifference(detailLine).isPositive()) {
                final LaborLedgerExpenseTransferSourceAccountingLine sourceLine = kualiModuleService
                        .getResponsibleModuleService(LaborLedgerExpenseTransferSourceAccountingLine.class)
                        .createNewObjectFromExternalizableClass(LaborLedgerExpenseTransferSourceAccountingLine.class);
                addAccountingLineIntoList(sourceAccountingLines, sourceLine, effortCertificationDocument,
                        detailLine);
            }
        }
        return sourceAccountingLines;
    }

    /**
     * build the target accounting lines for a salary expense transfer document from the given effort certification
     * document. In the holder, the first item is source accounting line list and the second the target accounting
     * line list.
     *
     * @param effortCertificationDocument the given effort certification document
     * @return the target accounting lines for a salary expense transfer document built from the given effort
     *         certification document
     */
    protected List<LaborLedgerExpenseTransferAccountingLine> buildTargetAccountingLines(
            final EffortCertificationDocument effortCertificationDocument) {
        final List<LaborLedgerExpenseTransferAccountingLine> targetAccountingLines = new ArrayList<>();

        final List<EffortCertificationDetail> effortCertificationDetailLines =
                effortCertificationDocument.getEffortCertificationDetailLines();
        for (final EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            if (getDifference(detailLine).isNegative()) {
                final LaborLedgerExpenseTransferTargetAccountingLine targetLine = kualiModuleService
                        .getResponsibleModuleService(LaborLedgerExpenseTransferTargetAccountingLine.class)
                        .createNewObjectFromExternalizableClass(LaborLedgerExpenseTransferTargetAccountingLine.class);
                addAccountingLineIntoList(targetAccountingLines, targetLine, effortCertificationDocument,
                        detailLine);
            }
        }
        return targetAccountingLines;
    }

    /**
     * get all fiscal officers of the detail line accounts where the salary amounts are changed
     *
     * @param effortCertificationDocument the given document that contains the detail lines
     * @return all fiscal officers of the detail line accounts where the salary amounts are changed
     */
    protected Set<String> getFiscalOfficersIfAmountChanged(final EffortCertificationDocument effortCertificationDocument) {
        final Set<String> fiscalOfficers = new HashSet<>();

        final List<EffortCertificationDetail> effortCertificationDetailLines =
                effortCertificationDocument.getEffortCertificationDetailLines();
        for (final EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            if (getDifference(detailLine).isNonZero()) {
                final Account account = detailLine.getAccount();
                final String accountFiscalOfficerPersonUserId = account.getAccountFiscalOfficerUser().getPrincipalName();

                if (StringUtils.isEmpty(accountFiscalOfficerPersonUserId)) {
                    fiscalOfficers.add(accountFiscalOfficerPersonUserId);
                }
            }
        }
        return fiscalOfficers;
    }

    /**
     * add a new accounting line into the given accounting line list. The accounting line is generated from the given
     * detail line
     *
     * @param accountingLineList          a list of accounting lines
     * @param effortCertificationDocument the given effort certification document that contains the given detail line
     * @param detailLine                  the given detail line that is used to generate an accounting line
     */
    protected void addAccountingLineIntoList(
            final List<LaborLedgerExpenseTransferAccountingLine> accountingLineList,
            final LaborLedgerExpenseTransferAccountingLine accountingLine,
            final EffortCertificationDocument effortCertificationDocument, final EffortCertificationDetail detailLine) {
        accountingLine.setSequenceNumber(accountingLineList.size() + 1);

        populateAccountingLine(effortCertificationDocument, detailLine, accountingLine);
        accountingLineList.add(accountingLine);
    }

    /**
     * populate an accounting line from the given detail line
     *
     * @param effortCertificationDocument the given effort certification document that contains the given detail line
     * @param detailLine                  the given detail line
     * @param accountingLine              the accounting line needed to be populated
     */
    protected void populateAccountingLine(
            final EffortCertificationDocument effortCertificationDocument,
            final EffortCertificationDetail detailLine, final LaborLedgerExpenseTransferAccountingLine accountingLine) {
        accountingLine.setChartOfAccountsCode(detailLine.getChartOfAccountsCode());
        accountingLine.setAccountNumber(detailLine.getAccountNumber());
        accountingLine.setSubAccountNumber(detailLine.getSubAccountNumber());

        accountingLine.setPostingYear(detailLine.getUniversityFiscalYear());
        accountingLine.setFinancialObjectCode(detailLine.getFinancialObjectCode());
        accountingLine.setBalanceTypeCode(KFSConstants.BALANCE_TYPE_ACTUAL);

        accountingLine.setAmount(getDifference(detailLine).abs());

        accountingLine.setFinancialSubObjectCode(null);
        accountingLine.setProjectCode(null);
        accountingLine.setOrganizationReferenceId(null);

        accountingLine.setEmplid(effortCertificationDocument.getEmplid());
        accountingLine.setPositionNumber(detailLine.getPositionNumber());
        accountingLine.setPayrollTotalHours(BigDecimal.ZERO);

        final EffortCertificationReportDefinition reportDefinition =
                effortCertificationDocument.getEffortCertificationReportDefinition();
        accountingLine.setPayrollEndDateFiscalYear(reportDefinition.getExpenseTransferFiscalYear());
        accountingLine.setPayrollEndDateFiscalPeriodCode(reportDefinition.getExpenseTransferFiscalPeriodCode());

        accountingLine.refreshNonUpdateableReferences();

        final AccountingLineOverride override = laborModuleService.determineNeededOverrides(null, accountingLine);

        // if an expired account override is needed, set it, otherwise validations on the downstream ST doc could fail
        accountingLine.setAccountExpiredOverrideNeeded(override.hasComponent(COMPONENT.EXPIRED_ACCOUNT));
        accountingLine.setAccountExpiredOverride(accountingLine.getAccountExpiredOverrideNeeded());

        // if a non-budgeted object code override is needed, set it, otherwise validations on the downstream ST doc
        // could fail
        accountingLine.setObjectBudgetOverrideNeeded(override.hasComponent(COMPONENT.NON_BUDGETED_OBJECT));
        accountingLine.setObjectBudgetOverride(accountingLine.isObjectBudgetOverrideNeeded());

        if (accountingLine.getAccountExpiredOverrideNeeded() || accountingLine.isObjectBudgetOverrideNeeded()) {
            accountingLine.setOverrideCode(override.getCode());
        }

    }

    /**
     * get the difference between the original amount and updated amount of the given detail line
     *
     * @param detailLine the given detail line
     * @return the difference between the original amount and updated amount of the given detail line
     */
    protected KualiDecimal getDifference(final EffortCertificationDetail detailLine) {
        return detailLine.getEffortCertificationOriginalPayrollAmount()
                .subtract(detailLine.getEffortCertificationPayrollAmount());
    }

    public void setLaborModuleService(final LaborModuleService laborModuleService) {
        this.laborModuleService = laborModuleService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setContractsAndGrantsModuleService(final ContractsAndGrantsModuleService contractsAndGrantsModuleService) {
        this.contractsAndGrantsModuleService = contractsAndGrantsModuleService;
    }

    public void setKualiModuleService(final KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }

    public void setPersonService(final PersonService personService) {
        this.personService = personService;
    }
}
