/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.util;

import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.module.ec.EffortParameterConstants;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * A convenient utility that can delegate the calling client to retrieve system parameters of effort certification
 * module.
 */
public final class EffortCertificationParameterFinder {

    private static ParameterService parameterService;

    /**
     * Private Constructor since this is a util class that should never be instantiated.
     */
    private EffortCertificationParameterFinder() {
    }

    /**
     * @return the run indicator setup in system parameters
     */
    public static boolean getRunIndicator() {
        return getParameterService().getParameterValueAsBoolean(
                KFSConstants.OptionalModuleNamespaces.EFFORT_CERTIFICATION,
                EffortParameterConstants.EFFORT_CERTIFICATION_COMPONENT_CODE,
                EffortParameterConstants.RUN_IND);
    }

    /**
     * @return the federal agency type codes setup in system parameters
     */
    public static Collection<String> getFederalAgencyTypeCodes() {
        return getParameterService().getParameterValuesAsString(KfsParameterConstants.FINANCIAL_SYSTEM_ALL.class,
                KfsParameterConstants.FEDERAL_AGENCY_TYPE);
    }

    public static boolean getFederalOnlyBalanceIndicator() {
        return getParameterService().getParameterValueAsBoolean(
                KFSConstants.OptionalModuleNamespaces.EFFORT_CERTIFICATION,
                EffortParameterConstants.EFFORT_CERTIFICATION_COMPONENT_CODE,
                EffortParameterConstants.FEDERAL_ACCOUNT_IND
        );
    }

    public static boolean getFederalOnlyRouteIndicator() {
        return getParameterService().getParameterValueAsBoolean(
                KFSConstants.OptionalModuleNamespaces.EFFORT_CERTIFICATION,
                EffortParameterConstants.EFFORT_CERTIFICATION_COMPONENT_CODE,
                EffortParameterConstants.FEDERAL_ROUTE_IND
        );
    }

    public static List<String> getFederalOnlyBalanceIndicatorAsString() {
        final List<String> indicatorList = new ArrayList<>();
        indicatorList.add(Boolean.toString(getFederalOnlyBalanceIndicator()));
        return indicatorList;
    }

    /**
     * @return the account type codes setup in system parameters
     */
    public static Collection<String> getAccountTypeCodes() {
        return getParameterService().getParameterValuesAsString(
                KFSConstants.OptionalModuleNamespaces.EFFORT_CERTIFICATION,
                EffortParameterConstants.EFFORT_CERTIFICATION_COMPONENT_CODE,
                EffortParameterConstants.ACCOUNT_TYPES
        );
    }

    /**
     * @return the report fiscal year setup in system parameters
     */
    public static Integer getExtractReportFiscalYear() {
        return Integer.valueOf(getParameterService().getParameterValueAsString(
                KFSConstants.OptionalModuleNamespaces.EFFORT_CERTIFICATION,
                EffortParameterConstants.EFFORT_CERTIFICATION_COMPONENT_CODE,
                EffortParameterConstants.EXTRACT_FISCAL_YEAR
        ));
    }

    /**
     * @return the report number setup in system parameters
     */
    public static String getExtractReportNumber() {
        return getParameterService().getParameterValueAsString(
                KFSConstants.OptionalModuleNamespaces.EFFORT_CERTIFICATION,
                EffortParameterConstants.EFFORT_CERTIFICATION_COMPONENT_CODE,
                EffortParameterConstants.EXTRACT_REPORT_NUMBER
        );
    }

    /**
     * @return the report fiscal year setup in system parameters
     */
    public static Integer getCreateReportFiscalYear() {
        return Integer.valueOf(getParameterService().getParameterValueAsString(
                KFSConstants.OptionalModuleNamespaces.EFFORT_CERTIFICATION,
                EffortParameterConstants.EFFORT_CERTIFICATION_COMPONENT_CODE,
                EffortParameterConstants.CREATE_FISCAL_YEAR));
    }

    /**
     * @return the report number setup in system parameters
     */
    public static String getCreateReportNumber() {
        return getParameterService().getParameterValueAsString(
                KFSConstants.OptionalModuleNamespaces.EFFORT_CERTIFICATION,
                EffortParameterConstants.EFFORT_CERTIFICATION_COMPONENT_CODE,
                EffortParameterConstants.CREATE_REPORT_NUMBER);
    }

    private static ParameterService getParameterService() {
        if (parameterService == null) {
            parameterService = SpringContext.getBean(ParameterService.class);
        }
        return parameterService;
    }

}
