/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.util;

import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.core.api.util.type.KualiDecimal;

/**
 * To hold the payroll amount and percent
 */
public class PayrollAmountHolder {

    private KualiDecimal payrollAmount;
    private Integer payrollPercent;

    private KualiDecimal totalAmount;

    private KualiDecimal accumulatedAmount;
    private Integer accumulatedPercent;

    /**
     * @param totalAmount        the total payroll amount
     * @param accumulatedAmount  the accumulated payroll amount
     * @param accumulatedPercent the accumulated payroll percent
     */
    public PayrollAmountHolder(final KualiDecimal totalAmount, final KualiDecimal accumulatedAmount, final Integer accumulatedPercent) {
        super();
        this.totalAmount = totalAmount;
        this.accumulatedAmount = accumulatedAmount;
        this.accumulatedPercent = accumulatedPercent;
    }

    public KualiDecimal getPayrollAmount() {
        return payrollAmount;
    }

    public void setPayrollAmount(final KualiDecimal payrollAmount) {
        this.payrollAmount = payrollAmount;
    }

    public Integer getPayrollPercent() {
        return payrollPercent;
    }

    public void setPayrollPercent(final Integer payrollPercent) {
        this.payrollPercent = payrollPercent;
    }

    public KualiDecimal getTotalAmount() {
        return totalAmount;
    }

    public void setTotalAmount(final KualiDecimal totalAmount) {
        this.totalAmount = totalAmount;
    }

    public KualiDecimal getAccumulatedAmount() {
        return accumulatedAmount;
    }

    public void setAccumulatedAmount(final KualiDecimal accumulatedAmount) {
        this.accumulatedAmount = accumulatedAmount;
    }

    public Integer getAccumulatedPercent() {
        return accumulatedPercent;
    }

    public void setAccumulatedPercent(final Integer accumulatedPercent) {
        this.accumulatedPercent = accumulatedPercent;
    }

    /**
     * calculate the payroll percentage based on the given information in payroll amount holder
     *
     * @param payrollAmountHolder the given payroll amount holder containing relating information
     */
    public static void calculatePayrollPercent(final PayrollAmountHolder payrollAmountHolder) {
        final KualiDecimal totalAmount = payrollAmountHolder.getTotalAmount();
        if (totalAmount.isZero()) {
            return;
        }

        final KualiDecimal payrollAmount = payrollAmountHolder.getPayrollAmount();
        KualiDecimal accumulatedAmount = payrollAmountHolder.getAccumulatedAmount();
        accumulatedAmount = accumulatedAmount.add(payrollAmount);

        int accumulatedPercent = payrollAmountHolder.getAccumulatedPercent();
        final int quotientOne = Math.round(payrollAmount.multiply(KFSConstants.CurrencyTypeAmounts.HUNDRED_DOLLAR_AMOUNT)
                .divide(totalAmount).floatValue());
        accumulatedPercent = accumulatedPercent + quotientOne;

        int quotientTwo = Math.round(accumulatedAmount.multiply(KFSConstants.CurrencyTypeAmounts.HUNDRED_DOLLAR_AMOUNT)
                .divide(totalAmount).floatValue());
        quotientTwo = quotientTwo - accumulatedPercent;

        payrollAmountHolder.setAccumulatedAmount(accumulatedAmount);
        payrollAmountHolder.setAccumulatedPercent(accumulatedPercent + quotientTwo);
        payrollAmountHolder.setPayrollPercent(quotientOne + quotientTwo);
    }

    /**
     * recalculate the payroll amount based on the given total amount and effort percent
     *
     * @param totalPayrollAmount the given total amount
     * @param effortPercent      the given effort percent
     * @return the payroll amount calculated from the given total amount and effort percent
     */
    public static KualiDecimal recalculatePayrollAmount(final KualiDecimal totalPayrollAmount, final Integer effortPercent) {
        final double amount = totalPayrollAmount.doubleValue() * effortPercent /
                              KFSConstants.CurrencyTypeAmounts.HUNDRED_DOLLAR_AMOUNT.doubleValue();

        return new KualiDecimal(amount);
    }

    /**
     * recalculate the effort percent based on the given total amount and payroll amount
     *
     * @param totalPayrollAmount the given total amount
     * @param payrollAmount      the given payroll amount
     * @return the effort percent calculated from the given total amount and payroll amount
     */
    public static Double recalculateEffortPercent(final KualiDecimal totalPayrollAmount, final KualiDecimal payrollAmount) {
        return payrollAmount.doubleValue() * KFSConstants.CurrencyTypeAmounts.HUNDRED_DOLLAR_AMOUNT.doubleValue() /
                totalPayrollAmount.doubleValue();
    }

    /**
     * recalculate the effort percent based on the given total amount and payroll amount and return it as of type
     * String
     *
     * @param totalPayrollAmount the given total amount
     * @param payrollAmount      the given payroll amount
     * @return the effort percent as String calculated from the given total amount and payroll amount
     */
    public static String recalculateEffortPercentAsString(final KualiDecimal totalPayrollAmount, final KualiDecimal payrollAmount) {
        double actualPercentAsDouble = 0;
        if (totalPayrollAmount.isNonZero()) {
            actualPercentAsDouble = recalculateEffortPercent(totalPayrollAmount, payrollAmount);
        }

        return String.format("%.4f%s", actualPercentAsDouble, KFSConstants.PERCENTAGE_SIGN);
    }
}
