/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.businessobject.SubAccount;
import org.kuali.kfs.coa.service.AccountService;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleService;
import org.kuali.kfs.integration.ld.LaborModuleService;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ec.document.EffortCertificationDocument;
import org.kuali.kfs.module.ec.util.EffortCertificationParameterFinder;
import org.kuali.kfs.module.ec.util.PayrollAmountHolder;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLineOverride;
import org.kuali.kfs.sys.businessobject.SystemOptions;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.core.api.util.type.KualiDecimal;

import java.util.Collection;
import java.util.List;

/**
 * Business Object for the Effort Certification Detail Table.
 */
public class EffortCertificationDetail extends PersistableBusinessObjectBase {

    private String documentNumber;
    private String chartOfAccountsCode;
    private String accountNumber;
    private String subAccountNumber;
    private String positionNumber;
    private String financialObjectCode;
    private String sourceChartOfAccountsCode;
    private String sourceAccountNumber;


    private KualiDecimal effortCertificationPayrollAmount;
    private KualiDecimal effortCertificationOriginalPayrollAmount;
    private Integer effortCertificationCalculatedOverallPercent;
    private Integer effortCertificationUpdatedOverallPercent;
    private String costShareSourceSubAccountNumber;

    private Integer universityFiscalYear;

    private KualiDecimal originalFringeBenefitAmount;

    private boolean accountExpiredOverride;
    private boolean accountExpiredOverrideNeeded;
    private String overrideCode = AccountingLineOverride.CODE.NONE;
    // to indicate if this detail line has been persisted or not
    private boolean newLineIndicator;

    // holds last saved updated payroll amount so business rule can check if it has been updated at the route level
    private KualiDecimal persistedPayrollAmount;
    private Integer persistedEffortPercent;
    private String groupId;

    private EffortCertificationDocument effortCertificationDocument;
    private ObjectCode financialObject;
    private Chart chartOfAccounts;
    private Account account;
    private Chart sourceChartOfAccounts;
    private Account sourceAccount;
    private SubAccount subAccount;
    private SystemOptions options;
    private Boolean federalOrFederalPassThroughIndicator;

    protected String effectiveDate;

    public EffortCertificationDetail() {
        super();

        if (SpringContext.isInitialized()) {
            subAccountNumber = KFSConstants.getDashSubAccountNumber();
        }

        effortCertificationPayrollAmount = KualiDecimal.ZERO;
        effortCertificationOriginalPayrollAmount = KualiDecimal.ZERO;
        effortCertificationCalculatedOverallPercent = 0;
        effortCertificationUpdatedOverallPercent = 0;
        originalFringeBenefitAmount = KualiDecimal.ZERO;
        effectiveDate = KFSConstants.EMPTY_STRING;
    }

    public EffortCertificationDetail(final EffortCertificationDetail effortCertificationDetail) {
        super();
        if (effortCertificationDetail != null) {
            chartOfAccountsCode = effortCertificationDetail.getChartOfAccountsCode();
            accountNumber = effortCertificationDetail.getAccountNumber();
            subAccountNumber = effortCertificationDetail.getSubAccountNumber();
            positionNumber = effortCertificationDetail.getPositionNumber();
            financialObjectCode = effortCertificationDetail.getFinancialObjectCode();
            sourceChartOfAccountsCode = effortCertificationDetail.getSourceChartOfAccountsCode();
            sourceAccountNumber = effortCertificationDetail.getSourceAccountNumber();
            effortCertificationPayrollAmount = effortCertificationDetail.getEffortCertificationPayrollAmount();
            effortCertificationCalculatedOverallPercent =
                    effortCertificationDetail.getEffortCertificationCalculatedOverallPercent();
            effortCertificationUpdatedOverallPercent =
                    effortCertificationDetail.getEffortCertificationUpdatedOverallPercent();
            universityFiscalYear = effortCertificationDetail.getUniversityFiscalYear();
            costShareSourceSubAccountNumber = effortCertificationDetail.getCostShareSourceSubAccountNumber();
            effortCertificationOriginalPayrollAmount =
                    effortCertificationDetail.getEffortCertificationOriginalPayrollAmount();
            originalFringeBenefitAmount = effortCertificationDetail.getOriginalFringeBenefitAmount();
            effectiveDate = effortCertificationDetail.getEffectiveDate();
        }
    }

    public String getDocumentNumber() {
        return documentNumber;
    }

    public void setDocumentNumber(final String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public String getChartOfAccountsCode() {
        return chartOfAccountsCode;
    }

    public void setChartOfAccountsCode(final String chartOfAccountsCode) {
        this.chartOfAccountsCode = chartOfAccountsCode;
    }

    public String getAccountNumber() {
        return accountNumber;
    }

    public void setAccountNumber(final String accountNumber) {
        this.accountNumber = accountNumber;
        // if accounts can't cross charts, set chart code whenever account number is set
        final AccountService accountService = SpringContext.getBean(AccountService.class);
        if (!accountService.accountsCanCrossCharts()) {
            final Account account = accountService.getUniqueAccountForAccountNumber(accountNumber);
            if (ObjectUtils.isNotNull(account)) {
                setChartOfAccountsCode(account.getChartOfAccountsCode());
                setChartOfAccounts(account.getChartOfAccounts());

            }
        }
    }

    public String getSubAccountNumber() {
        return subAccountNumber;
    }

    public void setSubAccountNumber(final String subAccountNumber) {
        this.subAccountNumber = subAccountNumber;
    }

    public String getPositionNumber() {
        return positionNumber;
    }

    public void setPositionNumber(final String positionNumber) {
        this.positionNumber = positionNumber;
    }

    public String getEffectiveDate() {
        return effectiveDate;
    }

    public void setEffectiveDate(final String effectiveDate) {
        this.effectiveDate = effectiveDate;
    }

    public String getFinancialObjectCode() {
        return financialObjectCode;
    }

    public void setFinancialObjectCode(final String financialObjectCode) {
        this.financialObjectCode = financialObjectCode;
    }

    public String getSourceChartOfAccountsCode() {
        return sourceChartOfAccountsCode;
    }

    public void setSourceChartOfAccountsCode(final String sourceChartOfAccountsCode) {
        this.sourceChartOfAccountsCode = sourceChartOfAccountsCode;
    }

    public String getSourceAccountNumber() {
        return sourceAccountNumber;
    }

    public void setSourceAccountNumber(final String sourceAccountNumber) {
        this.sourceAccountNumber = sourceAccountNumber;
    }

    public KualiDecimal getEffortCertificationPayrollAmount() {
        return effortCertificationPayrollAmount;
    }

    public void setEffortCertificationPayrollAmount(final KualiDecimal effortCertificationPayrollAmount) {
        this.effortCertificationPayrollAmount = effortCertificationPayrollAmount;
    }

    public Integer getEffortCertificationCalculatedOverallPercent() {
        return effortCertificationCalculatedOverallPercent;
    }

    public void setEffortCertificationCalculatedOverallPercent(final Integer effortCertificationCalculatedOverallPercent) {
        this.effortCertificationCalculatedOverallPercent = effortCertificationCalculatedOverallPercent;
    }

    public Integer getEffortCertificationUpdatedOverallPercent() {
        return effortCertificationUpdatedOverallPercent;
    }

    public void setEffortCertificationUpdatedOverallPercent(final Integer effortCertificationUpdatedOverallPercent) {
        this.effortCertificationUpdatedOverallPercent = effortCertificationUpdatedOverallPercent;
    }

    public String getCostShareSourceSubAccountNumber() {
        return costShareSourceSubAccountNumber;
    }

    public void setCostShareSourceSubAccountNumber(final String costShareSourceSubAccountNumber) {
        this.costShareSourceSubAccountNumber = costShareSourceSubAccountNumber;
    }

    public KualiDecimal getEffortCertificationOriginalPayrollAmount() {
        return effortCertificationOriginalPayrollAmount;
    }

    public void setEffortCertificationOriginalPayrollAmount(final KualiDecimal effortCertificationOriginalPayrollAmount) {
        this.effortCertificationOriginalPayrollAmount = effortCertificationOriginalPayrollAmount;
    }

    public ObjectCode getFinancialObject() {
        return financialObject;
    }

    @Deprecated
    public void setFinancialObject(final ObjectCode financialObject) {
        this.financialObject = financialObject;
    }

    public Chart getChartOfAccounts() {
        return chartOfAccounts;
    }

    @Deprecated
    public void setChartOfAccounts(final Chart chartOfAccounts) {
        this.chartOfAccounts = chartOfAccounts;
    }

    public EffortCertificationDocument getEffortCertificationDocument() {
        return effortCertificationDocument;
    }

    @Deprecated
    public void setEffortCertificationDocument(final EffortCertificationDocument effortCertificationDocument) {
        this.effortCertificationDocument = effortCertificationDocument;
    }

    public Account getAccount() {
        if (account == null && StringUtils.isNotBlank(getChartOfAccountsCode())
                && StringUtils.isNotBlank(getAccountNumber())) {
            refreshReferenceObject(KFSPropertyConstants.ACCOUNT);
        }

        return account;
    }

    @Deprecated
    public void setAccount(final Account account) {
        this.account = account;
    }

    public Chart getSourceChartOfAccounts() {
        return sourceChartOfAccounts;
    }

    @Deprecated
    public void setSourceChartOfAccounts(final Chart sourceChartOfAccounts) {
        this.sourceChartOfAccounts = sourceChartOfAccounts;
    }

    public Account getSourceAccount() {
        return sourceAccount;
    }

    @Deprecated
    public void setSourceAccount(final Account sourceAccount) {
        this.sourceAccount = sourceAccount;
    }

    public SubAccount getSubAccount() {
        return subAccount;
    }

    @Deprecated
    public void setSubAccount(final SubAccount subAccount) {
        this.subAccount = subAccount;
    }

    public SystemOptions getOptions() {
        return options;
    }

    @Deprecated
    public void setOptions(final SystemOptions options) {
        this.options = options;
    }

    public boolean isNewLineIndicator() {
        return newLineIndicator;
    }

    public void setNewLineIndicator(final boolean newLineIndicator) {
        this.newLineIndicator = newLineIndicator;
    }

    /**
     * If the account of this detail line is closed, the line cannot be edited.
     *
     * @return Returns true if line can be edited, false otherwise
     */
    public boolean isEditable() {
        return getAccount() == null || getAccount().isActive();
    }

    /**
     * Gets the federalOrFederalPassThroughIndicator attribute. If this line is associated with a valid account, the
     * indicator will be retrieved and updated.
     *
     * @return Returns the federalOrFederalPassThroughIndicator.
     */
    public boolean isFederalOrFederalPassThroughIndicator() {
        if (federalOrFederalPassThroughIndicator != null) {
            return federalOrFederalPassThroughIndicator;
        }
        if (getAccount() != null) {
            final Collection<String> federalAgencyTypeCodes = EffortCertificationParameterFinder.getFederalAgencyTypeCodes();
            federalOrFederalPassThroughIndicator = SpringContext.getBean(ContractsAndGrantsModuleService.class)
                    .isAwardedByFederalAgency(
                getAccount().getChartOfAccountsCode(), getAccount().getAccountNumber(), federalAgencyTypeCodes);
            return federalOrFederalPassThroughIndicator;
        }

        return false;
    }

    /**
     * This is a marker method, which does nothing.
     */
    public void setFederalOrFederalPassThroughIndicator(final boolean federalOrFederalPassThroughIndicator) {
    }

    public String getOverrideCode() {
        return overrideCode;
    }

    public void setOverrideCode(final String overrideCode) {
        this.overrideCode = overrideCode;
    }

    public KualiDecimal getFringeBenefitAmount() {
        final KualiDecimal payrollAmount = getEffortCertificationPayrollAmount();

        return EffortCertificationDetail.calculateFringeBenefit(this, payrollAmount);
    }

    /**
     * This is a marker method, which does nothing.
     */
    public void setFringeBenefitAmount(final KualiDecimal fringeBenefitAmount) {
    }

    public KualiDecimal getOriginalFringeBenefitAmount() {
        if (originalFringeBenefitAmount == null || originalFringeBenefitAmount.isZero()) {
            recalculateOriginalFringeBenefit();
        }
        return originalFringeBenefitAmount;
    }

    public void setOriginalFringeBenefitAmount(final KualiDecimal originalFringeBenefitAmount) {
        this.originalFringeBenefitAmount = originalFringeBenefitAmount;
    }

    public Integer getUniversityFiscalYear() {
        return universityFiscalYear;
    }

    public void setUniversityFiscalYear(final Integer universityFiscalYear) {
        this.universityFiscalYear = universityFiscalYear;
    }

    public KualiDecimal getPersistedPayrollAmount() {
        return persistedPayrollAmount;
    }

    public void setPersistedPayrollAmount(final KualiDecimal persistedPayrollAmount) {
        this.persistedPayrollAmount = persistedPayrollAmount;
    }

    public boolean isAccountExpiredOverride() {
        return accountExpiredOverride;
    }

    public void setAccountExpiredOverride(final boolean accountExpiredOverride) {
        this.accountExpiredOverride = accountExpiredOverride;
    }

    public boolean isAccountExpiredOverrideNeeded() {
        return accountExpiredOverrideNeeded;
    }

    public void setAccountExpiredOverrideNeeded(final boolean accountExpiredOverrideNeeded) {
        this.accountExpiredOverrideNeeded = accountExpiredOverrideNeeded;
    }

    /**
     * calculate the total effort percent of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total effort percent
     */
    public static Integer getTotalEffortPercent(final List<EffortCertificationDetail> effortCertificationDetailLines) {
        Integer totalEffortPercent = 0;

        for (final EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalEffortPercent += detailLine.getEffortCertificationUpdatedOverallPercent();
        }

        return totalEffortPercent;
    }

    /**
     * calculate the total persisted effort percent of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total persisted effort percent
     */
    public static Integer getTotalPersistedEffortPercent(final List<EffortCertificationDetail> effortCertificationDetailLines) {
        Integer totalEffortPercent = 0;

        for (final EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalEffortPercent += detailLine.getPersistedEffortPercent();
        }

        return totalEffortPercent;
    }

    /**
     * calculate the total original effort percent of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total original effort percent
     */
    public static Integer getTotalOriginalEffortPercent(
            final List<EffortCertificationDetail> effortCertificationDetailLines) {
        Integer totalOriginalEffortPercent = 0;

        for (final EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalOriginalEffortPercent += detailLine.getEffortCertificationCalculatedOverallPercent();
        }

        return totalOriginalEffortPercent;
    }

    /**
     * calculate the total payroll amount of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total original payroll amount
     */
    public static KualiDecimal getTotalPayrollAmount(final List<EffortCertificationDetail> effortCertificationDetailLines) {
        KualiDecimal totalPayrollAmount = KualiDecimal.ZERO;

        for (final EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalPayrollAmount = totalPayrollAmount.add(detailLine.getEffortCertificationPayrollAmount());
        }

        return totalPayrollAmount;
    }

    /**
     * calculate the total payroll amount of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total original payroll amount
     */
    public static KualiDecimal getTotalPersistedPayrollAmount(
            final List<EffortCertificationDetail> effortCertificationDetailLines) {
        KualiDecimal totalPayrollAmount = KualiDecimal.ZERO;

        for (final EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalPayrollAmount = totalPayrollAmount.add(detailLine.getPersistedPayrollAmount());
        }

        return totalPayrollAmount;
    }

    /**
     * calculate the total original payroll amount of the given detail lines
     *
     * @param effortCertificationDetailLines the given detail lines
     * @return Returns the total original payroll amount
     */
    public static KualiDecimal getTotalOriginalPayrollAmount(
            final List<EffortCertificationDetail> effortCertificationDetailLines) {
        KualiDecimal totalOriginalPayrollAmount = KualiDecimal.ZERO;

        for (final EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalOriginalPayrollAmount = totalOriginalPayrollAmount.add(detailLine
                    .getEffortCertificationOriginalPayrollAmount());
        }

        return totalOriginalPayrollAmount;
    }

    public static KualiDecimal getTotalFringeBenefit(final List<EffortCertificationDetail> effortCertificationDetailLines) {
        KualiDecimal totalFringeBenefit = KualiDecimal.ZERO;

        for (final EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalFringeBenefit = totalFringeBenefit.add(detailLine.getFringeBenefitAmount());
        }

        return totalFringeBenefit;
    }

    public static KualiDecimal getTotalOriginalFringeBenefit(
            final List<EffortCertificationDetail> effortCertificationDetailLines) {
        KualiDecimal totalOriginalFringeBenefit = KualiDecimal.ZERO;

        for (final EffortCertificationDetail detailLine : effortCertificationDetailLines) {
            totalOriginalFringeBenefit = totalOriginalFringeBenefit.add(detailLine.getOriginalFringeBenefitAmount());
        }

        return totalOriginalFringeBenefit;
    }

    /**
     * recalculate the payroll amount of the current detail line
     *
     * @param totalPayrollAmount the total payroll amount of the hosting document
     */
    public void recalculatePayrollAmount(final KualiDecimal totalPayrollAmount) {
        final Integer effortPercent = getEffortCertificationUpdatedOverallPercent();
        final KualiDecimal payrollAmount = PayrollAmountHolder.recalculatePayrollAmount(totalPayrollAmount, effortPercent);
        setEffortCertificationPayrollAmount(payrollAmount);
    }

    /**
     * recalculate the original fringe benefit of the current detail line
     */
    public void recalculateOriginalFringeBenefit() {
        final KualiDecimal originalPayrollAmount = getEffortCertificationOriginalPayrollAmount();
        final KualiDecimal fringeBenefit = EffortCertificationDetail.calculateFringeBenefit(this, originalPayrollAmount);
        setOriginalFringeBenefitAmount(fringeBenefit);
    }

    /**
     * recalculate the original fringe benefit of the current detail line
     */
    public static KualiDecimal calculateFringeBenefit(
            final EffortCertificationDetail detailLine,
            final KualiDecimal payrollAmount) {
        final LaborModuleService laborModuleService = SpringContext.getBean(LaborModuleService.class);
        final Integer fiscalYear = EffortCertificationParameterFinder.getCreateReportFiscalYear();
        final String chartOfAccountsCode = detailLine.getChartOfAccountsCode();
        final String objectCode = detailLine.getFinancialObjectCode();
        final String accountNumber = detailLine.getAccountNumber();
        final String subAccountNumber = detailLine.getSubAccountNumber();

        return laborModuleService.calculateFringeBenefit(fiscalYear, chartOfAccountsCode, objectCode, payrollAmount,
                accountNumber, subAccountNumber);
    }

    public Integer getPersistedEffortPercent() {
        return persistedEffortPercent;
    }

    public void setPersistedEffortPercent(final Integer persistedEffortPercent) {
        this.persistedEffortPercent = persistedEffortPercent;
    }

    public String getGroupId() {
        return groupId;
    }

    public void setGroupId(final String groupId) {
        this.groupId = groupId;
    }

}
