/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.document.web.struts;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.businessobject.SubAccount;
import org.kuali.kfs.coa.service.AccountService;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.kns.inquiry.Inquirable;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.lookup.HtmlData.AnchorHtmlData;
import org.kuali.kfs.kns.lookup.LookupUtils;
import org.kuali.kfs.krad.bo.DataObjectRelationship;
import org.kuali.kfs.krad.service.PersistenceStructureService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ec.EffortConstants;
import org.kuali.kfs.module.ec.EffortPropertyConstants;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationDetail;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationDetailLineOverride;
import org.kuali.kfs.module.ec.businessobject.inquiry.EffortLedgerBalanceInquirableImpl;
import org.kuali.kfs.module.ec.businessobject.inquiry.EffortPositionDataDetailsInquirableImpl;
import org.kuali.kfs.module.ec.document.EffortCertificationDocument;
import org.kuali.kfs.module.ec.util.PayrollAmountHolder;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.web.struts.FinancialSystemTransactionalDocumentFormBase;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.krad.bo.BusinessObject;

import javax.servlet.http.HttpServletRequest;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * Action form for Effort Certification Document.
 */
public class EffortCertificationForm extends FinancialSystemTransactionalDocumentFormBase {

    private static final Logger LOG = LogManager.getLogger();

    protected EffortCertificationDetail newDetailLine;

    public EffortCertificationForm() {
        super();

        setNewDetailLine(createNewDetailLine());
    }

    @Override
    protected String getDefaultDocumentTypeName() {
        return EffortConstants.EffortDocumentTypes.EFFORT_CERTIFICATION_DOCUMENT;
    }

    /**
     * @return the initialized detail line
     */
    public EffortCertificationDetail createNewDetailLine() {
        final EffortCertificationDetail detailLine = new EffortCertificationDetail();
        detailLine.setEffortCertificationUpdatedOverallPercent(null);
        detailLine.setEffortCertificationPayrollAmount(null);
        detailLine.setSubAccountNumber(null);
        return detailLine;
    }

    public EffortCertificationDetail getNewDetailLine() {
        return newDetailLine;
    }

    public void setNewDetailLine(final EffortCertificationDetail newDetailLine) {
        this.newDetailLine = newDetailLine;
    }

    public EffortCertificationDocument getEffortCertificationDocument() {
        return (EffortCertificationDocument) getDocument();
    }

    @Override
    public void populate(final HttpServletRequest request) {
        super.populate(request);

        for (final EffortCertificationDetail detailLine : getDetailLines()) {
            EffortCertificationDetailLineOverride.populateFromInput(detailLine);
        }
    }

    public List<EffortCertificationDetail> getDetailLines() {
        final EffortCertificationDocument effortCertificationDocument = (EffortCertificationDocument) getDocument();
        return effortCertificationDocument.getEffortCertificationDetailLines();
    }

    /**
     * @return the relationship metadata for the detail line fields
     */
    public Map<String, DataObjectRelationship> getRelationshipMetadata() {
        LOG.debug("getRelationshipMetadata() start");

        final PersistenceStructureService persistenceStructureService =
                SpringContext.getBean(PersistenceStructureService.class);

        final Map<String, DataObjectRelationship> relationshipMetadata = new HashMap<>();
        for (final String attributeName : getInquirableFieldNames()) {
            final Map<String, Class<? extends BusinessObject>> primitiveReference =
                    LookupUtils.getPrimitiveReference(newDetailLine, attributeName);

            if (primitiveReference != null && !primitiveReference.isEmpty()) {
                final DataObjectRelationship primitiveRelationship = getPrimitiveDataObjectRelationship(
                        persistenceStructureService.getRelationshipMetadata(newDetailLine.getClass(), attributeName));
                relationshipMetadata.put(attributeName, primitiveRelationship);
            }
        }

        return relationshipMetadata;
    }

    /**
     * @return the inquiryUrl for the detail lines in the document.
     */
    public List<Map<String, HtmlData>> getDetailLineFieldInquiryUrl() {
        LOG.debug("getDetailLineFieldInquiryUrl() start");
        return getDetailLineFieldInquiryUrl(getDetailLines());
    }

    /**
     * @return the inquiryUrl for the detail lines in the document.
     */
    protected List<Map<String, HtmlData>> getDetailLineFieldInquiryUrl(final List<EffortCertificationDetail> detailLines) {
        LOG.debug("getDetailLineFieldInquiryUrl(List<EffortCertificationDetail>) start");

        final Map<String, String> noninquirableFieldValues = getNoninquirableFieldValues();
        final Inquirable inquirable = getInquirable();

        final List<Map<String, HtmlData>> inquiryURL = new ArrayList<>();
        for (final EffortCertificationDetail detailLine : detailLines) {
            detailLine.refreshNonUpdateableReferences();

            final Map<String, HtmlData> inquiryURLForAttribute = new HashMap<>();
            for (final String attributeName : getInquirableFieldNames()) {
                // exclude the non inquirable field values
                final Object attributeValue = ObjectUtils.getPropertyValue(detailLine, attributeName);
                final String noninquirableFieldValue = noninquirableFieldValues.get(attributeName);
                if (noninquirableFieldValue != null && noninquirableFieldValue.equals(attributeValue)) {
                    continue;
                }

                final HtmlData inquiryHref;
                if (getCustomizedInquirableFieldNames().contains(attributeName)) {
                    inquiryHref = getCustomizedInquiryUrl(detailLine, attributeName);
                } else {
                    inquiryHref = inquirable.getInquiryUrl(detailLine, attributeName, false);
                }

                inquiryURLForAttribute.put(attributeName, inquiryHref);
            }

            inquiryURL.add(inquiryURLForAttribute);
        }

        return inquiryURL;
    }

    public List<Map<String, String>> getFieldInfo() {
        LOG.debug("getFieldInfo() start");
        return getFieldInfo(getDetailLines());
    }

    protected Map<String, String> getFieldInfo(final EffortCertificationDetail detailLine) {
        LOG.info("getFieldInfo(List<EffortCertificationDetail>) start");

        final EffortCertificationDocument document = (EffortCertificationDocument) getDocument();
        final KualiDecimal totalOriginalPayrollAmount = document.getTotalOriginalPayrollAmount();

        detailLine.refreshNonUpdateableReferences();

        final Map<String, String> fieldInfoForAttribute = new HashMap<>();

        fieldInfoForAttribute.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE,
                ObjectUtils.isNotNull(detailLine.getChartOfAccounts()) ?
                        detailLine.getChartOfAccounts().getFinChartOfAccountDescription() : "");

        final String accountInfo = buildAccountInfo(detailLine.getAccount());
        fieldInfoForAttribute.put(KFSPropertyConstants.ACCOUNT_NUMBER, accountInfo);

        final SubAccount subAccount = detailLine.getSubAccount();
        if (ObjectUtils.isNotNull(subAccount)) {
            fieldInfoForAttribute.put(KFSPropertyConstants.SUB_ACCOUNT_NUMBER, subAccount.getSubAccountName());
        }

        final ObjectCode objectCode = detailLine.getFinancialObject();
        if (ObjectUtils.isNotNull(objectCode)) {
            fieldInfoForAttribute.put(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, objectCode.getFinancialObjectCodeName());
        }

        final Account sourceAccount = detailLine.getSourceAccount();
        if (ObjectUtils.isNotNull(sourceAccount)) {
            fieldInfoForAttribute.put(EffortPropertyConstants.SOURCE_ACCOUNT_NUMBER, sourceAccount.getAccountName());
        }

        final Chart sourceChart = detailLine.getSourceChartOfAccounts();
        if (ObjectUtils.isNotNull(sourceChart)) {
            fieldInfoForAttribute.put(EffortPropertyConstants.SOURCE_CHART_OF_ACCOUNTS_CODE,
                    sourceChart.getFinChartOfAccountDescription());
        }

        final KualiDecimal originalPayrollAmount = detailLine.getEffortCertificationOriginalPayrollAmount();
        final String actualOriginalPercent = PayrollAmountHolder.recalculateEffortPercentAsString(totalOriginalPayrollAmount,
                originalPayrollAmount);
        fieldInfoForAttribute.put(EffortPropertyConstants.EFFORT_CERTIFICATION_CALCULATED_OVERALL_PERCENT,
                actualOriginalPercent);

        return fieldInfoForAttribute;
    }

    protected List<Map<String, String>> getFieldInfo(final List<EffortCertificationDetail> detailLines) {
        LOG.debug("getFieldInfo(List<EffortCertificationDetail>) start");

        final List<Map<String, String>> fieldInfo = new ArrayList<>();
        final EffortCertificationDocument document = (EffortCertificationDocument) getDocument();
        final KualiDecimal totalOriginalPayrollAmount = document.getTotalOriginalPayrollAmount();

        for (final EffortCertificationDetail detailLine : detailLines) {
            detailLine.refreshNonUpdateableReferences();

            final Map<String, String> fieldInfoForAttribute = new HashMap<>();

            fieldInfoForAttribute.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE,
                    detailLine.getChartOfAccounts().getFinChartOfAccountDescription());

            final String accountInfo = buildAccountInfo(detailLine.getAccount());
            fieldInfoForAttribute.put(KFSPropertyConstants.ACCOUNT_NUMBER, accountInfo);

            final SubAccount subAccount = detailLine.getSubAccount();
            if (ObjectUtils.isNotNull(subAccount)) {
                fieldInfoForAttribute.put(KFSPropertyConstants.SUB_ACCOUNT_NUMBER, subAccount.getSubAccountName());
            }

            final ObjectCode objectCode = detailLine.getFinancialObject();
            if (ObjectUtils.isNotNull(objectCode)) {
                fieldInfoForAttribute.put(KFSPropertyConstants.FINANCIAL_OBJECT_CODE,
                        objectCode.getFinancialObjectCodeName());
            }

            final Account sourceAccount = detailLine.getSourceAccount();
            if (ObjectUtils.isNotNull(sourceAccount)) {
                fieldInfoForAttribute.put(EffortPropertyConstants.SOURCE_ACCOUNT_NUMBER, sourceAccount.getAccountName());
            }

            final Chart sourceChart = detailLine.getSourceChartOfAccounts();
            if (ObjectUtils.isNotNull(sourceChart)) {
                fieldInfoForAttribute.put(EffortPropertyConstants.SOURCE_CHART_OF_ACCOUNTS_CODE,
                        sourceChart.getFinChartOfAccountDescription());
            }

            final KualiDecimal originalPayrollAmount = detailLine.getEffortCertificationOriginalPayrollAmount();
            final String actualOriginalPercent = PayrollAmountHolder.recalculateEffortPercentAsString(
                    totalOriginalPayrollAmount, originalPayrollAmount);
            fieldInfoForAttribute.put(EffortPropertyConstants.EFFORT_CERTIFICATION_CALCULATED_OVERALL_PERCENT,
                    actualOriginalPercent);

            fieldInfo.add(fieldInfoForAttribute);
        }

        return fieldInfo;
    }

    /**
     * pick up the primitive relationship for an attribute from a set of relationships. Generally, the primitive
     * relationship is that has the minimum number of primary keys.
     *
     * @param relationshipMetadata the relationship metadata that contains the primitive relationship
     * @return the primitive relationship for an attribute from a set of relationships.
     */
    protected DataObjectRelationship getPrimitiveDataObjectRelationship(
            final Map<String, DataObjectRelationship> relationshipMetadata) {
        int minCountOfKeys = Integer.MAX_VALUE;
        DataObjectRelationship primitiveRelationship = null;

        for (final String attribute : relationshipMetadata.keySet()) {
            final DataObjectRelationship currentRelationship = relationshipMetadata.get(attribute);

            final Map<String, String> parentToChildReferences = currentRelationship.getParentToChildReferences();
            if (parentToChildReferences.size() < minCountOfKeys) {
                minCountOfKeys = parentToChildReferences.size();
                primitiveRelationship = currentRelationship;
            }
        }
        return primitiveRelationship;
    }

    /**
     * get the inquiry URL for the specified attribute
     *
     * @param detailLine    the detail line containing the given attribute
     * @param attributeName the specified attribute name
     * @return the inquiry URL for the specified attribute
     */
    protected HtmlData getCustomizedInquiryUrl(final EffortCertificationDetail detailLine, final String attributeName) {
        if (StringUtils.equals(attributeName, KFSPropertyConstants.POSITION_NUMBER)) {
            final AnchorHtmlData inquiryHref = (AnchorHtmlData) getEffortPositionDataDetailsInquirableImpl()
                    .getInquiryUrl(detailLine, attributeName);
            inquiryHref.setHref(getCompleteURL(inquiryHref.getHref()));
            return inquiryHref;
        }

        final AnchorHtmlData inquiryHref = (AnchorHtmlData) getInquirable().getInquiryUrl(detailLine, attributeName, false);
        inquiryHref.setHref(getCompleteURL(inquiryHref.getHref()));

        return inquiryHref;
    }

    public Map<String, String> getNoninquirableFieldValues() {
        final Map<String, String> inquirableFieldNames = new HashMap<>();
        inquirableFieldNames.put(KFSPropertyConstants.SUB_ACCOUNT_NUMBER, KFSConstants.getDashSubAccountNumber());
        inquirableFieldNames.put(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, KFSConstants.getDashFinancialObjectCode());
        inquirableFieldNames.put(KFSPropertyConstants.POSITION_NUMBER, EffortConstants.DASH_POSITION_NUMBER);

        inquirableFieldNames.put(EffortPropertyConstants.SOURCE_CHART_OF_ACCOUNTS_CODE,
                EffortConstants.DASH_CHART_OF_ACCOUNTS_CODE);
        inquirableFieldNames.put(EffortPropertyConstants.SOURCE_ACCOUNT_NUMBER, EffortConstants.DASH_ACCOUNT_NUMBER);
        inquirableFieldNames.put(EffortPropertyConstants.COST_SHARE_SOURCE_SUB_ACCOUNT_NUMBER,
                KFSConstants.getDashSubAccountNumber());

        return inquirableFieldNames;
    }

    public List<String> getInquirableFieldNames() {
        final List<String> inquirableFieldNames = new ArrayList<>();
        inquirableFieldNames.add(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        inquirableFieldNames.add(KFSPropertyConstants.ACCOUNT_NUMBER);
        inquirableFieldNames.add(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
        inquirableFieldNames.add(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
        inquirableFieldNames.add(KFSPropertyConstants.POSITION_NUMBER);

        inquirableFieldNames.add(EffortPropertyConstants.SOURCE_CHART_OF_ACCOUNTS_CODE);
        inquirableFieldNames.add(EffortPropertyConstants.SOURCE_ACCOUNT_NUMBER);
        inquirableFieldNames.add(EffortPropertyConstants.COST_SHARE_SOURCE_SUB_ACCOUNT_NUMBER);

        inquirableFieldNames.add(EffortPropertyConstants.EFFORT_CERTIFICATION_ORIGINAL_PAYROLL_AMOUNT);
        inquirableFieldNames.add(EffortPropertyConstants.EFFORT_CERTIFICATION_PAYROLL_AMOUNT);

        return inquirableFieldNames;
    }

    /**
     * @return the inquirable field names that need to be handled specially
     */
    public List<String> getCustomizedInquirableFieldNames() {
        final List<String> inquirableFieldNames = new ArrayList<>();

        inquirableFieldNames.add(KFSPropertyConstants.POSITION_NUMBER);
        inquirableFieldNames.add(EffortPropertyConstants.EFFORT_CERTIFICATION_ORIGINAL_PAYROLL_AMOUNT);
        inquirableFieldNames.add(EffortPropertyConstants.EFFORT_CERTIFICATION_PAYROLL_AMOUNT);

        return inquirableFieldNames;
    }

    /**
     * append the extract query string into the given base URL
     *
     * @param baseURL the given base URL. If the parameter is blank, the base URL won't be changed
     * @return the complete URL built from the given base URL and extra query strings
     */
    protected String getCompleteURL(final String baseURL) {
        if (StringUtils.isBlank(baseURL)) {
            return baseURL;
        }

        final String completeURL = baseURL;
        final EffortCertificationDocument document = (EffortCertificationDocument) getDocument();
        final Properties properties = new Properties();

        properties.put(KFSPropertyConstants.EMPLID, document.getEmplid());
        properties.put(EffortPropertyConstants.EFFORT_CERTIFICATION_REPORT_NUMBER,
                document.getEffortCertificationReportNumber());
        properties.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, document.getUniversityFiscalYear());

        final StringBuilder queryString = new StringBuilder();
        for (final Object key : properties.keySet()) {
            queryString.append("&").append(key).append("=").append(properties.get(key));
        }

        return completeURL.concat(queryString.toString());
    }

    public Map<String, String> getDetailLineFieldInfo() {
        LOG.info("getSummarizedDetailLineFieldInfo() start");

        return getFieldInfo(getNewDetailLine());
    }

    protected Inquirable getInquirable() {
        return new EffortLedgerBalanceInquirableImpl();
    }

    protected EffortPositionDataDetailsInquirableImpl getEffortPositionDataDetailsInquirableImpl() {
        return new EffortPositionDataDetailsInquirableImpl();
    }

    /**
     * build the descriptive information of the given account. The information includes account name and project
     * director's name if any
     *
     * @param account   the given account
     * @return the descriptive information of the given account
     */
    public static String buildAccountInfo(final Account account) {
        if (ObjectUtils.isNull(account)) {
            return KFSConstants.EMPTY_STRING;
        }

        String projectDirectorName = KFSConstants.EMPTY_STRING;

        try {
            final ContractsAndGrantsModuleService contractsAndGrantsModuleService =
                    SpringContext.getBean(ContractsAndGrantsModuleService.class);
            final Person projectDirector = contractsAndGrantsModuleService.getProjectDirectorForAccount(account);

            projectDirectorName = projectDirector != null ? MessageFormat.format("  ({0})",
                    projectDirector.getName()) : KFSConstants.EMPTY_STRING;
        } catch (final Exception e) {
            LOG.error("Cannot find a project director for the account:{}", account);
        }

        return MessageFormat.format("{0}{1}", account.getAccountName(), projectDirectorName);
    }

    /**
     * load the descriptive information of the given account. This method is used by DWR.
     *
     * @param chartOfAccountsCode the given chart of accounts code
     * @param accountNumber       the given account number
     * @return the descriptive information of the given account
     */
    public static String loadAccountInfo(final String chartOfAccountsCode, final String accountNumber) {
        final Account account = SpringContext.getBean(AccountService.class).getByPrimaryIdWithCaching(chartOfAccountsCode,
                accountNumber);
        return buildAccountInfo(account);
    }
}
