/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ec.service.impl;

import org.kuali.kfs.integration.ld.LaborLedgerBalance;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationDetailBuild;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationDocumentBuild;
import org.kuali.kfs.module.ec.businessobject.EffortCertificationReportDefinition;
import org.kuali.kfs.module.ec.service.EffortCertificationDetailBuildService;
import org.kuali.kfs.module.ec.service.EffortCertificationDocumentBuildService;
import org.kuali.kfs.module.ec.util.LedgerBalanceConsolidationHelper;
import org.kuali.kfs.module.ec.util.PayrollAmountHolder;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This class Provide the facility used to generate documents (build) from the labor ledger balances
 */
@Transactional
public class EffortCertificationDocumentBuildServiceImpl implements EffortCertificationDocumentBuildService {

    protected EffortCertificationDetailBuildService effortCertificationDetailBuildService;
    protected BusinessObjectService businessObjectService;

    @Override
    public void removeExistingDocumentBuild(final Map<String, String> fieldValues) {
        final Collection<EffortCertificationDocumentBuild> documents =
                businessObjectService.findMatching(EffortCertificationDocumentBuild.class, fieldValues);
        businessObjectService.delete(new ArrayList<>(documents));
    }

    @Override
    public List<EffortCertificationDocumentBuild> generateDocumentBuildList(
            final Integer postingYear,
            final EffortCertificationReportDefinition reportDefinition, final List<LaborLedgerBalance> ledgerBalances) {
        final List<EffortCertificationDocumentBuild> documentList = new ArrayList<>();

        final Map<String, List<LaborLedgerBalance>> ledgerBalanceGroups = buildLedgerBalanceGroups(ledgerBalances);
        for (final String key : ledgerBalanceGroups.keySet()) {
            final List<LaborLedgerBalance> balanceList = ledgerBalanceGroups.get(key);

            final EffortCertificationDocumentBuild document = generateDocumentBuild(postingYear, reportDefinition,
                    balanceList);
            documentList.add(document);
        }

        return documentList;
    }

    @Override
    public EffortCertificationDocumentBuild generateDocumentBuild(
            final Integer postingYear,
            final EffortCertificationReportDefinition reportDefinition, final List<LaborLedgerBalance> ledgerBalances) {
        final Map<Integer, Set<String>> reportPeriods = reportDefinition.getReportPeriods();

        final KualiDecimal totalAmount = LedgerBalanceConsolidationHelper.calculateTotalAmountWithinReportPeriod(
                ledgerBalances, reportPeriods);
        final PayrollAmountHolder payrollAmountHolder = new PayrollAmountHolder(totalAmount, KualiDecimal.ZERO, 0);

        final LaborLedgerBalance headOfBalanceList = ledgerBalances.get(0);
        final EffortCertificationDocumentBuild document = populateDocument(reportDefinition, headOfBalanceList);
        final List<EffortCertificationDetailBuild> detailLineList = document.getEffortCertificationDetailLinesBuild();

        for (final LaborLedgerBalance balance : ledgerBalances) {
            final EffortCertificationDetailBuild detailLine =
                    effortCertificationDetailBuildService.generateDetailBuild(postingYear, balance, reportDefinition);
            detailLine.setEffortCertificationBuildNumber(document.getEffortCertificationBuildNumber());

            payrollAmountHolder.setPayrollAmount(detailLine.getEffortCertificationPayrollAmount());
            PayrollAmountHolder.calculatePayrollPercent(payrollAmountHolder);

            detailLine.setEffortCertificationCalculatedOverallPercent(payrollAmountHolder.getPayrollPercent());
            detailLine.setEffortCertificationUpdatedOverallPercent(payrollAmountHolder.getPayrollPercent());

            updateDetailLineList(detailLineList, detailLine);
        }

        return document;
    }

    /**
     * populate a document build object through the given information
     *
     * @param reportDefinition the given report definition
     * @param ledgerBalance    the given ledger balance
     * @return a document build object populated with the given information
     */
    protected static EffortCertificationDocumentBuild populateDocument(
            final EffortCertificationReportDefinition reportDefinition, final LaborLedgerBalance ledgerBalance) {
        final EffortCertificationDocumentBuild document = new EffortCertificationDocumentBuild();

        document.setEffortCertificationBuildNumber(null);
        document.setEffortCertificationDocumentCode(false);

        document.setEffortCertificationReportNumber(reportDefinition.getEffortCertificationReportNumber());
        document.setUniversityFiscalYear(reportDefinition.getUniversityFiscalYear());
        document.setEmplid(ledgerBalance.getEmplid());

        return document;
    }

    /**
     * group the given ledger balances according to the combination of the values in the specified fields
     *
     * @param ledgerBalances the given ledger balances
     * @return the map holding ledger balance groups
     */
    protected Map<String, List<LaborLedgerBalance>> buildLedgerBalanceGroups(final List<LaborLedgerBalance> ledgerBalances) {
        final Map<String, List<LaborLedgerBalance>> ledgerBalanceGroups = new HashMap<>();

        for (final LaborLedgerBalance balance : ledgerBalances) {
            final String consolidationKeys = balance.getEmplid();
            LedgerBalanceConsolidationHelper.groupLedgerBalancesByKeys(ledgerBalanceGroups, balance, consolidationKeys);
        }
        return ledgerBalanceGroups;
    }

    /**
     * update the given detail line if the given detail line is in the list; otherwise, add the given line into the
     * list
     *
     * @param detailLineList the given list of detail lines
     * @param detailLine     the given detail line
     */
    protected void updateDetailLineList(
            final List<EffortCertificationDetailBuild> detailLineList,

            final EffortCertificationDetailBuild detailLine) {
        final int index = detailLineList.indexOf(detailLine);
        if (index >= 0) {
            final EffortCertificationDetailBuild existingDetailLine = detailLineList.get(index);

            final int calculatedOverallPercent = existingDetailLine.getEffortCertificationCalculatedOverallPercent() +
                                                 detailLine.getEffortCertificationCalculatedOverallPercent();
            existingDetailLine.setEffortCertificationCalculatedOverallPercent(calculatedOverallPercent);

            final int updatedOverallPercent = existingDetailLine.getEffortCertificationUpdatedOverallPercent() +
                                              detailLine.getEffortCertificationUpdatedOverallPercent();
            existingDetailLine.setEffortCertificationUpdatedOverallPercent(updatedOverallPercent);

            final KualiDecimal originalPayrollAmount = existingDetailLine.getEffortCertificationOriginalPayrollAmount()
                    .add(detailLine.getEffortCertificationOriginalPayrollAmount());
            existingDetailLine.setEffortCertificationOriginalPayrollAmount(originalPayrollAmount);

            final KualiDecimal payrollAmount = existingDetailLine.getEffortCertificationPayrollAmount()
                    .add(detailLine.getEffortCertificationPayrollAmount());
            existingDetailLine.setEffortCertificationPayrollAmount(payrollAmount);

        } else {
            detailLineList.add(detailLine);
        }
    }

    public void setEffortCertificationDetailBuildService(
            final EffortCertificationDetailBuildService effortCertificationDetailBuildService) {
        this.effortCertificationDetailBuildService = effortCertificationDetailBuildService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }
}
