/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.uif.field;

import org.apache.commons.lang.StringUtils;
import org.kuali.kfs.krad.uif.UifConstants.Position;
import org.kuali.kfs.krad.uif.component.Component;
import org.kuali.kfs.krad.uif.component.ComponentBase;
import org.kuali.kfs.krad.uif.view.View;

import java.util.List;

/**
 * Contains a label for another <code>Field</code> instance.
 * <p>
 * <p>
 * The <code>LabelField</code> exists so that the label can be placed separate from the component in a layout manager
 * such as the <code>GridLayoutManager</code>. It addition it can be used to style the label (from the inherited
 * styleClass and style properties).
 */
public class LabelField extends FieldBase {

    private static final long serialVersionUID = -6491546893195180114L;

    private String labelText;
    private String labelForComponentId;

    private boolean renderColon;

    private Position requiredMessagePlacement;
    private MessageField requiredMessageField;

    public LabelField() {
        renderColon = true;
        requiredMessagePlacement = Position.LEFT;
    }

    /**
     * The following finalization is performed:
     * <p>
     * <ul>
     * <li>If label text is blank, set render to false for field</li>
     * </ul>
     *
     * @see ComponentBase#performFinalize(View, Object, Component)
     */
    @Override
    public void performFinalize(View view, Object model, Component parent) {
        super.performFinalize(view, model, parent);

        if (StringUtils.isBlank(getLabelText())) {
            setRender(false);
        }
    }

    /**
     * @see ComponentBase#getComponentsForLifecycle()
     */
    @Override
    public List<Component> getComponentsForLifecycle() {
        List<Component> components = super.getComponentsForLifecycle();
        components.add(requiredMessageField);
        return components;
    }

    /**
     * Used for setting the labelFor attribute of the corresponding HTML element. Note this gets set automatically by
     * the framework during the initialize phase.
     *
     * @return String component id for the component the label applies to.
     */
    public String getLabelForComponentId() {
        return this.labelForComponentId;
    }

    /**
     * @param labelForComponentId the component id the label applies to to set.
     */
    public void setLabelForComponentId(String labelForComponentId) {
        this.labelForComponentId = labelForComponentId;
    }

    /**
     * @return label text to display.
     */
    public String getLabelText() {
        return this.labelText;
    }

    /**
     * @param labelText the label text to set.
     */
    public void setLabelText(String labelText) {
        this.labelText = labelText;
    }

    /**
     * @return boolean true if a colon should be rendered after the label text, generally used when the label appears to
     *         the left of the field's control or value, false if it should not be.
     */
    public boolean isRenderColon() {
        return this.renderColon;
    }

    /**
     * @param renderColon the render colon indicator to set.
     */
    public void setRenderColon(boolean renderColon) {
        this.renderColon = renderColon;
    }

    /**
     * To indicate a field must have a value (required input) the required message field can be set to display an
     * indicator or message along with the label. The message field also dictates the styling of the required message.
     *
     * @return MessageField instance that will display a required indicator.
     */
    public MessageField getRequiredMessageField() {
        return this.requiredMessageField;
    }

    /**
     * @param requiredMessageField the required message field to set.
     */
    public void setRequiredMessageField(MessageField requiredMessageField) {
        this.requiredMessageField = requiredMessageField;
    }

    /**
     * @return Position where the required message field should be placed in relation to the label field, valid options
     *         are 'LEFT' and 'RIGHT'.
     */
    public Position getRequiredMessagePlacement() {
        return this.requiredMessagePlacement;
    }

    /**
     * @param requiredMessagePlacement the required message field placement to set.
     */
    public void setRequiredMessagePlacement(Position requiredMessagePlacement) {
        this.requiredMessagePlacement = requiredMessagePlacement;
    }

}
