/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.kns.datadictionary;

import org.kuali.kfs.krad.bo.BusinessObjectBase;
import org.kuali.kfs.krad.exception.AuthorizationException;
import org.kuali.rice.kim.api.identity.Person;

import java.io.File;

/**
 * Entity to manage business object operations, such as "download", "delete", "edit", etc. Authorization for the
 * operations is the responsibility of the implementation and not of the caller.
 */
public interface BusinessObjectAdminService {

    /**
     * @param businessObjectClass Class of the BusinessObject on whose behalf this query is being made.
     * @param person Person whom we are verifying has permission to create this business object.
     * @return {@code true} if this instance supports creating business objects via this interface
     * and this user has appropriate permissions to do so; {@code false}
     */
    boolean allowsNew(Class<? extends BusinessObjectBase> businessObjectClass, Person person);

    /**
     * @param businessObject BusinessObjectBase on whose behalf this query is being made.
     * @param person Person whom we are verifying has permission to download this business object.
     * @return {@code true} if this instance supports downloading business objects via this interface
     * and this user has appropriate permissions to do so; {@code false}
     */
    boolean allowsDownload(BusinessObjectBase businessObject, Person person);

    /**
     * @param id identifier for the instance of the business object that should be retrieved.
     * @return File containing the business object identified by the provided {@code id}.
     * @throws AuthorizationException if the user does not have authorization to delete the business object instance.
     * @throws EntityNotFoundException if a problem is encountered while attempting to access the information needed.
     */
    File download(String id) throws AuthorizationException, EntityNotFoundException;

    /**
     * @param businessObject BusinessObjectBase on whose behalf this query is being made.
     * @param person Person whom we are verifying has permission to delete this business object.
     * @return {@code true} if this instance supports deleting business objects via this interface
     * and this user has appropriate permissions to do so; {@code false}
     */
    boolean allowsDelete(BusinessObjectBase businessObject, Person person);

    /**
     * @param id identifier for the instance of the business object that should be deleted.
     * @return {@code true} if the delete operation was successful; {@code false} otherwise.
     * @throws AuthorizationException if the user does not have authorization to delete the business object instance.
     * @throws EntityNotFoundException if a problem is encountered while attempting to access the information needed.
     */
    boolean delete(String id) throws AuthorizationException, EntityNotFoundException;

    /**
     * @param businessObject BusinessObjectBase on whose behalf this query is being made.
     * @param person Person whom we are verifying has permission to edit this business object.
     * @return {@code true} if this instance supports editing business objects
     * and this user has appropriate permissions to do so; {@code false} otherwise.
     */
    boolean allowsEdit(BusinessObjectBase businessObject, Person person);

    /**
     * @param businessObject BusinessObjectBase on whose behalf this query is being made.
     * @param person Person whom we are verifying has permission to copy this business object.
     * @return {@code true} if this instance supports copying business objects
     * and this user has appropriate permissions to do so {@code false} otherwise.
     */
    boolean allowsCopy(BusinessObjectBase businessObject, Person person);

    /**
     * @param businessObjectClass BusinessObjectBase Class on whose behalf this query is being made.
     * @param person Person whom we are verifying has permission to initiate a business object of the provided class.
     * @return {@code true} if this instance supports initiation and this user has appropriate permissions to do so
     * {@code false} otherwise.
     */
    boolean allowsCreate(Class<? extends BusinessObjectBase> businessObjectClass, Person person);

}
