/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.service;

import org.kuali.kfs.krad.datadictionary.AttributeDefinition;
import org.kuali.kfs.krad.datadictionary.AttributeSecurity;
import org.kuali.kfs.krad.datadictionary.DataDictionary;
import org.kuali.kfs.krad.datadictionary.InactivationBlockingMetadata;
import org.kuali.kfs.krad.datadictionary.control.ControlDefinition;
import org.kuali.kfs.krad.datadictionary.exception.UnknownDocumentTypeException;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.uif.UifConstants.ViewType;
import org.kuali.kfs.krad.uif.view.View;
import org.kuali.rice.core.web.format.Formatter;
import org.kuali.rice.krad.bo.BusinessObject;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * Defines the API for interacting with the data dictionary.
 */
public interface DataDictionaryService {

    /**
     * Sequentially adds each package named (as a String) in the given List as a source of unique entries to the
     * DataDictionary being constructed. Duplicate entries among any of the XML files in any of these packages will 
     * result in exceptions being thrown, hence service-initialization failure.
     *
     * @param baselinePackages
     * @throws IOException if any of the given packages can't be located
     */
    void setBaselinePackages(List baselinePackages) throws IOException;

    /**
     * @return current DataDictionary.
     */
    DataDictionary getDataDictionary();

    void addDataDictionaryLocations(List<String> locations) throws IOException;

    /**
     * @return the html control type used to render the field.
     */
    ControlDefinition getAttributeControlDefinition(Class dataObjectClass, String attributeName);

    /**
     * @return the display size of the field if text control.
     */
    Integer getAttributeSize(Class dataObjectClass, String attributeName);

    /**
     * @return the max length defined for the given attribute name.
     */
    Integer getAttributeMaxLength(Class dataObjectClass, String attributeName);

    /**
     * @return the regular expression defined to validate the given attribute name.
     */
    Pattern getAttributeValidatingExpression(Class dataObjectClass, String attributeName);

    /**
     * @return the label to be used for displaying the attribute.
     */
    String getAttributeLabel(Class dataObjectClass, String attributeName);

    /**
     * @return the short label to be used for displaying the attribute.
     */
    String getAttributeShortLabel(Class dataObjectClass, String attributeName);

    /**
     * @return the "label (short label)" used for displaying error messages.
     */
    String getAttributeErrorLabel(Class dataObjectClass, String attributeName);

    /**
     * @return the formatter class used to format the attribute value.
     */
    Class<? extends Formatter> getAttributeFormatter(Class dataObjectClass, String attributeName);

    /**
     * @return whether or not to force input text into uppercase.
     */
    Boolean getAttributeForceUppercase(Class dataObjectClass, String attributeName);

    /**
     * @return short help text for attribute.
     */
    String getAttributeSummary(Class dataObjectClass, String attributeName);

    /**
     * @return detailed help text for attribute.
     */
    String getAttributeDescription(Class dataObjectClass, String attributeName);

    /**
     * @return whether or not the named attribute is required.
     */
    Boolean isAttributeRequired(Class dataObjectClass, String attributeName);

    /**
     * @return whether or not the named attribute is defined in the business object xml.
     */
    Boolean isAttributeDefined(Class dataObjectClass, String attributeName);

    /**
     * @return the label to be used for displaying the collection.
     */
    String getCollectionLabel(Class dataObjectClass, String collectionName);

    /**
     * @return the short label to be used for displaying the collection.
     */
    String getCollectionShortLabel(Class dataObjectClass, String collectionName);

    /**
     * @return short help text for collection.
     */
    String getCollectionSummary(Class dataObjectClass, String collectionName);

    /**
     * @return detailed help text for collection.
     */
    String getCollectionDescription(Class dataObjectClass, String collectionName);

    /**
     * @return the html control type used to render the field.
     */
    ControlDefinition getAttributeControlDefinition(String entryName, String attributeName);

    /**
     * @return the display size of the field if text control.
     */
    Integer getAttributeSize(String entryName, String attributeName);

    /**
     * @return the min length defined for the given attribute name.
     */
    Integer getAttributeMinLength(String entryName, String attributeName);

    /**
     * @return the max length defined for the given attribute name.
     */
    Integer getAttributeMaxLength(String entryName, String attributeName);

    /**
     * @param entryName
     * @param attributeName
     * @return the exclusive minimum for the specified attribute, or {@code null} if none.
     */
    String getAttributeExclusiveMin(String entryName, String attributeName);

    /**
     * @param entryName
     * @param attributeName
     * @return the inclusive maximum for the specified attribute, or {@code null} if none.
     */
    String getAttributeInclusiveMax(String entryName, String attributeName);

    /**
     * @return the regular expression defined to validate the given attribute name.
     */
    Pattern getAttributeValidatingExpression(String entryName, String attributeName);

    /**
     * @return the label to be used for displaying the attribute.
     */
    String getAttributeLabel(String entryName, String attributeName);

    /**
     * @return the short label to be used for displaying the attribute.
     */
    String getAttributeShortLabel(String entryName, String attributeName);

    /**
     * @return the "label (short label)" used for displaying error messages.
     */
    String getAttributeErrorLabel(String entryName, String attributeName);

    /**
     * @return the formatter class used to format the attribute value.
     */
    Class<? extends Formatter> getAttributeFormatter(String entryName, String attributeName);

    /**
     * @return whether or not to force input text into uppercase.
     */
    Boolean getAttributeForceUppercase(String entryName, String attributeName);

    /**
     * @return the AttributeSecurity object defined for the attribute's data value.
     */
    AttributeSecurity getAttributeSecurity(String entryName, String attributeName);

    /**
     * @return short help text for attribute.
     */
    String getAttributeSummary(String entryName, String attributeName);

    /**
     * @return detailed help text for attribute.
     */
    String getAttributeDescription(String entryName, String attributeName);

    String getAttributeValidatingErrorMessageKey(String entryName, String attributeName);

    String[] getAttributeValidatingErrorMessageParameters(String entryName, String attributeName);

    /**
     * @return whether or not the named attribute is required.
     */
    Boolean isAttributeRequired(String entryName, String attributeName);

    /**
     * @return whether or not the named attribute is defined in the business object xml.
     */
    Boolean isAttributeDefined(String entryName, String attributeName);

    /**
     * @return AttributeDefinition associated with the given attributeName within the given entry.
     */
    AttributeDefinition getAttributeDefinition(String entryName, String attributeName);

    /**
     * @return the label to be used for displaying the collection.
     */
    String getCollectionLabel(String entryName, String collectionName);

    /**
     * @return the short label to be used for displaying the collection.
     */
    String getCollectionShortLabel(String entryName, String collectionName);

    /**
     * @return the element label to be used for displaying the collection.
     */
    String getCollectionElementLabel(String entryName, String collectionName, Class dataObjectClass);

    /**
     * @return short help text for collection.
     */
    String getCollectionSummary(String entryName, String collectionName);

    /**
     * @return detailed help text for collection.
     */
    String getCollectionDescription(String entryName, String collectionName);

    /**
     * @param entryName
     * @param relationshipName
     * @return source Class for the given relationship, or null if there is no relationship with that name.
     */
    Class<? extends BusinessObject> getRelationshipSourceClass(String entryName, String relationshipName);

    /**
     * @param entryName
     * @param relationshipName
     * @return target Class for the given relationship, or null if there is no relationship with that name.
     */
    Class<? extends BusinessObject> getRelationshipTargetClass(String entryName, String relationshipName);

    /**
     * @param entryName
     * @param relationshipName
     * @return List<String> of source attributeNames for the given relationship, or null if there is no relationship
     *         with that name.
     */
    List<String> getRelationshipSourceAttributes(String entryName, String relationshipName);

    /**
     * @param entryName
     * @param relationshipName
     * @return List<String> of target attributeNames for the given relationship, or null if there is no relationship
     *         with that name.
     */
    List<String> getRelationshipTargetAttributes(String entryName, String relationshipName);

    /**
     * @param entryName        Name of the Business Object entry
     * @param relationshipName Name of the relationship
     * @return Map that specifies the attributes of the relationship; Target field as key, source field as value.
     */
    Map<String, String> getRelationshipAttributeMap(String entryName, String relationshipName);

    /**
     * @param entryName           Name of the Business Object entry
     * @param sourceAttributeName name of the source attribute
     * @return the names of all entries that use the sourceAttributeName as a primitive attribute.
     */
    List<String> getRelationshipEntriesForSourceAttribute(String entryName, String sourceAttributeName);

    /**
     * @param entryName           Name of the Business Object entry
     * @param targetAttributeName name of the target attribute
     * @return the names of all entries that use the targetAttributeName as a primitive attribute.
     */
    List<String> getRelationshipEntriesForTargetAttribute(String entryName, String targetAttributeName);

    /**
     * @param entryName        name of the BO entry
     * @param relationshipName name of the relationship for the entry
     * @return true if there is a relationship defined for the given entry with the given name.
     */
    boolean hasRelationship(String entryName, String relationshipName);

    /**
     * @param entryName of the BO entry
     * @return a list of all DD defined mappings for the given BO.
     */
    List<String> getRelationshipNames(String entryName);

    /**
     * @param documentTypeName
     * @return the user friendly label based on the workflow doc type name.
     */
    String getDocumentLabelByTypeName(String documentTypeName);

    /**
     * @param documentOrBusinessObjectClass
     * @return the user friendly label based on the document or business object class.
     */
    String getDocumentLabelByClass(Class documentOrBusinessObjectClass);

    /**
     * @param documentClass
     * @return the document type name declared in the dd for the given document class. If no valid document type is
     *         found {@code null} is returned.
     */
    String getDocumentTypeNameByClass(Class documentClass);

    /**
     * @param documentClass
     * @return the document type name declared in the dd for the given document class. If no valid document type is
     *         found an {@link UnknownDocumentTypeException} is thrown.
     */
    String getValidDocumentTypeNameByClass(Class documentClass);

    /**
     * @param documentTypeName
     * @return the document class declared in the dd for the given document type name. If no document entry is found
     *         with given document type name, {@code null} will be returned.
     */
    Class<? extends Document> getDocumentClassByTypeName(String documentTypeName);

    /**
     * @param documentTypeName
     * @return the document class declared in the dd for the given document type name. If no document entry is found
     *         with given document type name, and {@link UnknownDocumentTypeException} will be thrown.
     */
    Class<? extends Document> getValidDocumentClassByTypeName(String documentTypeName);

    /**
     * @param businessObjectClass business object class to get configured list for
     * @return the list of attributes that should be used for grouping when determining the current status of a
     *         business object that implements InactivateableFromTo.
     */
    List<String> getGroupByAttributesForEffectiveDating(Class businessObjectClass);

    /**
     * @param inactivationBlockedBusinessObjectClass
     * @return a set of all inactivation blocks registered for a particular business object.
     */
    Set<InactivationBlockingMetadata> getAllInactivationBlockingDefinitions(
        Class inactivationBlockedBusinessObjectClass);

    /**
     * @param viewId unique id for view
     * @return View instance associated with the id.
     */
    View getViewById(String viewId);

    /**
     * @param id id or name for the bean definition
     * @return object instance from the dictionary by its spring bean name or id.
     */
    Object getDictionaryObject(String id);

    /**
     * @param id id of the bean to check for
     * @return boolean true if dictionary contains bean with the given id, false otherwise.
     */
    boolean containsDictionaryObject(String id);

    /**
     * @param viewTypeName type name for the view
     * @param indexKey     Map of index key parameters, these are the parameters the indexer used to index the view
     *                     initially and needs to identify an unique view instance
     * @return View instance identified by the view type name and index.
     */
    View getViewByTypeIndex(ViewType viewTypeName, Map<String, String> indexKey);
}
