/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.service;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.service.BusinessObjectDictionaryService;
import org.kuali.kfs.kns.service.KNSServiceLocator;
import org.kuali.kfs.krad.bo.BusinessObjectBase;
import org.kuali.kfs.krad.datadictionary.LookupResultAttributeDefinition;
import org.kuali.kfs.kns.inquiry.Inquirable;
import org.kuali.rice.kim.api.identity.Person;

import javax.ws.rs.core.MultivaluedMap;
import java.util.List;
import java.util.Map;

public abstract class LookupSearchService {
    static protected String KRAD_URL_PREFIX = "kr/";
    private static final Log LOG = LogFactory.getLog(LookupSearchService.class);

    private BusinessObjectDictionaryService businessObjectDictionaryService;

    /**
     * @param businessObjectClass Class of the BusinessObject to be searched for
     * @param fieldValues         {@link MultivaluedMap} containing the search parameters; in the normal expected use
     *                            case, these will be query parameters.
     * @return a List of all the results; each result extends BusinessObjectBase.
     */
    public abstract List<BusinessObjectBase> getSearchResults(Class<? extends BusinessObjectBase> businessObjectClass,
            MultivaluedMap<String, String> fieldValues);

    /**
     *
     * @param businessObject Business object used to check for actions
     * @param user Person object to check if the given user has permission to perform available actions
     * @return a List of actions links
     */
    public abstract List<Map<String, Object>> getActionLinks(BusinessObjectBase businessObject, Person user);

    /**
     *
     * @param businessObject The business object to serialize
     * @param user The user to check for permission for particular actions on the business object
     * @return A Map object where the keys are properties of the BusinessObject and values of those properties
     */
    public abstract Map<String, Object> serializeBusinessObject(BusinessObjectBase businessObject, Person user);

    /**
     * @param classForType Class of the BusinessObject whose search results shape is desired.
     * @return List of {@link LookupResultAttributeDefinition}s depicting the attributes to be included in search
     *         results.
     */
    public List<LookupResultAttributeDefinition> getSearchResultsAttributes(Class classForType) {
        return getLookupResultAttributeDefinitionsForClass(classForType);
    }

    /**
     * @param businessObject The business object used to check for inquiry
     * @param resultAttributeDefinition The attribute on the business object to check for an inquiry
     * @return Boolean denoting whether or not an inquiry exists for the business object property
     */
    public boolean hasInquiry(BusinessObjectBase businessObject, LookupResultAttributeDefinition resultAttributeDefinition) {
        String url = inquiryUrl(businessObject, resultAttributeDefinition.getName());
        url = url.replace(KRAD_URL_PREFIX, "");
        return !resultAttributeDefinition.isDisableInquiry() && !StringUtils.isEmpty(url);
    }

    /**
     * @param businessObject The business object used to build the inquiry URL
     * @param propertyName The property name used on the business object to check for an inquiry URL
     * @return A string representing a URL to the inquiry for the property name and business object given
     */
    public String inquiryUrl(BusinessObjectBase businessObject, String propertyName) {
        Class<? extends BusinessObjectBase> businessObjectClass = businessObject.getClass();
        Class<Inquirable> inquirableClass = businessObjectDictionaryService.getInquirableClass(businessObjectClass);

        Inquirable inquirable;
        if (inquirableClass != null) {
            try {
                inquirable = inquirableClass.newInstance();
            } catch (InstantiationException | IllegalAccessException ex) {
                LOG.warn("Could not instantiate inquirable for class" + inquirableClass.getSimpleName());
                return "";
            }
        } else {
            inquirable = KNSServiceLocator.getKualiInquirable();
        }

        Boolean forceLookupResultFieldInquiry =
                businessObjectDictionaryService.forceLookupResultFieldInquiry(businessObjectClass, propertyName);
        forceLookupResultFieldInquiry = Boolean.TRUE.equals(forceLookupResultFieldInquiry);

        HtmlData.AnchorHtmlData inquiryData =
                (HtmlData.AnchorHtmlData) inquirable.getInquiryUrl(businessObject, propertyName, forceLookupResultFieldInquiry);
        return KRAD_URL_PREFIX + inquiryData.getHref();
    }

    protected List<LookupResultAttributeDefinition> getLookupResultAttributeDefinitionsForClass(Class classForType) {
        return businessObjectDictionaryService.getLookupResultAttributeDefinitions(classForType);
    }

    public void setBusinessObjectDictionaryService(BusinessObjectDictionaryService businessObjectDictionaryService) {
        this.businessObjectDictionaryService = businessObjectDictionaryService;
    }

}
