/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.service.impl;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.reflection.ObjectAccessException;
import com.thoughtworks.xstream.converters.reflection.PureJavaReflectionProvider;
import com.thoughtworks.xstream.converters.reflection.ReflectionConverter;
import com.thoughtworks.xstream.converters.reflection.ReflectionProvider;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.mapper.Mapper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.ojb.broker.core.proxy.ListProxyDefaultImpl;
import org.kuali.kfs.krad.service.KRADServiceLocator;
import org.kuali.kfs.krad.service.PersistenceService;
import org.kuali.kfs.krad.service.XmlObjectSerializerService;
import org.kuali.kfs.krad.util.DateTimeConverter;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * This class is the service implementation for the XmlObjectSerializer structure. This is the default implementation that gets
 * delivered with Kuali. It utilizes the XStream open source libraries and framework.
 */
public class XmlObjectSerializerServiceImpl implements XmlObjectSerializerService {
    private static final Log LOG = LogFactory.getLog(XmlObjectSerializerServiceImpl.class);

    private PersistenceService persistenceService;

    private XStream xstream;

    public XmlObjectSerializerServiceImpl() {
        xstream = new XStream(new ProxyAwareJavaReflectionProvider());
        xstream.registerConverter(new ProxyConverter(xstream.getMapper(), xstream.getReflectionProvider()));
        xstream.addDefaultImplementation(ArrayList.class, ListProxyDefaultImpl.class);
        xstream.registerConverter(new DateTimeConverter());
    }

    public String toXml(Object object) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("toXml(" + object + ") : \n" + xstream.toXML(object));
        }
        return xstream.toXML(object);
    }

    public Object fromXml(String xml) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("fromXml() : \n" + xml);
        }
        if (xml != null) {
            xml = xml.replaceAll("--EnhancerByCGLIB--[0-9a-f]{0,8}", "");
        }
        return xstream.fromXML(xml);
    }

    /**
     * This custom converter only handles proxies for BusinessObjects.  List-type proxies are handled by configuring XStream to treat
     * ListProxyDefaultImpl as ArrayLists (see constructor for this service).
     */
    public class ProxyConverter extends ReflectionConverter {
        public ProxyConverter(Mapper mapper, ReflectionProvider reflectionProvider) {
            super(mapper, reflectionProvider);
        }

        @Override
        // since the ReflectionConverter supertype defines canConvert without using a parameterized Class type, we must declare
        // the overridden version the same way
        @SuppressWarnings("unchecked")
        public boolean canConvert(Class clazz) {
            return clazz.getName().contains("CGLIB");
        }

        @Override
        public void marshal(Object obj, HierarchicalStreamWriter writer, MarshallingContext context) {
            super.marshal(getPersistenceService().resolveProxy(obj), writer, context);
        }

        // we shouldn't need an unmarshal method because all proxy metadata is taken out of the XML, so we'll reserialize as a base BO.
    }

    public class ProxyAwareJavaReflectionProvider extends PureJavaReflectionProvider {

        public ProxyAwareJavaReflectionProvider() {
            super();
        }

        @Override
        public void visitSerializableFields(Object object, Visitor visitor) {
            for (Iterator iterator = fieldDictionary.serializableFieldsFor(object.getClass()); iterator.hasNext(); ) {
                Field field = (Field) iterator.next();
                if (!fieldModifiersSupported(field)) {
                    continue;
                }
                validateFieldAccess(field);
                Object value = null;
                try {
                    value = field.get(object);
                    if (value != null && getPersistenceService().isProxied(value)) {
                        value = getPersistenceService().resolveProxy(value);
                    }
                } catch (IllegalArgumentException e) {
                    throw new ObjectAccessException("Could not get field " + field.getClass() + "." + field.getName(), e);
                } catch (IllegalAccessException e) {
                    throw new ObjectAccessException("Could not get field " + field.getClass() + "." + field.getName(), e);
                }
                visitor.visit(field.getName(), field.getType(), field.getDeclaringClass(), value);
            }
        }

    }

    public PersistenceService getPersistenceService() {
        if (persistenceService == null) {
            persistenceService = KRADServiceLocator.getPersistenceService();
        }
        return persistenceService;
    }

}
