/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.kns.datadictionary;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.krad.datadictionary.DataDictionaryDefinition;
import org.kuali.kfs.krad.datadictionary.DataDictionaryDefinitionBase;
import org.kuali.kfs.krad.datadictionary.HelpDefinition;
import org.kuali.kfs.krad.datadictionary.LookupAttributeDefinition;
import org.kuali.kfs.krad.datadictionary.LookupResultAttributeDefinition;
import org.kuali.kfs.krad.datadictionary.SortDefinition;
import org.kuali.kfs.krad.datadictionary.exception.DuplicateEntryException;
import org.kuali.kfs.krad.service.KRADServiceLocator;
import org.kuali.kfs.krad.service.LookupSearchService;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.rice.core.api.config.property.ConfigurationService;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Contains lookup-related information relating to the parent BusinessObject.
 * <p>
 * The lookup element is used to specify the rules for "looking up" a business object. These specifications define the
 * following:
 * <ul>
 * <li>How to specify the search criteria used to locate a set of business objects</li>
 * <li>How to display the search results</li>
 * </ul>
 * <p>
 * DD: See LookupDefinition.java
 * <p>
 * JSTL: The lookup element is a Map which is accessed using a key of "lookup".
 * <br>This map contains the following keys:
 * <ul>
 * <li>lookupableID (String, optional)</li>
 * <li>title (String)</li>
 * <li>menubar (String, optional)</li>
 * <li>defaultSort (Map, optional)</li>
 * <li>lookupFields (Map)</li>
 * <li>resultFields (Map)</li>
 * <li>resultSetLimit (String, optional)</li></ul>
 * <p>
 * @see org.kuali.kfs.kns.datadictionary.exporter.LookupMapBuilder
 * <p>
 * Note: the setters do copious amounts of validation, to facilitate generating errors during the parsing process.
 */
@Deprecated
public class LookupDefinition extends DataDictionaryDefinitionBase {
    private static final long serialVersionUID = 6733008572890721359L;

    protected String lookupableID;
    protected String title;
    protected String menubar;
    protected SortDefinition defaultSort;

    private LookupSearchService lookupSearchService;
    private List<LookupAttributeDefinition> lookupAttributeDefinitions = new LinkedList<>();
    private List<LookupResultAttributeDefinition> lookupResultAttributeDefinitions = new LinkedList<>();
    protected Map<String, LookupResultAttributeDefinition> lookupResultMap = new LinkedHashMap<>();
    protected Map<String, LookupAttributeDefinition> lookupAttributeMap = new LinkedHashMap<>();

    protected List<FieldDefinition> lookupFields = new ArrayList<FieldDefinition>();
    protected Map<String, FieldDefinition> lookupFieldMap = new LinkedHashMap<String, FieldDefinition>();
    protected List<FieldDefinition> resultFields = new ArrayList<FieldDefinition>();
    protected Map<String, FieldDefinition> resultFieldMap = new LinkedHashMap<String, FieldDefinition>();
    protected Integer resultSetLimit = null;
    protected Integer multipleValuesResultSetLimit = null;
    protected String extraButtonSource;
    protected String extraButtonParams;
    protected String searchIconOverride;

    protected int numOfColumns = 2;

    protected HelpDefinition helpDefinition;
    protected String helpUrl;
    protected boolean translateCodes = false;
    protected boolean disableSearchButtons = false;

    public LookupDefinition() {
    }

    /**
     * The lookupableID element identifies the name of the Spring bean which will be used to obtain the lookupable
     * helper service for the business object.
     * For example, the Balance.xml file has a lookupableId = "glBalanceLookupable".
     * The KualiSpringBeansGL.xml file determines that the helper service will be an instance of
     * BalanceLookupableHelperServiceImpl.
     * <p>
     * If this field is omitted, the default bean id used will be kualiLookupable which uses the
     * KualiLookupableHelperServiceImpl helper service.
     */
    public void setLookupableID(String lookupableID) {
        if (lookupableID == null) {
            throw new IllegalArgumentException("invalid (null) lookupableID");
        }
        this.lookupableID = lookupableID;
    }

    /**
     * @return custom lookupable id.
     */
    public String getLookupableID() {
        return this.lookupableID;
    }

    /**
     * @return title.
     */
    public String getTitle() {
        return title;
    }

    /**
     * @param title value to set for title.
     * @throws IllegalArgumentException if the given title is blank.
     */
    public void setTitle(String title) {
        if (StringUtils.isBlank(title)) {
            throw new IllegalArgumentException("invalid (blank) title");
        }
        this.title = title;
    }

    /**
     * @return true if this instance has a menubar.
     */
    public boolean hasMenubar() {
        return (menubar != null);
    }

    /**
     * @return menubar
     */
    public String getMenubar() {
        return menubar;
    }

    /**
     * The menubar element is used to add additional html code to the header line on the lookup screen.
     * <p>
     * For example, Account.xml uses this element to add the "create new global" button to the Account Lookup header.
     *
     * @throws IllegalArgumentException if the given menubar is blank.
     */
    public void setMenubar(String menubar) {
        if (StringUtils.isBlank(menubar)) {
            throw new IllegalArgumentException("invalid (blank) menubar");
        }
        // TODO: catch exception if service locator call fails
        ConfigurationService kualiConfigurationservice = KRADServiceLocator.getKualiConfigurationService();
        this.menubar = menubar.replace("${kr.externalizable.images.url}",
                kualiConfigurationservice.getPropertyValueAsString(KRADConstants.EXTERNALIZABLE_IMAGES_URL_KEY))
                .replace("${externalizable.images.url}",
                        kualiConfigurationservice.getPropertyValueAsString(
                                KRADConstants.APPLICATION_EXTERNALIZABLE_IMAGES_URL_KEY));
        this.menubar = this.menubar.replace("${application.url}", kualiConfigurationservice.getPropertyValueAsString(
            KRADConstants.APPLICATION_URL_KEY));
    }

    /**
     * @return true if this instance has a default sort defined.
     */
    public boolean hasDefaultSort() {
        return (defaultSort != null);
    }

    /**
     * @return defaultSort.
     */
    public SortDefinition getDefaultSort() {
        return defaultSort;
    }

    /**
     * The defaultSort element specifies the sequence in which the lookup search results should be displayed. It
     * contains an ascending/descending indicator and a list of attribute names.
     * <p>
     * DD: See SortDefinition.java
     * <p>
     * JSTL: defaultSort is a Map with the following keys:
     * <ul>
     * <li>sortAscending (boolean String)</li>
     * <li>sortAttributes (Map)</li>
     * </ul>
     * <p>
     * By the time JSTL export occurs, the optional attributeName from the defaultSort tag will have been converted
     * into the first contained sortAttribute.
     * <p>
     * @see org.kuali.kfs.kns.datadictionary.exporter.LookupMapBuilder
     *
     * @throws IllegalArgumentException if the given defaultSort is blank.
     */
    public void setDefaultSort(SortDefinition defaultSort) {
        if (defaultSort == null) {
            throw new IllegalArgumentException("invalid (null) defaultSort");
        }
        this.defaultSort = defaultSort;
    }

    /**
     * @return List of attributeNames of all lookupField FieldDefinitions associated with this LookupDefinition, in the
     *         order in which they were added.
     */
    public List getLookupFieldNames() {
        List fieldNames = new ArrayList();
        fieldNames.addAll(this.lookupFieldMap.keySet());
        return fieldNames;
    }

    /**
     * @return Collection of all lookupField FieldDefinitions associated with this LookupDefinition, in the order in
     *         which they were added.
     */
    public List<FieldDefinition> getLookupFields() {
        return lookupFields;
    }

    /**
     * @param attributeName
     * @return FieldDefinition associated with the named lookup field, or null if there is none
     */
    public FieldDefinition getLookupField(String attributeName) {
        return lookupFieldMap.get(attributeName);
    }

    /**
     * @return List of attributeNames of all resultField FieldDefinitions associated with this LookupDefinition, in the
     *         order in which they were added.
     */
    public List<String> getResultFieldNames() {
        List<String> fieldNames = new ArrayList<String>();
        fieldNames.addAll(resultFieldMap.keySet());
        return fieldNames;
    }

    /**
     * @return Collection of all resultField FieldDefinitions associated with this LookupDefinition, in the order in
     *         which they were added.
     */
    public List<FieldDefinition> getResultFields() {
        return resultFields;
    }

    /**
     * @param attributeName
     * @return FieldDefinition associated with the named result field, or null if there is none.
     */
    public FieldDefinition getResultField(String attributeName) {
        return resultFieldMap.get(attributeName);
    }

    /**
     * The resultSetLimit element specifies the maximum number of records that will be listed as a result of the lookup
     * search.
     */
    public void setResultSetLimit(Integer resultSetLimit) {
        this.resultSetLimit = resultSetLimit;
    }

    /**
     * @return true if this instance has a result set limit.
     */
    public boolean hasResultSetLimit() {
        return (resultSetLimit != null);
    }

    /**
     * The resultSetLimit element specifies the maximum number of records that will be listed as a result of the lookup
     * search.
     */
    public Integer getResultSetLimit() {
        return resultSetLimit;
    }

    /**
     * The multipleValuesResultSetLimit element specifies the maximum number of records that will be listed as a result
     * of a multiple values lookup search.
     */
    public void setMultipleValuesResultSetLimit(Integer multipleValuesResultSetLimit) {
        this.multipleValuesResultSetLimit = multipleValuesResultSetLimit;
    }

    /**
     * @return true if this instance has a multiple values result set limit.
     */
    public boolean hasMultipleValuesResultSetLimit() {
        return (multipleValuesResultSetLimit != null);
    }

    /**
     * The multipleValuesResultSetLimit element specifies the maximum number of records that will be listed as a result
     * of a multiple values lookup search.
     */
    public Integer getMultipleValuesResultSetLimit() {
        return multipleValuesResultSetLimit;
    }

    /**
     * Directly validate simple fields, call completeValidation on Definition fields.
     *
     * @see DataDictionaryDefinition#completeValidation(Class, Class)
     */
    public void completeValidation(Class rootBusinessObjectClass, Class otherBusinessObjectClass) {
        if (hasDefaultSort()) {
            defaultSort.completeValidation(rootBusinessObjectClass, null);
        }

        for (FieldDefinition lookupField : lookupFields) {
            lookupField.completeValidation(rootBusinessObjectClass, null);
        }

        for (FieldDefinition resultField : resultFields) {
            resultField.completeValidation(rootBusinessObjectClass, null);
        }
    }

    /**
     * @return true if this instance has extraButtonSource.
     */
    public boolean hasExtraButtonSource() {
        return extraButtonSource != null;
    }

    /**
     * @return extraButtonSource.
     */
    public String getExtraButtonSource() {
        return extraButtonSource;
    }

    /**
     * The extraButton element is used to define additional buttons which will appear on the lookup screen next to the
     * Search and Clear buttons. You can define the image source and additional html parameters for each button.
     * <p>
     * The extraButtonSource element defines the location of an image file to use for the extra button.
     *
     * @throws IllegalArgumentException if the given source is blank.
     */
    public void setExtraButtonSource(String extraButtonSource) {
        if (StringUtils.isBlank(extraButtonSource)) {
            throw new IllegalArgumentException("invalid (blank) button source");
        }
        this.extraButtonSource = extraButtonSource;
    }

    /**
     * @return true if this instance has extraButtonParams.
     */
    public boolean hasExtraButtonParams() {
        return extraButtonParams != null;
    }

    /**
     * @return extraButtonParams.
     */
    public String getExtraButtonParams() {
        return extraButtonParams;
    }

    /**
     * The extraButton element is used to define additional buttons which will appear on the lookup screen next to the
     * Search and Clear buttons. You can define the image source and additional html parameters for each button.
     * <p>
     * The extraButtonParams contains extra HTML parameters that be associated with the button.
     */
    public void setExtraButtonParams(String extraButtonParams) {
        this.extraButtonParams = extraButtonParams;
    }

    /**
     * @return true if this instance has an alternate icon to use for lookup icon.
     */
    public boolean hasSearchIconOverride() {
        return searchIconOverride != null;
    }

    /**
     * @return search icon override url.
     */
    public String getSearchIconOverride() {
        return searchIconOverride;
    }

    /**
     * The searchIconOverride element is used to define alternative icons appear on the lookup screen next to the Search
     * and Clear buttons. You can define the image source.
     *
     * @throws IllegalArgumentException if the given source is blank.
     */
    public void setSearchIconOverride(String searchIconOverride) {
        if (StringUtils.isBlank(searchIconOverride)) {
            throw new IllegalArgumentException("invalid (blank) search icon override");
        }
        this.searchIconOverride = searchIconOverride;
    }

    public String toString() {
        return "LookupDefinition '" + getTitle() + "'";
    }

    /**
     * The lookupFields element defines the set of fields in which the user can enter values representing search
     * selection criteria.  A search result record will be returned only if the criteria entered in all the lookup
     * fields are met.
     * <p>
     * DD:  See LookupDefinition.java
     * <p>
     * JSTL: lookupFields is a Map which is accessed using a key of "lookupFields".
     * This map contains the following keys:
     * attributeName of first lookup field
     * attributeName of second lookup field
     * etc.
     * The corresponding values are lookupField Export Maps.
     * @see org.kuali.kfs.kns.datadictionary.exporter.LookupMapBuilder
     * <p>
     * The lookupField element defines one lookup search criterion field.
     * DD: See LookupDefinition.java.
     * <p>
     * JSTL: lookupField is a Map which is accessed by a key which is the attributeName of a lookup field. This map
     * contains entries with the following keys:
     * "attributeName" (String)
     * "required" (boolean String)
     * <p>
     * lookupField attribute definitions:
     * <p>
     * required = true means that the user must enter something
     * into the search criterion lookup field
     * forceLookup = this attribute is not used
     * noLookup = true means that field should not include magnifying glass (i.e. quickfinder)
     */
    public void setLookupFields(List<FieldDefinition> lookupFields) {
        lookupFieldMap.clear();
        for (FieldDefinition lookupField : lookupFields) {
            if (lookupField == null) {
                throw new IllegalArgumentException("invalid (null) lookupField");
            }
            String keyName = lookupField.getAttributeName();
            if (lookupFieldMap.containsKey(keyName)) {
                throw new DuplicateEntryException("duplicate lookupField entry for attribute '" + keyName + "'");
            }

            lookupFieldMap.put(keyName, lookupField);
        }
        this.lookupFields = lookupFields;
    }

    /**
     * The resultFields element specifies the list of fields that are shown as a result of the lookup search.
     * <p>
     * JSTL: resultFields is a Map which is accesseed by a key of "resultFields". This map contains entries with the
     * following keys:
     * attributeName of first result field
     * attributeName of second result field
     * etc.
     * The corresponding values are ExportMap's
     * <p>
     * The ExportMaps are accessed using a key of attributeName.
     * Each ExportMap contains a single entry as follows:
     * "attributeName"
     * The corresponding value is the attributeName of the field.
     * <p>
     * @see org.kuali.kfs.kns.datadictionary.exporter.LookupMapBuilder
     */
    public void setResultFields(List<FieldDefinition> resultFields) {
        resultFieldMap.clear();
        for (FieldDefinition resultField : resultFields) {
            if (resultField == null) {
                throw new IllegalArgumentException("invalid (null) resultField");
            }

            String keyName = resultField.getAttributeName();
            if (resultFieldMap.containsKey(keyName)) {
                throw new DuplicateEntryException("duplicate resultField entry for attribute '" + keyName + "'");
            }

            resultFieldMap.put(keyName, resultField);
        }
        this.resultFields = resultFields;
    }

    /**
     * @return the numOfColumns.
     */
    public int getNumOfColumns() {
        return this.numOfColumns;
    }

    /**
     * @param numOfColumns the numOfColumns to set.
     */
    public void setNumOfColumns(int numOfColumns) {
        this.numOfColumns = numOfColumns;
    }

    /**
     * @return the helpDefinition.
     */
    public HelpDefinition getHelpDefinition() {
        return this.helpDefinition;
    }

    /**
     * @param helpDefinition the helpDefinition to set.
     */
    public void setHelpDefinition(HelpDefinition helpDefinition) {
        this.helpDefinition = helpDefinition;
    }

    /**
     * @return the helpUrl.
     */
    public String getHelpUrl() {
        return this.helpUrl;
    }

    /**
     * @param helpUrl the helpUrl to set.
     */
    public void setHelpUrl(String helpUrl) {
        this.helpUrl = helpUrl;
    }

    public boolean isTranslateCodes() {
        return this.translateCodes;
    }

    public void setTranslateCodes(boolean translateCodes) {
        this.translateCodes = translateCodes;
    }

    public boolean isDisableSearchButtons() {
        return this.disableSearchButtons;
    }

    public void setDisableSearchButtons(boolean disableSearchButtons) {
        this.disableSearchButtons = disableSearchButtons;
    }

    /**
     * @return {@link LookupSearchService} to use for the associated business object.
     */
    public LookupSearchService getLookupSearchService() {
        return lookupSearchService;
    }

    /**
     * @param lookupSearchService implementation to use for the associated business object.
     */
    public void setLookupSearchService(LookupSearchService lookupSearchService) {
        this.lookupSearchService = lookupSearchService;
    }

    public List<LookupAttributeDefinition> getLookupAttributeDefinitions() {
        return lookupAttributeDefinitions;
    }

    public void setLookupAttributeDefinitions(
            List<LookupAttributeDefinition> lookupAttributeDefinitions) {
        this.lookupAttributeDefinitions = lookupAttributeDefinitions;
        this.lookupAttributeMap = lookupAttributeDefinitions.stream()
                .collect(Collectors.toMap(LookupAttributeDefinition::getName, Function.identity()));
    }

    public LookupAttributeDefinition getLookupAttributeDefinition(String definitionName) {
        return this.lookupAttributeMap.get(definitionName);
    }

    public List<LookupResultAttributeDefinition> getLookupResultAttributeDefinitions() {
        return lookupResultAttributeDefinitions;
    }

    public LookupResultAttributeDefinition getLookupResultAttributeDefinition(String definitionName) {
        return this.lookupResultMap.get(definitionName);
    }

    public void setLookupResultAttributeDefinitions(List<LookupResultAttributeDefinition> lookupResultDefinitions) {
        this.lookupResultAttributeDefinitions= lookupResultDefinitions;
        this.lookupResultMap = this.lookupResultAttributeDefinitions.stream()
                .collect(Collectors.toMap(LookupResultAttributeDefinition::getName, Function.identity()));

    }
}
