/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.kns.datadictionary.control;

import org.apache.commons.lang3.ClassUtils;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.kuali.kfs.krad.datadictionary.DataDictionaryDefinitionBase;
import org.kuali.kfs.krad.datadictionary.control.ControlDefinition;
import org.kuali.kfs.krad.datadictionary.exception.ClassValidationException;
import org.kuali.kfs.krad.datadictionary.exception.CompletionException;
import org.kuali.kfs.krad.keyvalues.KeyValuesFinder;
import org.kuali.rice.core.api.util.ClassLoaderUtils;
import org.kuali.rice.krad.bo.BusinessObject;

import java.util.Objects;

/**
 * A single HTML control definition in the DataDictionary, which contains information relating to the HTML control
 * used to realize a specific attribute. All types of controls are represented by an instance of this class; you have
 * to call one of the is* methods to figure out which of the other accessors should return useful values.
 */
@Deprecated
public abstract class ControlDefinitionBase extends DataDictionaryDefinitionBase implements ControlDefinition {
    private static final long serialVersionUID = 4372435175782501152L;

    protected boolean datePicker;
    protected boolean expandedTextArea;
    protected String script;
    protected KeyValuesFinder valuesFinder;
    protected String businessObjectClass;
    protected String keyAttribute;
    protected String labelAttribute;
    protected Boolean includeBlankRow;
    protected Boolean includeKeyInLabel;
    protected Integer size;
    protected Integer rows;
    protected Integer cols;
    protected boolean ranged;

    public ControlDefinitionBase() {
        ranged = true;
    }

    public boolean isDatePicker() {
        return datePicker;
    }

    /**
     * Whether this control should have a date picker button next to the field. Valid for text fields.
     */
    public void setDatePicker(boolean datePicker) {
        this.datePicker = datePicker;
    }

    public boolean isExpandedTextArea() {
        return expandedTextArea;
    }

    /**
     * Whether this control should have a expanded text area button next to the field. Valid for textarea fields.
     */
    public void setExpandedTextArea(boolean eTextArea) {
        this.expandedTextArea = eTextArea;
    }

    public boolean isCheckbox() {
        return false;
    }

    public boolean isHidden() {
        return false;
    }

    public boolean isRadio() {
        return false;
    }

    public boolean isSelect() {
        return false;
    }

    public boolean isMultiselect() {
        return false;
    }

    public boolean isText() {
        return false;
    }

    public boolean isTextarea() {
        return false;
    }

    public boolean isCurrency() {
        return false;
    }

    public boolean isKualiUser() {
        return false;
    }

    public boolean isWorkflowWorkgroup() {
        return false;
    }

    public boolean isFile() {
        return false;
    }

    public boolean isLookupHidden() {
        return false;
    }

    public boolean isLookupReadonly() {
        return false;
    }

    public boolean isButton() {
        return false;
    }

    public boolean isLink() {
        return false;
    }

    public boolean isTitleLinkedText() {
        return false;
    }

    public void setValuesFinder(KeyValuesFinder valuesFinder) {
        if (valuesFinder == null) {
            throw new IllegalArgumentException("invalid (null) valuesFinder");
        }

        this.valuesFinder = valuesFinder;
    }

    /**
     * @return the dataObjectClass
     */
    public String getBusinessObjectClass() {
        return this.businessObjectClass;
    }

    /**
     * Used by a PersistableBusinessObjectValuesFinder to automatically query and display a list
     * of business objects as part of a select list or set of radio buttons.
     * <p>
     * The keyAttribute, labelAttribute, and includeKeyInLabel are used with this property.
     *
     * @param businessObjectClass the dataObjectClass to set
     */
    public void setBusinessObjectClass(String businessObjectClass) {
        if (businessObjectClass == null) {
            throw new IllegalArgumentException("invalid (null) dataObjectClass");
        }

        this.businessObjectClass = businessObjectClass;
    }

    /**
     * @return the includeBlankRow
     */
    public Boolean getIncludeBlankRow() {
        return this.includeBlankRow;
    }

    /**
     * @return the includeBlankRow
     */
    public void setIncludeBlankRow(Boolean includeBlankRow) {
        this.includeBlankRow = includeBlankRow;
    }

    /**
     * @return the includeKeyInLabel
     */
    public Boolean getIncludeKeyInLabel() {
        return this.includeKeyInLabel;
    }

    /**
     * Whether to include the key in the label for select lists and radio buttons.
     */
    public void setIncludeKeyInLabel(Boolean includeKeyInLabel) {
        this.includeKeyInLabel = includeKeyInLabel;
    }

    /**
     * @return the keyAttribute
     */
    public String getKeyAttribute() {
        return this.keyAttribute;
    }

    /**
     * Attribute of the given dataObjectClass to use as the value of a select list
     * or set of radio buttons.
     */
    public void setKeyAttribute(String keyAttribute) {
        this.keyAttribute = keyAttribute;
    }

    /**
     * @return the labelAttribute
     */
    public String getLabelAttribute() {
        return this.labelAttribute;
    }

    /**
     * Attribute of the given dataObjectClass to use as the displayed label on a select list
     * or set of radio buttons.
     */
    public void setLabelAttribute(String labelAttribute) {
        this.labelAttribute = labelAttribute;
    }

    public KeyValuesFinder getValuesFinder() {
        return valuesFinder;
    }

    /**
     * Size of a text control.
     */
    public void setSize(Integer size) {
        this.size = size;
    }

    public Integer getSize() {
        return size;
    }

    public boolean hasScript() {
        return false;
    }

    /**
     * Number of rows to display on a text-area widget.
     */
    public void setRows(Integer rows) {
        this.rows = rows;
    }

    public Integer getRows() {
        return rows;
    }

    /**
     * Number of columns to display on a text-area widget.
     */
    public void setCols(Integer cols) {
        this.cols = cols;
    }

    public Integer getCols() {
        return cols;
    }

    /**
     * Directly validate simple fields.
     */
    public void completeValidation(Class rootBusinessObjectClass, Class otherBusinessObjectClass) {
        if (!isCheckbox() && !isHidden() && !isRadio() && !isSelect() && !isMultiselect() && !isText() && !isTextarea()
                && !isCurrency() && !isKualiUser() && !isLookupHidden() && !isLookupReadonly() && !isWorkflowWorkgroup()
                && !isFile() && !isButton() && !isLink()) {
            throw new CompletionException("error validating " + rootBusinessObjectClass
                    .getName() + " control: unknown control type in control definition (" + "" + ")");
        }
        if (businessObjectClass != null) {
            try {
                Class businessObjectClassObject = ClassUtils.getClass(ClassLoaderUtils.getDefaultClassLoader(),
                        getBusinessObjectClass());
                if (!BusinessObject.class.isAssignableFrom(businessObjectClassObject)) {
                    throw new ClassValidationException(
                            "dataObjectClass is not a valid instance of " + BusinessObject.class
                                    .getName() + " instead was: " + businessObjectClassObject.getName());
                }
            } catch (ClassNotFoundException e) {
                throw new ClassValidationException("dataObjectClass could not be found: " + getBusinessObjectClass(),
                        e);
            }
        }
    }

    public String getScript() {
        return script;
    }

    /**
     * JavaScript script to run when a select control's value is changed.
     */
    public void setScript(String script) {
        this.script = script;
    }

    public boolean isRanged() {
        return this.ranged;
    }

    /**
     * Sets the control as a ranged (from and to) date field if true, or a single date field if false
     *
     * @param ranged boolean true for a ranged control, false for a single date field
     */
    public void setRanged(boolean ranged) {
        this.ranged = ranged;
    }

    @Override
    public boolean equals(Object object) {
        if (!(object instanceof ControlDefinitionBase)) {
            return false;
        }
        ControlDefinitionBase rhs = (ControlDefinitionBase) object;
        return new EqualsBuilder()
            .append(this.cols, rhs.cols)
            .append(this.businessObjectClass, rhs.businessObjectClass)
            .append(this.valuesFinder, rhs.valuesFinder)
            .append(this.rows, rhs.rows)
            .append(this.script, rhs.script)
            .append(this.size, rhs.size)
            .append(this.datePicker, rhs.datePicker)
            .append(this.ranged, rhs.ranged)
            .append(this.labelAttribute, rhs.labelAttribute)
            .append(this.includeKeyInLabel, rhs.includeKeyInLabel)
            .append(this.keyAttribute, rhs.keyAttribute)
            .isEquals();
    }

    @Override
    public int hashCode() {
        return Objects.hash(cols, businessObjectClass, valuesFinder, rows, script, size, datePicker, ranged,
                labelAttribute, includeKeyInLabel, keyAttribute);
    }
}
