/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.kns.document.authorization;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.kns.web.ui.Field;
import org.kuali.kfs.krad.datadictionary.mask.MaskFormatter;

/**
 * This class represents the authorization restrictions (or lack of) for a given field.
 */

public class FieldRestriction {

    private String fieldName;
    private boolean editable;
    private boolean viewable;
    private boolean masked;
    private boolean partiallyMasked;
    private MaskFormatter maskFormatter;
    private boolean shouldBeEncrypted;

    public FieldRestriction() {
        editable = true;
        viewable = true;
    }

    /**
     * @param fieldName name of field to represent
     * @param canEdit true if the field is editable in this context, false otherwise
     * @param canView true if the field is viewable in this context, false otherwise
     *
     */
    public FieldRestriction(String fieldName, boolean canEdit, boolean canView) {
        this.fieldName = fieldName;
        setEditable(canEdit); // using setters here to run impossible combinations check
        setViewable(canView);
    }

    /**
     * @param fieldName name of the field to represent
     * @param fieldAuthorizationFlag Field.HIDDEN, Field.READONLY, or Field.EDITABLE
     */
    public FieldRestriction(String fieldName, String fieldAuthorizationFlag) {
        // if an invalid flag is passed in, the choke on it
        if (!fieldAuthorizationFlag.equals(Field.EDITABLE) && !fieldAuthorizationFlag.equals(Field.READONLY)
            && !fieldAuthorizationFlag.equals(Field.HIDDEN) && !fieldAuthorizationFlag.equals(Field.MASKED)
            && !fieldAuthorizationFlag.equals(Field.PARTIALLY_MASKED)) {
            throw new IllegalArgumentException("The only allowable values are " +
                "Field.HIDDEN, Field.READONLY, Field.EDITABLE, Field.MASKED and Field.PARTIALLY_MASKED");
        }

        this.fieldName = fieldName;

        if (fieldAuthorizationFlag.equals(Field.EDITABLE)) {
            this.editable = true;
            this.viewable = true;
        } else if (fieldAuthorizationFlag.equals(Field.READONLY)) {
            this.editable = false;
            this.viewable = true;
        } else if (fieldAuthorizationFlag.equals(Field.HIDDEN)) {
            this.editable = false;
            this.viewable = false;
        } else if (fieldAuthorizationFlag.equals(Field.MASKED)) {
            this.masked = true;
            this.viewable = true;
            this.editable = false;
        } else if (fieldAuthorizationFlag.equals(Field.PARTIALLY_MASKED)) {
            this.partiallyMasked = true;
            this.viewable = true;
            this.editable = false;
        }
    }

    /**
     * This method returns the correct flag from the Kuali Field object, that corresponds to the particular
     * combination of editable and viewable set on this object.
     *
     * @return Field.HIDDEN, Field.READONLY, or Field.EDITABLE
     */
    public String getKualiFieldDisplayFlag() {
        if (!editable && !viewable) {
            return Field.HIDDEN;
        }
        if (!editable && viewable) {
            return Field.READONLY;
        } else {
            return Field.EDITABLE;
        }
    }

    /**
     * @return {@code true} if the FieldAuthorization is some kind of restriction, and {@code false} if it is an
     *         editable field.
     */
    public boolean isRestricted() {
        if (!editable || !viewable) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * @return {@code true} if this authorization prohibits Viewing and Editing, resulting in a hidden field.
     */
    public boolean isHidden() {
        if (!editable && !viewable) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * @return {@code true} if this authorization prohibits Editing but not Viewing, resulting in a ReadOnly field.
     */
    public boolean isReadOnly() {
        if (!editable && viewable) {
            return true;
        } else {
            return false;
        }
    }

    public boolean isEditable() {
        return editable;
    }

    /**
     * Sets the editable attribute value.
     *
     * Note that if editable is being set to true, and the internal value of viewable is false, viewable will be
     * flipped to true, to avoid impossible combinations of flags.
     *
     * @param editable The editable to set.
     */
    public void setEditable(boolean editable) {
        if (editable && !this.viewable) {
            this.viewable = true;
        }
        this.editable = editable;
    }

    public String getFieldName() {
        return fieldName;
    }

    public void setFieldName(String fieldName) {
        this.fieldName = fieldName;
    }

    public boolean isViewable() {
        return viewable;
    }

    /**
     * Sets the viewable attribute value.
     *
     * Note that if viewable is being set to false, and the internal value of editable is true, then editable will be
     * silently flipped to false. This is done to avoid impossible combinations of authorization flags.
     *
     * @param viewable The viewable to set.
     */
    public void setViewable(boolean viewable) {
        if (!viewable && this.editable) {
            this.editable = false;
        }
        this.viewable = viewable;
    }

    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append(this.fieldName);
        sb.append(" [");
        if (this.editable) {
            sb.append("editable");
        } else {
            sb.append("not editable");
        }
        sb.append(",");
        if (this.viewable) {
            sb.append("viewable");
        } else {
            sb.append("not viewable");
        }
        sb.append("]");
        return sb.toString();
    }

    public boolean equals(Object obj) {
        boolean equal = false;

        if (obj != null) {
            if (this.getClass().equals(obj.getClass())) {
                FieldRestriction other = (FieldRestriction) obj;

                if (StringUtils.equals(this.fieldName, other.getFieldName())) {
                    if (this.editable == other.isEditable() && this.viewable == other.isViewable()) {
                        equal = true;
                    }
                }
            }
        }

        return equal;
    }

    public int hashCode() {
        return toString().hashCode();
    }

    public boolean isMasked() {
        return this.masked;
    }

    public boolean isPartiallyMasked() {
        return this.partiallyMasked;
    }

    /**
     * @return {@code true} if this instance indicates full or partial masking is needed; {@code false} otherwise.
     */
    public boolean hasAnyMasking() {
        return this.partiallyMasked || this.masked;
    }

    public boolean isShouldBeEncrypted() {
        return this.shouldBeEncrypted;
    }

    public void setShouldBeEncrypted(boolean shouldBeEncrypted) {
        this.shouldBeEncrypted = shouldBeEncrypted;
    }

    public MaskFormatter getMaskFormatter() {
        return this.maskFormatter;
    }

    public void setMaskFormatter(MaskFormatter maskFormatter) {
        this.maskFormatter = maskFormatter;
    }

}
