/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.uif.component;

import org.kuali.kfs.krad.uif.container.Container;
import org.kuali.kfs.krad.uif.field.Field;
import org.kuali.kfs.krad.uif.modifier.ComponentModifier;
import org.kuali.kfs.krad.uif.service.ViewHelperService;
import org.kuali.kfs.krad.uif.util.ComponentFactory;
import org.kuali.kfs.krad.uif.view.View;
import org.kuali.kfs.krad.uif.widget.Widget;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

/**
 * All classes of the UIF that are used as a rendering element implement the component interface. This interface defines
 * basic properties and methods that all such classes much implement. All components within the framework have the
 * following structure:
 * <ul>
 * <li>Dictionary Configuration/Composition</li>
 * <li>Java Class (the Component implementation</li>
 * <li>>JSP Template Renderer</li>
 * </ul>
 * <p>
 * There are three basic types of components:
 * <ul>
 * <li>Container Components: <code>View</code>, <code>Group</code></li>
 * <li>Field Components: <code>Field</code></li>
 * <li>Widget Components: <code>Widget</code></li>
 * </ul>
 *
 * @see Container
 * @see Field
 * @see Widget
 */
public interface Component extends Configurable, Serializable, Ordered, ScriptEventSupport {

    /**
     * The unique id (within a given tree) for the component
     * <p>
     * <p>
     * The id will be used by renderers to set the HTML element id. This gives a way to find various elements for
     * scripting. If the id is not given, a default will be generated by the framework
     *
     * @return String id
     */
    String getId();

    /**
     * @param id the unique id (within a given tree) for the component to set as the component id
     */
    void setId(String id);

    /**
     * Holds the id for the component that can be used to request new instances of that component from the
     * {@link ComponentFactory}
     * <p>
     * <p>
     * During component refreshes the component is reinitialized and the lifecycle is performed again to
     * reflect the component state based on the latest updates (data, other component state). Since the lifecycle
     * is only performed on the component, a new instance with configured initial state needs to be retrieved. Some
     * component instances, such as those that are nested or created in code, cannot be obtained from the spring
     * factory. For those the initial state is captured during the perform initialize phase and the factory id
     * generated for referencing retrieving that configuration during a refresh
     *
     * @return String bean id for component
     */
    String getFactoryId();

    /**
     * Sets the factory id that backs the component instance
     *
     * @param factoryId
     */
    void setFactoryId(String factoryId);

    /**
     * This is used within the rendering layer to pass the component instance into the template. The component instance
     * is exported under the name given by this method.
     *
     * @return name for the component type
     */
    String getComponentTypeName();

    /**
     * The path should be relative to the web root. An attribute will be available to the component to use under the
     * name given by the method <code>getComponentTypeName</code>. Based on the component type, additional attributes
     * could be available for use. See the component documentation for more information on such attributes.
     * <p>
     * <p>
     * e.g. '/krad/WEB-INF/jsp/tiles/component.jsp'
     *
     * @return String representing the template path to the JSP file that should be called to render the component
     */
    String getTemplate();

    /**
     * @param template components template to set.
     */
    void setTemplate(String template);

    /**
     * @return A title for the component. Depending on the component can be used in various ways. For example with a
     *         Container component the title is used to set the header text. For components like controls other other
     *         components that render an HTML element it is used to set the HTML title attribute
     */
    String getTitle();

    /**
     * @param title the components title to set.
     */
    void setTitle(String title);

    /**
     * Should be called to initialize the component
     * <p>
     * <p>
     * Where components can set defaults and setup other necessary state. The initialize method should only be called
     * once per component lifecycle and is invoked within the initialize phase of the view lifecycle.
     *
     * @param view  view instance in which the component belongs
     * @param model object instance containing the view data
     * @see ViewHelperService#performInitialization(View, Object)
     */
    void performInitialization(View view, Object model);

    /**
     * Called after the initialize phase to perform conditional logic based on the model data
     * <p>
     * <p>
     * Where components can perform conditional logic such as dynamically generating new fields or setting field state
     * based on the given data
     *
     * @param view  view instance to which the component belongs
     * @param model Top level object containing the data (could be the form or a top level business object, dto)
     */
    void performApplyModel(View view, Object model, Component parent);

    /**
     * The last phase before the view is rendered. Here final preparations can be made based on the updated view state
     *
     * @param view   view instance that should be finalized for rendering
     * @param model  top level object containing the data
     * @param parent parent component
     */
    void performFinalize(View view, Object model, Component parent);

    /**
     * Used by <code>ViewHelperService</code> for the various lifecycle callbacks
     *
     * @return List<Component> child components that are contained within the component and should be sent through the
     *         lifecycle
     */
    List<Component> getComponentsForLifecycle();

    /**
     * Prototypes are held for configuring how a component should be created during the lifecycle. An example of this
     * are the fields in a collection group that are created for each collection record. They only participate in the
     * initialize phase.
     *
     * @return List<Component> child component prototypes that are maintained for creating other component instances
     */
    List<Component> getComponentPrototypes();

    /**
     * Used to get all the nested components in the property replacer's
     *
     * @return  List of components that are contained within the List of <code>PropertyReplacer</code> in component
     */
    List<Component> getPropertyReplacerComponents();

    /**
     * <code>ComponentModifier</code> instances that should be invoked to initialize the component
     * <p>
     * <p>
     * These provide dynamic initialization behavior for the component and are configured through the components
     * definition. Each initializer will get invoked by the initialize method.
     *
     * @return List of component modifiers
     * @see ViewHelperService#performInitialization(View, Object)
     */
    List<ComponentModifier> getComponentModifiers();

    /**
     * @param componentModifiers for the components List of <code>ComponentModifier</code> instances
     */
    void setComponentModifiers(List<ComponentModifier> componentModifiers);

    /**
     * If set to false, the corresponding component template will not be invoked (therefore nothing will be rendered to
     * the UI).
     *
     * @return boolean true if the component should be rendered, false if it should not be
     */
    boolean isRender();

    /**
     * @param render the components render indicator to set.
     */
    void setRender(boolean render);

    /**
     * How the hidden data is maintained depends on the views persistence mode. If the mode is request, the
     * corresponding data will be rendered to the UI but not visible. If the mode is session, the data will not be
     * rendered to the UI but maintained server side.
     * <p>
     * <p>
     * For a <code>Container</code> component, the hidden setting will apply to all contained components (making a
     * section hidden makes all fields within the section hidden)
     *
     * @return boolean true if the component should be hidden in the UI, false if it should be visible
     */
    boolean isHidden();

    /**
     * @param hidden the hidden indicator to set.
     */
    void setHidden(boolean hidden);

    /**
     * When readOnly the controls and widgets of <code>Field</code> components will not be rendered. If the Field has an
     * underlying value it will be displayed readOnly to the user.
     * <p>
     * <p>
     * For a <code>Container</code> component, the readOnly setting will apply to all contained components (making a
     * section readOnly makes all fields within the section readOnly)
     *
     * @return boolean true if the component should be readOnly, false if is allows editing
     */
    boolean isReadOnly();

    /**
     * @param readOnly the read only indicator to set.
     */
    void setReadOnly(boolean readOnly);

    /**
     * At the general component level required means there is some action the user needs to take within the component.
     * For example, within a section it might mean the fields within the section should be completed. At a field level,
     * it means the field should be completed. This provides the ability for the renderers to indicate the required
     * action.
     *
     * @return boolean true if the component is required, false if it is not required
     */
    Boolean getRequired();

    /**
     * @param required the required indicator to set.
     */
    void setRequired(Boolean required);

    /**
     * Any style override or additions can be specified with this attribute. This is used by the renderer to set the
     * style attribute on the corresponding element.
     * <p>
     * <p>
     * e.g. 'color: #000000;text-decoration: underline;'
     *
     * @return css style string to be applied to the component
     */
    String getStyle();

    /**
     * @param style the components style to set.
     */
    void setStyle(String style);

    /**
     * Declares style classes for the component. Multiple classes are specified with a space delimiter. This is used by
     * the renderer to set the class attribute on the corresponding element. The class(s) declared must be available in
     * the common style sheets or the style sheets specified for the view
     * <p>
     * <p>
     * e.g. 'header left'
     *
     * @return List<String> css style classes to be applied to the component
     */
    List<String> getStyleClasses();

    /**
     * @param styleClasses the style classes to set
     */
    void setStyleClasses(List<String> styleClasses);

    /**
     * Adds a single style to the list of styles on this component
     *
     * @param styleClass style to be added
     */
    void addStyleClass(String styleClass);

    /**
     * @param itemStyle
     */
    void appendToStyle(String itemStyle);

    /**
     * All components belong to a <code>Container</code> and are placed using a <code>LayoutManager</code>. This
     * property specifies how many places horizontally the component should take up within the container. This is
     * only applicable for table based layout managers. Default is 1
     * <p>
     * TODO: this should not be on component interface since it only applies if the layout manager supports it, need
     * some sort of layoutOptions map for field level options that depend on the manager
     *
     * @return int number of columns the component should span horizontally in the container
     */
    int getColSpan();

    /**
     * @param colSpan column span to set.
     */
    void setColSpan(int colSpan);

    /**
     * All components belong to a <code>Container</code> and are placed using a <code>LayoutManager</code>. This property
     * specifies how many places vertically the component should take up within the container. This is only applicable
     * for table based layout managers. Default is 1
     * <p>
     * TODO: this should not be on component interface since it only applies if the layout manager supports it, need
     * some sort of layoutOptions map for field level options that depend on the manager
     *
     * @return int number of rows the component should span vertically in the container
     */
    int getRowSpan();

    /**
     * @param rowSpan the component row span to set.
     */
    void setRowSpan(int rowSpan);

    /**
     * Any el statements configured for the components properties (e.g. title="@{foo.property}") are evaluated using
     * the el context map. This map will get populated with default objects like the model, view, and request from the
     * <code>ViewHelperService</code>. Other components can push further objects into the context so that they are
     * available for use with that component. For example, <code>Field</code> instances that are part of a collection
     * line as receive the current line instance
     * <p>
     * <p>
     * Context map also provides objects to methods that are invoked for <code>GeneratedField</code> instances
     * <p>
     * <p>
     * The Map key gives the name of the variable that can be used within expressions, and the Map value gives the object
     * instance for which expressions containing the variable should evaluate against
     * <p>
     * <p>
     * NOTE: Calling getContext().putAll() will skip updating any configured property replacers for the component.
     * Instead you should call #pushAllToContext
     *
     * @return Map<String, Object> context for the component
     */
    Map<String, Object> getContext();

    /**
     * @param context context Map to set.
     */
    void setContext(Map<String, Object> context);

    /**
     * Places the given object into the context Map for the component with the given name
     * <p>
     * <p>
     * Note this also will push context to property replacers configured on the component. To place multiple objects in
     * the context, you should use #pushAllToContext since that will call this method for each and update property
     * replacers. Using #getContext().putAll() will bypass property replacers.
     *
     * @param objectName name the object should be exposed under in the context map
     * @param object     object instance to place into context
     */
    void pushObjectToContext(String objectName, Object object);

    /**
     * Places each entry of the given Map into the context for the component
     * <p>
     * <p>
     * Note this will call #pushObjectToContext for each entry which will update any configured property
     * replacers as well. This should be used in place of getContext().putAll()
     *
     * @param objects Map<String, Object> objects to add to context, where the entry key will be the context key
     *                and the entry value will be the context value
     */
    void pushAllToContext(Map<String, Object> objects);

    /**
     * @return List<PropertyReplacer> replacers to evaluate during the view lifecycle to conditionally set properties on
     *         the <code>Component</code> based on expression evaluations
     */
    List<PropertyReplacer> getPropertyReplacers();

    /**
     * @param propertyReplacers property substitutions to set.
     */
    void setPropertyReplacers(List<PropertyReplacer> propertyReplacers);

    /**
     * @return Map<String, String> options that are passed through to the Component renderer. The Map key is the option
     *         name, with the Map value as the option value. See documentation on the particular widget render for
     *         available options.
     */
    Map<String, String> getComponentOptions();

    /**
     * @param componentOptions component options to set.
     */
    void setComponentOptions(Map<String, String> componentOptions);

    /**
     * @return Options that are passed through to the Component renderer. See documentation on the particular widget
     *         render for available options.
     */
    String getComponentOptionsJSString();

    /**
     * @param componentOptions component options to set.
     */
    void setComponentOptionsJSString(String componentOptions);

    /**
     * Can be used to order a component within a List of other components, lower numbers are placed higher up in the list,
     * while higher numbers are placed lower in the list
     *
     * @return int ordering number
     * @see org.springframework.core.Ordered#getOrder()
     */
    int getOrder();

    /**
     * @param order order to set.
     */
    void setOrder(int order);

    /**
     * Name of the method that should be invoked for finalizing the component configuration (full method name, without
     * parameters or return type)
     * <p>
     * <p>
     * Note the method can also be set with the finalizeMethodInvoker targetMethod property. If the method is on the
     * configured <code>ViewHelperService</code>, only this property needs to be configured
     * <p>
     * <p>
     * The model backing the view will be passed as the first argument method and then the <code>Component</code>
     * instance as the second argument. If any additional method arguments are declared with the
     * finalizeMethodAdditionalArguments, they will then be passed in the order declared in the list
     * <p>
     * <p>
     * If the component is selfRendered, the finalize method can return a string which will be set as the component's
     * renderOutput. The selfRendered indicator will also be set to true on the component.
     *
     * @return String method name
     */
    String getFinalizeMethodToCall();

    /**
     * These arguments are passed to the finalize method after the standard model and component arguments. They are
     * passed in the order declared in the list
     *
     * @return List<Object> additional method arguments that should be passed to the finalize method
     */
    List<Object> getFinalizeMethodAdditionalArguments();

    /**
     * If self rendered is true, the corresponding template for the component will not be invoked and the renderOutput
     * String will be written to the response as is.
     *
     * @return boolean true if component is self rendered (through the renderOutput property), false if not (renders
     *         through template)
     */
    boolean isSelfRendered();

    /**
     * @param selfRendered the self render indicator to set.
     */
    void setSelfRendered(boolean selfRendered);

    /**
     * @return Rendering output for the component that will be sent as part of the response (can contain static text and
     *         HTML)
     */
    String getRenderOutput();

    /**
     * @param renderOutput the component's render output to set.
     */
    void setRenderOutput(String renderOutput);

    /**
     * By default the framework nulls out any components that do not have a refresh condition or are needed for
     * collection processing. This can be a problem if custom application code is written to refresh a component
     * without setting the corresponding component flag. In this case this property can be set to true to force the
     * framework to keep the component in session. Defaults to false
     *
     * @return boolean true if the component should be stored in session view regardless of configuration, false if not
     */
    boolean isPersistInSession();

    /**
     * @param persistInSession the indicator to force persistence of the component in session to set.
     */
    void setPersistInSession(boolean persistInSession);

    /**
     * @return ComponentSecurity instance that indicates what authorization (permissions) exist for the component
     */
    ComponentSecurity getComponentSecurity();

    /**
     * @param componentSecurity security object to set.
     */
    void setComponentSecurity(ComponentSecurity componentSecurity);

    /**
     * @return the progressiveRender
     */
    String getProgressiveRender();

    /**
     * @param progressiveRender the progressiveRender to set
     */
    void setProgressiveRender(String progressiveRender);

    /**
     * @return the conditionalRefresh
     */
    String getConditionalRefresh();

    /**
     * @param conditionalRefresh the conditionalRefresh to set
     */
    void setConditionalRefresh(String conditionalRefresh);

    /**
     * @return the progressiveDisclosureControlNames
     */
    List<String> getProgressiveDisclosureControlNames();

    /**
     * @return the progressiveDisclosureConditionJs
     */
    String getProgressiveDisclosureConditionJs();

    /**
     * @return the conditionalRefreshConditionJs
     */
    String getConditionalRefreshConditionJs();

    /**
     * @return the conditionalRefreshControlNames
     */
    List<String> getConditionalRefreshControlNames();

    /**
     * @return the progressiveRenderViaAJAX
     */
    boolean isProgressiveRenderViaAJAX();

    /**
     * @param progressiveRenderViaAJAX the progressiveRenderViaAJAX to set
     */
    void setProgressiveRenderViaAJAX(boolean progressiveRenderViaAJAX);

    /**
     * If true, when the progressiveRender condition is satisfied, the component will always be retrieved from the
     * server and shown(as opposed to being stored on the client, but hidden, after the first retrieval as is the case
     * with the progressiveRenderViaAJAX option). <b>By default, this is false, so components with progressive render
     * capabilities will always be already within the client html and toggled to be hidden or visible.</b>
     *
     * @return the progressiveRenderAndRefresh
     */
    boolean isProgressiveRenderAndRefresh();

    /**
     * @param progressiveRenderAndRefresh the progressiveRenderAndRefresh to set
     */
    void setProgressiveRenderAndRefresh(boolean progressiveRenderAndRefresh);

    /**
     * Specifies a property by name that when it value changes will automatically perform a refresh on this component.
     * This can be a comma separated list of multiple properties that require this component to be refreshed when any of
     * them change. <Br>DO NOT use with progressiveRender unless it is know that progressiveRender condition will always
     * be satisfied before one of these fields can be changed.
     *
     * @return the refreshWhenChanged
     */
    String getRefreshWhenChanged();

    /**
     * @param refreshWhenChanged the refreshWhenChanged to set
     */
    void setRefreshWhenChanged(String refreshWhenChanged);

    /**
     * This is set by the framework for configured ajax action buttons, should not be set in configuration
     *
     * @return boolean true if the component is refreshed by an action, false if not
     */
    boolean isRefreshedByAction();

    /**
     * This is set by the framework for configured ajax action buttons, should not be set in configuration
     *
     * @param refreshedByAction the refreshed by action indicator to set.
     */
    void setRefreshedByAction(boolean refreshedByAction);

    /**
     * Indicates whether data
     *
     * @return boolean true if data contained within the component should be reset (set to default) when the component is
     *         refreshed, false if data should remain as is
     */
    boolean isResetDataOnRefresh();

    /**
     * @param resetDataOnRefresh the reset data on refresh indicator to set
     */
    void setResetDataOnRefresh(boolean resetDataOnRefresh);

    /**
     * @return result of the conditionalRefresh expression, true if satisfied, otherwise false. Note: not currently used
     *         for any processing, required by the expression evaluator.
     */
    boolean isRefresh();

    /**
     * @param refresh the refresh to set
     */
    void setRefresh(boolean refresh);

    /**
     * Control names which will refresh this component when they are changed, added internally
     *
     * @return the refreshWhenChangedControlNames
     */
    List<String> getRefreshWhenChangedControlNames();

}
