/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.uif.container;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.krad.uif.UifConstants;
import org.kuali.kfs.krad.uif.UifParameters;
import org.kuali.kfs.krad.uif.component.BindingInfo;
import org.kuali.kfs.krad.uif.component.Component;
import org.kuali.kfs.krad.uif.component.ComponentBase;
import org.kuali.kfs.krad.uif.component.ComponentSecurity;
import org.kuali.kfs.krad.uif.component.DataBinding;
import org.kuali.kfs.krad.uif.field.ActionField;
import org.kuali.kfs.krad.uif.field.DataField;
import org.kuali.kfs.krad.uif.field.Field;
import org.kuali.kfs.krad.uif.field.LabelField;
import org.kuali.kfs.krad.uif.util.ComponentUtils;
import org.kuali.kfs.krad.uif.view.View;
import org.kuali.kfs.krad.uif.widget.QuickFinder;
import org.kuali.rice.core.api.exception.RiceRuntimeException;

import java.util.ArrayList;
import java.util.List;

/**
 * Group that holds a collection of objects and configuration for presenting the collection in the UI. Supports
 * functionality such as add line, line actions, and nested collections.
 * <p>
 * <p>
 * Note the standard header/footer can be used to give a header to the collection as a whole, or to provide actions that
 * apply to the entire collection.
 * <p>
 * <p>
 * For binding purposes the binding path of each row field is indexed. The name property inherited from
 * <code>ComponentBase</code> is used as the collection name. The collectionObjectClass property is used to lookup
 * attributes from the data dictionary.
 */
public class CollectionGroup extends Group implements DataBinding {

    private static final long serialVersionUID = -6496712566071542452L;

    private Class<?> collectionObjectClass;

    private String propertyName;
    private BindingInfo bindingInfo;

    private boolean renderAddLine;
    private String addLinePropertyName;
    private BindingInfo addLineBindingInfo;
    private LabelField addLineLabelField;
    private List<? extends Component> addLineFields;
    private List<ActionField> addLineActionFields;

    private boolean renderLineActions;
    private List<ActionField> actionFields;

    private boolean renderSelectField;
    private String selectPropertyName;

    private QuickFinder collectionLookup;

    private boolean showHideInactiveButton;
    private boolean showInactive;
    private CollectionFilter activeCollectionFilter;
    private List<CollectionFilter> filters;

    private List<CollectionGroup> subCollections;
    private String subCollectionSuffix;

    private CollectionGroupBuilder collectionGroupBuilder;

    public CollectionGroup() {
        renderAddLine = true;
        renderLineActions = true;
        showInactive = false;
        showHideInactiveButton = true;
        renderSelectField = false;

        filters = new ArrayList<>();
        actionFields = new ArrayList<>();
        addLineFields = new ArrayList<Field>();
        addLineActionFields = new ArrayList<>();
        subCollections = new ArrayList<>();
    }

    /**
     * The following actions are performed:
     * <ul>
     * <li>Set fieldBindModelPath to the collection model path (since the fields have to belong to the same model as t
     * he collection)</li>
     * <li>Set defaults for binding</li>
     * <li>Default add line field list to groups items list</li>
     * <li>Sets default active collection filter if not set</li>
     * <li>Sets the dictionary entry (if blank) on each of the items to the collection class</li>
     * </ul>
     *
     * @see ComponentBase#performInitialization(View, java.lang.Object)
     */
    @Override
    public void performInitialization(View view, Object model) {
        setFieldBindingObjectPath(getBindingInfo().getBindingObjectPath());

        super.performInitialization(view, model);

        if (bindingInfo != null) {
            bindingInfo.setDefaults(view, getPropertyName());
        }

        if (addLineBindingInfo != null) {
            // add line binds to model property
            if (StringUtils.isNotBlank(addLinePropertyName)) {
                addLineBindingInfo.setDefaults(view, getPropertyName());
                addLineBindingInfo.setBindingName(addLinePropertyName);
                if (StringUtils.isNotBlank(getFieldBindByNamePrefix())) {
                    addLineBindingInfo.setBindByNamePrefix(getFieldBindByNamePrefix());
                }
            }
        }

        for (Component item : getItems()) {
            if (item instanceof DataField) {
                DataField field = (DataField) item;

                if (StringUtils.isBlank(field.getDictionaryObjectEntry())) {
                    field.setDictionaryObjectEntry(collectionObjectClass.getName());
                }
            }
        }

        for (Component addLineField : addLineFields) {
            if (addLineField instanceof DataField) {
                DataField field = (DataField) addLineField;

                if (StringUtils.isBlank(field.getDictionaryObjectEntry())) {
                    field.setDictionaryObjectEntry(collectionObjectClass.getName());
                }
            }
        }

        if ((addLineFields == null) || addLineFields.isEmpty()) {
            addLineFields = getItems();
        }

        // if active collection filter not set use default
        if (this.activeCollectionFilter == null) {
            activeCollectionFilter = new ActiveCollectionFilter();
        }

        // set static collection path on items
        String collectionPath = "";
        if (StringUtils.isNotBlank(getBindingInfo().getCollectionPath())) {
            collectionPath += getBindingInfo().getCollectionPath() + ".";
        }
        if (StringUtils.isNotBlank(getBindingInfo().getBindByNamePrefix())) {
            collectionPath += getBindingInfo().getBindByNamePrefix() + ".";
        }
        collectionPath += getBindingInfo().getBindingName();

        List<DataField> collectionFields = ComponentUtils.getComponentsOfTypeDeep(getItems(), DataField.class);
        for (DataField collectionField : collectionFields) {
            collectionField.getBindingInfo().setCollectionPath(collectionPath);
        }

        List<DataField> addLineCollectionFields = ComponentUtils.getComponentsOfTypeDeep(addLineFields, DataField.class);
        for (DataField collectionField : addLineCollectionFields) {
            collectionField.getBindingInfo().setCollectionPath(collectionPath);
        }

        // add collection entry to abstract classes
        if (!view.getAbstractTypeClasses().containsKey(collectionPath)) {
            view.getAbstractTypeClasses().put(collectionPath, getCollectionObjectClass());
        }

        // initialize container items and sub-collections (since they are not in child list)
        for (Component item : getItems()) {
            view.getViewHelperService().performComponentInitialization(view, model, item);
        }

        // initialize addLineFields
        for (Component item : addLineFields) {
            view.getViewHelperService().performComponentInitialization(view, model, item);
        }

        for (CollectionGroup collectionGroup : getSubCollections()) {
            collectionGroup.getBindingInfo().setCollectionPath(collectionPath);
            view.getViewHelperService().performComponentInitialization(view, model, collectionGroup);
        }
    }

    /**
     * Calls the configured <code>CollectionGroupBuilder</code> to build the necessary components based on the
     * collection data.
     *
     * @see ContainerBase#performApplyModel(View, java.lang.Object, Component)
     */
    @Override
    public void performApplyModel(View view, Object model, Component parent) {
        super.performApplyModel(view, model, parent);

        pushCollectionGroupToReference();

        // if rendering the collection group, build out the lines
        if (isRender()) {
            getCollectionGroupBuilder().build(view, model, this);
        }

        // TODO: is this necessary to call again?
        pushCollectionGroupToReference();
    }

    /**
     * Sets a reference in the context map for all nested components to the collection group instance, and sets name as
     * parameter for an action fields in the group.
     */
    protected void pushCollectionGroupToReference() {
        List<Component> components = this.getComponentsForLifecycle();

        ComponentUtils.pushObjectToContext(components, UifConstants.ContextVariableNames.COLLECTION_GROUP, this);

        List<ActionField> actionFields = ComponentUtils.getComponentsOfTypeDeep(components, ActionField.class);
        for (ActionField actionField : actionFields) {
            actionField.addActionParameter(UifParameters.SELLECTED_COLLECTION_PATH,
                this.getBindingInfo().getBindingPath());
        }
    }

    /**
     * New collection lines are handled in the framework by maintaining a map on the form. The map contains as a key the
     * collection name, and as value an instance of the collection type. An entry is created here for the collection
     * represented by the <code>CollectionGroup</code> if an instance is not available (clearExistingLine will force a
     * new instance). The given model must be a subclass of <code>UifFormBase</code> in order to find the Map.
     *
     * @param model             Model instance that contains the new collection lines Map
     * @param clearExistingLine boolean that indicates whether the line should be set to a new instance if it already
     *                          exists
     */
    public void initializeNewCollectionLine(View view, Object model, CollectionGroup collectionGroup,
                                            boolean clearExistingLine) {
        getCollectionGroupBuilder().initializeNewCollectionLine(view, model, collectionGroup, clearExistingLine);
    }

    /**
     * @see ContainerBase#getComponentsForLifecycle()
     */
    @Override
    public List<Component> getComponentsForLifecycle() {
        List<Component> components = super.getComponentsForLifecycle();
        components.add(addLineLabelField);
        components.add(collectionLookup);

        // remove the containers items because we don't want them as children (they will become children of the layout
        // manager as the rows are created)
        for (Component item : getItems()) {
            if (components.contains(item)) {
                components.remove(item);
            }
        }
        return components;
    }

    /**
     * @see Component#getComponentPrototypes()
     */
    @Override
    public List<Component> getComponentPrototypes() {
        List<Component> components = super.getComponentPrototypes();
        components.addAll(actionFields);
        components.addAll(addLineActionFields);
        components.addAll(getItems());
        components.addAll(getSubCollections());
        components.addAll(addLineFields);
        return components;
    }

    /**
     * @return Object class the collection maintains. Used to get dictionary information in addition to creating new
     *         instances for the collection when necessary.
     */
    public Class<?> getCollectionObjectClass() {
        return this.collectionObjectClass;
    }

    /**
     * @param collectionObjectClass the collection object class to set.
     */
    public void setCollectionObjectClass(Class<?> collectionObjectClass) {
        this.collectionObjectClass = collectionObjectClass;
    }

    public String getPropertyName() {
        return this.propertyName;
    }

    /**
     * @param propertyName the collections property name to set.
     */
    public void setPropertyName(String propertyName) {
        this.propertyName = propertyName;
    }

    /**
     * Determines the binding path for the collection. Used to get the collection value from the model in addition to
     * setting the binding path for the collection attributes.
     *
     * @see DataBinding#getBindingInfo()
     */
    public BindingInfo getBindingInfo() {
        return this.bindingInfo;
    }

    /**
     * @param bindingInfo the binding info instance to set.
     */
    public void setBindingInfo(BindingInfo bindingInfo) {
        this.bindingInfo = bindingInfo;
    }

    /**
     * @return List<ActionField> line action fields that should be rendered for each collection line. Example line
     *         action is the delete action
     */
    public List<ActionField> getActionFields() {
        return this.actionFields;
    }

    /**
     * @param actionFields the line action fields list to set.
     */
    public void setActionFields(List<ActionField> actionFields) {
        this.actionFields = actionFields;
    }

    /**
     * @return boolean true if the actions column should be rendered, false if not
     * @see #getActionFields()
     */
    public boolean isRenderLineActions() {
        return this.renderLineActions;
    }

    /**
     * @param renderLineActions the render line actions indicator to set.
     */
    public void setRenderLineActions(boolean renderLineActions) {
        this.renderLineActions = renderLineActions;
    }

    /**
     * @return boolean true if add line should be rendered, false if it should not be.
     */
    public boolean isRenderAddLine() {
        return this.renderAddLine;
    }

    /**
     * @param renderAddLine the render add line indicator to set.
     */
    public void setRenderAddLine(boolean renderAddLine) {
        this.renderAddLine = renderAddLine;
    }

    /**
     * Convenience getter for the add line label field text. The text is used to label the add line when rendered and
     * its placement depends on the <code>LayoutManager</code>.
     * <p>
     * For the <code>TableLayoutManager</code> the label appears in the sequence column to the left of the add line
     * fields. For the <code>StackedLayoutManager</code> the label is placed into the group header for the line.
     *
     * @return String add line label
     */
    public String getAddLineLabel() {
        if (getAddLineLabelField() != null) {
            return getAddLineLabelField().getLabelText();
        }
        return null;
    }

    /**
     * @param addLineLabel the add line label text to set.
     */
    public void setAddLineLabel(String addLineLabel) {
        if (getAddLineLabelField() != null) {
            getAddLineLabelField().setLabelText(addLineLabel);
        }
    }

    /**
     * @return add line label field
     * @see #getAddLineLabel()
     */
    public LabelField getAddLineLabelField() {
        return this.addLineLabelField;
    }

    /**
     * @param addLineLabelField the <code>LabelField</code> instance for the add line label to set.
     * @see #getAddLineLabel()
     */
    public void setAddLineLabelField(LabelField addLineLabelField) {
        this.addLineLabelField = addLineLabelField;
    }

    /**
     * @return Name of the property that contains an instance for the add line. If set this is used with the binding
     *         info to create the path to the add line. Can be left blank in which case the framework will manage the
     *         add line instance in a generic map.
     */
    public String getAddLinePropertyName() {
        return this.addLinePropertyName;
    }

    /**
     * @param addLinePropertyName the add line property name to set.
     */
    public void setAddLinePropertyName(String addLinePropertyName) {
        this.addLinePropertyName = addLinePropertyName;
    }

    /**
     * @return <code>BindingInfo</code> instance for the add line property used to determine the full binding path. If
     *         add line name given {@link #getAddLineLabel()} then it is set as the binding name on the binding info.
     *         Add line label and binding info are not required, in which case the framework will manage the new add
     *         line instances through a generic map (model must extend UifFormBase)
     */
    public BindingInfo getAddLineBindingInfo() {
        return this.addLineBindingInfo;
    }

    /**
     * @param addLineBindingInfo the add line binding info to set.
     */
    public void setAddLineBindingInfo(BindingInfo addLineBindingInfo) {
        this.addLineBindingInfo = addLineBindingInfo;
    }

    /**
     * @return List<? extends Component> instances that should be rendered for the collection add line (if enabled). If
     *         not set, the default group's items list will be used.
     */
    public List<? extends Component> getAddLineFields() {
        return this.addLineFields;
    }

    /**
     * @param addLineFields the add line field list to set.
     */
    public void setAddLineFields(List<? extends Component> addLineFields) {
        this.addLineFields = addLineFields;
    }

    /**
     * @return List<ActionField> that should be rendered for the add line. This is generally the add action (button) but
     *         can be configured to contain additional actions.
     */
    public List<ActionField> getAddLineActionFields() {
        return this.addLineActionFields;
    }

    /**
     * @param addLineActionFields the add line action fields to set.
     */
    public void setAddLineActionFields(List<ActionField> addLineActionFields) {
        this.addLineActionFields = addLineActionFields;
    }

    /**
     * Indicates whether lines of the collection group should be selected by rendering a field for each line that will
     * allow selection.
     * <p>
     * <p>
     * For example, having the select field enabled could allow selecting multiple lines from a search to return
     * (multi-value lookup).
     *
     * @return boolean true if select field should be rendered, false if not
     */
    public boolean isRenderSelectField() {
        return renderSelectField;
    }

    /**
     * @param renderSelectField the render selected field indicator to set.
     */
    public void setRenderSelectField(boolean renderSelectField) {
        this.renderSelectField = renderSelectField;
    }

    /**
     * When {@link #isRenderSelectField()} is true, gives the name of the property the select field should bind to.
     * <p>
     * <p>
     * Note if no prefix is given in the property name, such as 'form.', it is assumed the property is contained on the
     * collection line. In this case the binding path to the collection line will be appended. In other cases, it is
     * assumed the property is a list or set of String that will hold the selected identifier strings.
     * <p>
     * <p>
     * This property is not required. If not the set the framework will use a property contained on
     * <code>UifFormBase</code>
     *
     * @return property name for select field
     */
    public String getSelectPropertyName() {
        return selectPropertyName;
    }

    /**
     * @param selectPropertyName the property name that will bind to the select field to set.
     */
    public void setSelectPropertyName(String selectPropertyName) {
        this.selectPropertyName = selectPropertyName;
    }

    /**
     * If the collection lookup is enabled (by the render property of the quick finder), {@link
     * #getCollectionObjectClass()} will be used as the data object class for the lookup (if not set). Field
     * conversions need to be set as usual and will be applied for each line returned
     *
     * @return QuickFinder instance that configures a multi-value lookup for the collection
     */
    public QuickFinder getCollectionLookup() {
        return collectionLookup;
    }

    /**
     * @param collectionLookup the collection lookup quickfinder instance to set.
     */
    public void setCollectionLookup(QuickFinder collectionLookup) {
        this.collectionLookup = collectionLookup;
    }

    /**
     * Indicates whether inactive collections lines should be displayed.
     * <p>
     * <p>
     * Setting only applies when the collection line type implements the <code>Inactivatable</code> interface. If true
     * and showInactive is set to false, the collection will be filtered to remove any items whose active status returns
     * false.
     *
     * @return boolean true to show inactive records, false to not render inactive records.
     */
    public boolean isShowInactive() {
        return showInactive;
    }

    /**
     * @param showInactive the show inactive indicator to set.
     */
    public void setShowInactive(boolean showInactive) {
        this.showInactive = showInactive;
    }

    /**
     * @return CollectionFilter instance for filtering the collection data when the showInactive flag is set to false.
     */
    public CollectionFilter getActiveCollectionFilter() {
        return activeCollectionFilter;
    }

    /**
     * @param activeCollectionFilter CollectionFilter instance to use for filter inactive records from the collection.
     */
    public void setActiveCollectionFilter(CollectionFilter activeCollectionFilter) {
        this.activeCollectionFilter = activeCollectionFilter;
    }

    /**
     * @return List of {@link CollectionFilter} instances that should be invoked to filter the collection before
     *         displaying.
     */
    public List<CollectionFilter> getFilters() {
        return filters;
    }

    /**
     * @param filters the List of collection filters for which the collection will be filtered against to set.
     */
    public void setFilters(List<CollectionFilter> filters) {
        this.filters = filters;
    }

    /**
     * @return List<CollectionGroup> instances that are sub-collections of the collection represented by this collection
     *         group.
     */
    public List<CollectionGroup> getSubCollections() {
        return this.subCollections;
    }

    /**
     * @param subCollections the sub collection list to set.
     */
    public void setSubCollections(List<CollectionGroup> subCollections) {
        this.subCollections = subCollections;
    }

    /**
     * Built by the framework as the collection lines are being generated.
     *
     * @return suffix for IDs that identifies the collection line the sub-collection belongs to.
     */
    public String getSubCollectionSuffix() {
        return subCollectionSuffix;
    }

    /**
     * @param subCollectionSuffix the sub-collection suffix (used by framework, should not be set in configuration) to set.
     */
    public void setSubCollectionSuffix(String subCollectionSuffix) {
        this.subCollectionSuffix = subCollectionSuffix;
    }

    /**
     * @return CollectionGroupSecurity instance that indicates what authorization (permissions) exist for the collection
     */
    public CollectionGroupSecurity getCollectionGroupSecurity() {
        return (CollectionGroupSecurity) super.getComponentSecurity();
    }

    /**
     * Override to assert a {@link CollectionGroupSecurity} instance is set
     *
     * @param componentSecurity instance of CollectionGroupSecurity to set.
     */
    @Override
    public void setComponentSecurity(ComponentSecurity componentSecurity) {
        if (!(componentSecurity instanceof CollectionGroupSecurity)) {
            throw new RiceRuntimeException(
                "Component security for CollectionGroup should be instance of CollectionGroupSecurity");
        }

        super.setComponentSecurity(componentSecurity);
    }

    @Override
    protected Class<? extends ComponentSecurity> getComponentSecurityClass() {
        return CollectionGroupSecurity.class;
    }

    /**
     * @return CollectionGroupBuilder instance that will build the components dynamically for the collection instance
     */
    public CollectionGroupBuilder getCollectionGroupBuilder() {
        if (this.collectionGroupBuilder == null) {
            this.collectionGroupBuilder = new CollectionGroupBuilder();
        }
        return this.collectionGroupBuilder;
    }

    /**
     * @param collectionGroupBuilder the collection group building instance to set.
     */
    public void setCollectionGroupBuilder(CollectionGroupBuilder collectionGroupBuilder) {
        this.collectionGroupBuilder = collectionGroupBuilder;
    }

    /**
     * @param showHideInactiveButton the showHideInactiveButton to set
     */
    public void setShowHideInactiveButton(boolean showHideInactiveButton) {
        this.showHideInactiveButton = showHideInactiveButton;
    }

    /**
     * @return the showHideInactiveButton
     */
    public boolean isShowHideInactiveButton() {
        return showHideInactiveButton;
    }

}
