/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.uif.field;

import org.kuali.kfs.krad.uif.component.Component;
import org.kuali.kfs.krad.uif.component.ComponentBase;
import org.kuali.kfs.krad.uif.view.View;

/**
 * Field that contains a header element and optionally a <code>Group</code> to present along with the header text.
 * <p>
 * <p>
 * Generally the group is used to display content to the right of the header, such as links for the group or other
 * information.
 */
public class HeaderField extends FieldGroup {

    private static final long serialVersionUID = -6950408292923393244L;

    private String headerText;
    private String headerLevel;
    private String headerStyleClasses;
    private String headerStyle;
    private String headerDivStyleClasses;
    private String headerDivStyle;

    public HeaderField() {
        super();
    }

    /**
     * The following finalization is performed:
     * <p>
     * <ul>
     * <li>Set render on group to false if no items are configured</li>
     * </ul>
     *
     * @see ComponentBase#performFinalize(View, Object, Component)
     */
    @Override
    public void performFinalize(View view, Object model, Component parent) {
        super.performFinalize(view, model, parent);

        // don't render header group if no items were configured
        if ((getGroup() != null) && (getGroup().getItems().isEmpty())) {
            getGroup().setRender(false);
        }
    }

    /**
     * @return header text that should be displayed on the header.
     */
    public String getHeaderText() {
        return this.headerText;
    }

    /**
     * @param headerText the header text to set.
     */
    public void setHeaderText(String headerText) {
        this.headerText = headerText;
    }

    /**
     * @return HTML header level (h1 ... h6) that should be applied to the header text.
     */
    public String getHeaderLevel() {
        return this.headerLevel;
    }

    /**
     * @param headerLevel the header level to set.
     */
    public void setHeaderLevel(String headerLevel) {
        this.headerLevel = headerLevel;
    }

    /**
     * Note the style class given here applies to only the header text. The style class property inherited from the
     * <code>Component</code> interface can be used to set the class for the whole field div (which could include a
     * nested <code>Group</code>).
     *
     * @return style class that should be applied to the header text (h tag).
     * @see org.kuali.kfs.krad.uif.component.Component#getStyleClasses()
     */
    public String getHeaderStyleClasses() {
        return this.headerStyleClasses;
    }

    /**
     * @param headerStyleClasses the header style class to set.
     */
    public void setHeaderStyleClasses(String headerStyleClasses) {
        this.headerStyleClasses = headerStyleClasses;
    }

    /**
     * Note the style given here applies to only the header text. The style property inherited from the
     * <code>Component</code> interface can be used to set the style for the whole field div (which could include a
     * nested <code>Group</code>).
     *
     * @return header style that should be applied to the header text.
     * @see org.kuali.kfs.krad.uif.component.Component#getStyle()
     */
    public String getHeaderStyle() {
        return this.headerStyle;
    }

    /**
     * @param headerStyle the header style to set.
     */
    public void setHeaderStyle(String headerStyle) {
        this.headerStyle = headerStyle;
    }

    /**
     * Note the style class given here applies to the div surrounding the header tag only.
     *
     * @return style class that should be applied to the header div.
     * @see org.kuali.kfs.krad.uif.component.Component#getStyleClasses()
     */
    public String getHeaderDivStyleClasses() {
        return headerDivStyleClasses;
    }

    /**
     * @param headerDivStyleClasses the header div class to set.
     */
    public void setHeaderDivStyleClasses(String headerDivStyleClasses) {
        this.headerDivStyleClasses = headerDivStyleClasses;
    }

    /**
     * Note the style given here applies to the div surrounding the header tag only.
     *
     * @return header style that should be applied to the header div
     * @see org.kuali.kfs.krad.uif.component.Component#getStyle()
     */
    public String getHeaderDivStyle() {
        return headerDivStyle;
    }

    /**
     * @param headerStyle the header div style to set.
     */
    public void setHeaderDivStyle(String headerDivStyle) {
        this.headerDivStyle = headerDivStyle;
    }
}
