/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.krad.uif.layout;

import org.kuali.kfs.krad.uif.component.Component;
import org.kuali.kfs.krad.uif.component.Configurable;
import org.kuali.kfs.krad.uif.component.PropertyReplacer;
import org.kuali.kfs.krad.uif.container.Container;
import org.kuali.kfs.krad.uif.service.ViewHelperService;
import org.kuali.kfs.krad.uif.util.ComponentUtils;
import org.kuali.kfs.krad.uif.view.View;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Manages the rendering of <code>Component</code> instances within a <code>Container</code>
 */
public interface LayoutManager extends Configurable, Serializable {

    /**
     * The id is used to identify a <code>LayoutManager</code> instance within the tree and can be used by renderers
     *
     * @return the unique id (within a given tree) for the layout manager instance
     */
    String getId();

    /**
     * @param id the unique id (within a given tree) to set as the layout manager id
     */
    void setId(String id);

    /**
     * The path to the JSP file that should be called to invoke the layout manager
     * <p>
     * <p>
     * The path should be relative to the web root. All layout manager templates receive the list of items of be placed,
     * the configured layout manager, and the container to which the layout manager applies
     * <p>
     * <p>
     * e.g. '/krad/WEB-INF/jsp/tiles/boxLayout.jsp'
     *
     * @return String representing the template path
     */
    String getTemplate();

    /**
     * @param template the layout managers template to set.
     */
    void setTemplate(String template);

    /**
     * Should be called to initialize the layout manager
     * <p>
     * <p>
     * This is where layout managers can set defaults and setup other necessary state. The initialize method should only
     * be called once per layout manager lifecycle and is invoked within the initialize phase of the view lifecycle.
     *
     * @param view      View instance the layout manager is a part of
     * @param model     the object instance containing the view data
     * @param container Container the layout manager applies to
     * @see ViewHelperService#performInitialization
     */
    void performInitialization(View view, Object model, Container container);

    /**
     * Called after the initialize phase to perform conditional logic based on the model data
     *
     * @param view      view instance to which the layout manager belongs
     * @param model     Top level object containing the data (could be the form or a top level business object, dto)
     * @param container Container the layout manager applies to
     */
    void performApplyModel(View view, Object model, Container container);

    /**
     * The last phase before the view is rendered. Here final preparations can be made based on the updated view state
     *
     * @param view      view instance that should be finalized for rendering
     * @param model     top level object containing the data
     * @param container Container the layout manager applies to
     */
    void performFinalize(View view, Object model, Container container);

    /**
     * Determines what <code>Container</code> classes are supported by the <code>LayoutManager</code>
     *
     * @return Class<? extends Container> container class supported
     */
    Class<? extends Container> getSupportedContainer();

    /**
     * List of components that are contained within the layout manager that should be sent through the lifecycle
     * <p>
     * <p>
     * Used by <code>ViewHelperService</code> for the various lifecycle callbacks
     *
     * @return List<Component> child components
     */
    List<Component> getComponentsForLifecycle();

    /**
     * List of components that are maintained by the layout manager as prototypes for creating other component instances
     * <p>
     * <p>
     * Prototypes are held for configuring how a component should be created during the lifecycle. An example of this
     * are the fields in a collection group that are created for each collection record. They only participate in the
     * initialize phase.
     *
     * @return List<Component> child component prototypes
     */
    List<Component> getComponentPrototypes();

    /**
     * Used by the copy process to determine for which properties only the value reference should be copied (not a new
     * copy instance). Subclasses can define the properties for which only the reference should be copied
     *
     * @return Set<String> property names for which only the value reference should be copied
     * @see ComponentUtils.copy(T)
     */
    Set<String> getPropertiesForReferenceCopy();

    /**
     * CSS style string to be applied to the area (div) the layout manager generates for the items
     * <p>
     * <p>
     * Note the styleClass/style configured on the <code>Container</code> applies to all the container content (header,
     * body, footer), while the styleClass/style configured on the <code>LayoutManager</code> only applies to the div
     * surrounding the items placed by the manager (the container's body)
     * <p>
     * <p>
     * Any style override or additions can be specified with this attribute. This is used by the renderer to set the
     * style attribute on the corresponding element.
     * <p>
     * <p>
     * e.g. 'color: #000000;text-decoration: underline;'
     *
     * @return String css style string
     */
    String getStyle();

    /**
     * @param style the layout manager div style to set.
     */
    void setStyle(String style);

    /**
     * CSS style class(s) to be applied to the area (div) the layout manager generates for the items
     * <p>
     * <p>
     * Note the styleClass/style configured on the <code>Container</code> applies to all the container content (header,
     * body, footer), while the styleClass/style configured on the <code>LayoutManager</code> only applies to the div
     * surrounding the items placed by the manager (the container's body)
     * <p>
     * <p>
     * Declares additional style classes for the div. Multiple classes are specified with a space delimiter. This is
     * used by the renderer to set the class attribute on the corresponding element. The class(s) declared must be
     * available in the common style sheets or the style sheets specified for the view
     * <p>
     * <p>
     * e.g. 'header left'
     *
     * @return List<String> css style classes to apply
     */
    List<String> getStyleClasses();

    /**
     * @param styleClasses the layout manager div style classes to set.
     */
    void setStyleClasses(List<String> styleClasses);

    /**
     * This method adds a single style class to the list of css style classes on this component
     *
     * @param styleClass style class to add.
     */
    void addStyleClass(String styleClass);

    /**
     * @return Map<String, Object> context map.
     * @see org.kuali.kfs.krad.uif.component.Component#getContext()
     */
    Map<String, Object> getContext();

    /**
     * @param context context map to set.
     * @see org.kuali.kfs.krad.uif.component.Component#setContext(Map)
     */
    void setContext(Map<String, Object> context);

    /**
     * Places the given object into the context Map for the layout manager with the given name
     *
     * @see org.kuali.kfs.krad.uif.component.Component#pushObjectToContext(String, Object)
     */
    void pushObjectToContext(String objectName, Object object);

    /**
     * @return List of <code>PropertyReplacer</code> instances that will be evaluated during the view lifecycle to
     *         conditional set properties on the <code>LayoutManager</code> based on expression evaluations
     */
    List<PropertyReplacer> getPropertyReplacers();

    /**
     * @param propertyReplacers the layout managers property substitutions to set.
     */
    void setPropertyReplacers(List<PropertyReplacer> propertyReplacers);

}
