/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.businessobject;

import org.kuali.kfs.integration.ld.LaborLedgerExpenseTransferSourceAccountingLine;
import org.kuali.kfs.integration.ld.LaborLedgerObject;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.businessobject.SourceAccountingLine;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.rice.core.api.config.property.ConfigurationService;

import java.math.BigDecimal;

/**
 * Labor business object for special case {@link SourceAccountingLine} type for {@code ExpenseTransferDocument}.
 */
public class ExpenseTransferSourceAccountingLine extends SourceAccountingLine implements
        LaborLedgerExpenseTransferSourceAccountingLine, ExpenseTransferAccountingLine,
        Comparable<ExpenseTransferSourceAccountingLine> {

    private String positionNumber;
    private BigDecimal payrollTotalHours;
    private Integer payrollEndDateFiscalYear;
    private String payrollEndDateFiscalPeriodCode;
    private String emplid;
    private LaborObject laborObject;
    private String fringeBenefitView;

    public ExpenseTransferSourceAccountingLine() {
        super();
        payrollTotalHours = new BigDecimal(0);
    }

    @Override
    public String getPositionNumber() {
        return positionNumber;
    }

    @Override
    public void setPositionNumber(String positionNumber) {
        this.positionNumber = positionNumber;
    }

    @Override
    public BigDecimal getPayrollTotalHours() {
        return payrollTotalHours;
    }

    @Override
    public void setPayrollTotalHours(BigDecimal payrollTotalHours) {
        this.payrollTotalHours = payrollTotalHours;
    }

    @Override
    public Integer getPayrollEndDateFiscalYear() {
        return payrollEndDateFiscalYear;
    }

    @Override
    public void setPayrollEndDateFiscalYear(Integer payrollEndDateFiscalYear) {
        this.payrollEndDateFiscalYear = payrollEndDateFiscalYear;
    }

    @Override
    public String getPayrollEndDateFiscalPeriodCode() {
        return payrollEndDateFiscalPeriodCode;
    }

    @Override
    public void setPayrollEndDateFiscalPeriodCode(String payrollEndDateFiscalPeriodCode) {
        this.payrollEndDateFiscalPeriodCode = payrollEndDateFiscalPeriodCode;
    }

    public String getEmplid() {
        return emplid;
    }

    public void setEmplid(String emplid) {
        this.emplid = emplid;
    }

    public LaborObject getLaborObject() {
        return laborObject;
    }

    @Deprecated
    public void setLaborObject(LaborObject laborObject) {
        this.laborObject = laborObject;
    }

    @Override
    public LaborLedgerObject getLaborLedgerObject() {
        return this.laborObject;
    }

    @Override
    @Deprecated
    public void setLaborLedgerObject(LaborLedgerObject laborLedgerObject) {
        this.laborObject = (LaborObject) laborLedgerObject;
    }

    /**
     * Used to for sorting {@link ExpenseTransferAccountingLine} instances within a {@link java.util.Collection} by
     * payrollEndDateFisdalYear and payrollEndDateFiscalPeriodCode
     */
    @Override
    public int compareTo(ExpenseTransferSourceAccountingLine o) {
        if (o == null) {
            throw new NullPointerException("ExpenseTransferAccountingLine is null");
        }
        int retval = getPayrollEndDateFiscalYear().compareTo(o.getPayrollEndDateFiscalYear());

        if (retval == 0) {
            retval = new Integer(getPayrollEndDateFiscalPeriodCode()).compareTo(
                    new Integer(o.getPayrollEndDateFiscalPeriodCode()));
        }

        return retval;
    }

    /**
     * Used to copy a particular transaction line into another This method...
     *
     * @param from
     */
    public void copyFrom(ExpenseTransferAccountingLine from) {
        super.copyFrom(from);
        this.setPayrollTotalHours(from.getPayrollTotalHours());
        this.setPositionNumber(from.getPositionNumber());
        this.setPayrollEndDateFiscalYear(from.getPayrollEndDateFiscalYear());
        this.setPayrollEndDateFiscalPeriodCode(from.getPayrollEndDateFiscalPeriodCode());
        this.setEmplid(from.getEmplid());
    }

    /**
     * This method returns a string so that an fringe benefit inquiry can have a link to view inquiry page from
     * salary transfer.
     *
     * @return the String "View Organization Reversion"
     */
    public String getFringeBenefitView() {
        return SpringContext.getBean(ConfigurationService.class).getPropertyValueAsString(
                KFSKeyConstants.SALARY_TRANSFER_FRINGE_BENEFIT_INQUIRY_LABEL);
    }
}
