/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.dataaccess.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.ojb.broker.query.Criteria;
import org.apache.ojb.broker.query.QueryByCriteria;
import org.apache.ojb.broker.query.QueryFactory;
import org.kuali.kfs.module.ld.businessobject.LaborLedgerPendingEntry;
import org.kuali.kfs.module.ld.dataaccess.LaborLedgerPendingEntryDao;
import org.kuali.kfs.sys.dataaccess.impl.GeneralLedgerPendingEntryDaoOjb;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * OJB Implementation of LaborLedgerPendingEntryDao.
 */
public class LaborLedgerPendingEntryDaoOjb extends GeneralLedgerPendingEntryDaoOjb implements
        LaborLedgerPendingEntryDao {

    private static final Logger LOG = LogManager.getLogger(LaborLedgerPendingEntryDaoOjb.class);
    protected static final String FINANCIAL_DOCUMENT_APPROVED_CODE = "financialDocumentApprovedCode";

    @Override
    public Class getEntryClass() {
        return LaborLedgerPendingEntry.class;
    }

    public Iterator<LaborLedgerPendingEntry> findPendingLedgerEntriesForLedgerBalance(Map fieldValues,
            boolean isApproved, String currentFYPeriodCode, int currentFY, List<String> encumbranceBalanceTypes) {
        return this.findPendingEntries(fieldValues, isApproved, currentFYPeriodCode, currentFY,
                encumbranceBalanceTypes).iterator();
    }

    public Collection<LaborLedgerPendingEntry> hasPendingLaborLedgerEntry(Map fieldValues, Object businessObject) {
        LOG.debug("hasPendingLaborLedgerEntry() started");

        Criteria criteria = new Criteria();
        for (Object key : fieldValues.keySet()) {
            String element = (String) key;
            if ("documentNumber".equals(element)) {
                criteria.addNotEqualTo(element, fieldValues.get(element));
            } else {
                criteria.addEqualTo(element, fieldValues.get(element));
            }
        }

        QueryByCriteria qbc = QueryFactory.newQuery(getEntryClass(), criteria);

        return getPersistenceBrokerTemplate().getCollectionByQuery(qbc);
    }

    @Override
    public void deleteByFinancialDocumentApprovedCode(String financialDocumentApprovedCode) {
        LOG.debug("deleteByFinancialDocumentApprovedCode() started");

        Criteria criteria = new Criteria();
        criteria.addEqualTo(FINANCIAL_DOCUMENT_APPROVED_CODE, financialDocumentApprovedCode);

        QueryByCriteria qbc = QueryFactory.newQuery(this.getEntryClass(), criteria);
        getPersistenceBrokerTemplate().deleteByQuery(qbc);
        getPersistenceBrokerTemplate().clearCache();
    }

    @Override
    public Collection findPendingEntries(Map fieldValues, boolean isApproved, String currentFiscalPeriodCode,
            int currentFiscalYear, List<String> encumbranceBalanceTypes) {
        LOG.debug("findPendingEntries(Map, boolean) started");

        Collection<LaborLedgerPendingEntry> pendingEntries = super.findPendingEntries(fieldValues, isApproved,
                currentFiscalPeriodCode, currentFiscalYear, encumbranceBalanceTypes);

        for (LaborLedgerPendingEntry pendingEntry : pendingEntries) {

            String periodCode = pendingEntry.getUniversityFiscalPeriodCode();
            if (StringUtils.isEmpty(periodCode)) {
                pendingEntry.setUniversityFiscalPeriodCode(currentFiscalPeriodCode);
            }

            Integer fiscalYear = pendingEntry.getUniversityFiscalYear();
            if (fiscalYear == null || StringUtils.isEmpty(fiscalYear.toString())) {
                pendingEntry.setUniversityFiscalYear(currentFiscalYear);
            }
        }

        return pendingEntries;
    }
}
