/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document.authorization;

import org.kuali.kfs.module.ld.LaborPropertyConstants;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.document.AccountingDocument;
import org.kuali.kfs.sys.document.authorization.AccountingLineAuthorizerBase;
import org.kuali.kfs.sys.document.web.AccountingLineRenderingContext;
import org.kuali.kfs.sys.document.web.AccountingLineViewAction;

import java.util.Map;

/**
 * Data dictionary definition that includes metadata for an accounting document about one of its groups of accounting
 * lines (typically source vs. target, but this should open things up).
 */
public class LaborExpenseTransferAccountingLineAuthorizer extends AccountingLineAuthorizerBase {

    @Override
    public boolean determineEditPermissionOnField(AccountingDocument accountingDocument, AccountingLine accountingLine,
            String accountingLineCollectionProperty, String fieldName, boolean editablePage) {
        if (fieldName.equals(KFSPropertyConstants.AMOUNT)) {
            return true;
        }

        if (fieldName.equals(LaborPropertyConstants.PAYROLL_TOTAL_HOURS)) {
            return true;
        }

        return super.determineEditPermissionOnField(accountingDocument, accountingLine, accountingLineCollectionProperty,
                fieldName, editablePage);
    }

    @Override
    protected Map<String, AccountingLineViewAction> getActionMap(
            AccountingLineRenderingContext accountingLineRenderingContext, String accountingLinePropertyName,
            Integer accountingLineIndex, String groupTitle) {
        Map<String, AccountingLineViewAction> actionMap = super.getActionMap(accountingLineRenderingContext,
                accountingLinePropertyName, accountingLineIndex, groupTitle);

        if (accountingLineRenderingContext.getAccountingLine().isSourceAccountingLine()) {
            AccountingLineViewAction copyAction = getCopyAction(accountingLineRenderingContext.getAccountingLine(),
                    accountingLinePropertyName, accountingLineIndex, groupTitle);
            actionMap.put(KFSConstants.COPY_METHOD, copyAction);
        }

        return actionMap;
    }

    /**
     * construct the copy action for the given accounting line, typically, a new accounting line
     *
     * @param accountingLine             the given accounting line
     * @param accountingLinePropertyName the property name of the given account line, typically, the form name
     * @param accountingLineIndex        the index of the given accounting line in its accounting line group
     * @param groupTitle                 the title of the accounting line group
     * @return the copy action for the given accounting line
     */
    protected AccountingLineViewAction getCopyAction(AccountingLine accountingLine, String accountingLinePropertyName,
            Integer accountingLineIndex, String groupTitle) {
        String actionMethod = getCopyLineMethod(accountingLine, accountingLinePropertyName, accountingLineIndex);
        String actionLabel = getActionLabel(
                KFSKeyConstants.AccountingLineViewRendering.ACCOUNTING_LINE_COPY_ACTION_LABEL, groupTitle,
                accountingLineIndex + 1);

        return new AccountingLineViewAction(actionMethod, actionLabel, "btn clean", "Copy", "fa fa-files-o");
    }

    /**
     * Builds the action method name of the method that deletes accounting lines for this group
     *
     * @param accountingLine         the accounting line an action is being checked for
     * @param accountingLineProperty the property name of the accounting line
     * @param accountingLineIndex    the index of the given accounting line within the the group being rendered
     * @return the action method name of the method that deletes accounting lines for this group
     */
    protected String getCopyLineMethod(AccountingLine accountingLine, String accountingLineProperty,
            Integer accountingLineIndex) {
        final String infix = getActionInfixForExtantAccountingLine(accountingLine, accountingLineProperty);
        return "copyAccountingLine.line" + accountingLineIndex + ".anchoraccounting" + infix + "Anchor";
    }
}
