/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document.validation.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.kns.service.DataDictionaryService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.businessobject.LaborJournalVoucherDetail;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.document.validation.GenericValidation;
import org.kuali.kfs.sys.document.validation.event.AttributedDocumentEvent;
import org.kuali.rice.kim.api.identity.PersonService;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * Validates that a labor journal voucher document's accounting lines have valid Employee ID
 */
public class LaborJournalVoucherEmployeeIDExistenceCheckValidation extends GenericValidation {

    private AccountingLine accountingLineForValidation;
    private DataDictionaryService dataDictionaryService;
    private PersonService personService;

    /**
     * Validates that the accounting line in the labor journal voucher document have an existing employee id
     */
    public boolean validate(AttributedDocumentEvent event) {
        boolean result = true;

        LaborJournalVoucherDetail laborJournalVoucherDetail =
                (LaborJournalVoucherDetail) getAccountingLineForValidation();
        String emplid = laborJournalVoucherDetail.getEmplid();

        if (StringUtils.isBlank(emplid) || LaborConstants.getDashEmplId().equals(emplid)) {
            return true;
        }
        if (!employeeIDExistenceCheck(emplid)) {
            result = false;
        }
        return result;
    }

    /**
     * Checks whether employee id exists
     *
     * @param employeeId employee id is checked against the collection of universal users
     * @return True if the given employee id exists, false otherwise.
     */
    protected boolean employeeIDExistenceCheck(String employeeId) {
        boolean employeeIDExists = true;
        Map<String, String> criteria = new HashMap<>();
        criteria.put(KFSPropertyConstants.PERSON_PAYROLL_IDENTIFIER, employeeId);

        Collection emplidMatches = personService.findPeople(criteria);
        if (emplidMatches == null || emplidMatches.isEmpty()) {
            String label = dataDictionaryService.getDataDictionary()
                    .getBusinessObjectEntry(LaborJournalVoucherDetail.class.getName()).getAttributeDefinition(
                            KFSPropertyConstants.EMPLID).getLabel();
            GlobalVariables.getMessageMap().putError(KFSPropertyConstants.EMPLID, KFSKeyConstants.ERROR_EXISTENCE,
                    label);
            employeeIDExists = false;
        }

        return employeeIDExists;
    }

    public AccountingLine getAccountingLineForValidation() {
        return accountingLineForValidation;
    }

    public void setAccountingLineForValidation(AccountingLine accountingLineForValidation) {
        this.accountingLineForValidation = accountingLineForValidation;
    }

    public void setDataDictionaryService(DataDictionaryService dataDictionaryService) {
        this.dataDictionaryService = dataDictionaryService;
    }

    public void setPersonService(PersonService personService) {
        this.personService = personService;
    }
}

