/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document.validation.impl;

import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.ld.LaborKeyConstants;
import org.kuali.kfs.module.ld.LaborPropertyConstants;
import org.kuali.kfs.module.ld.businessobject.ExpenseTransferSourceAccountingLine;
import org.kuali.kfs.module.ld.businessobject.ExpenseTransferTargetAccountingLine;
import org.kuali.kfs.module.ld.document.SalaryExpenseTransferDocument;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.sys.document.AccountingDocument;
import org.kuali.kfs.sys.document.validation.GenericValidation;
import org.kuali.kfs.sys.document.validation.event.AttributedDocumentEvent;

import java.util.ArrayList;
import java.util.List;

public class SalaryExpenseTransferDuplicateAccountingLineValidation extends GenericValidation {

    private AccountingDocument accountingDocumentForValidation;

    public boolean validate(AttributedDocumentEvent event) {
        if (hasDuplicateAccountingLine()) {
            GlobalVariables.getMessageMap().addToErrorPath(KFSPropertyConstants.DOCUMENT);
            GlobalVariables.getMessageMap().putError(KFSPropertyConstants.TARGET_ACCOUNTING_LINES,
                    LaborKeyConstants.ERROR_DUPLICATE_ACCOUNTING_LINE);
            GlobalVariables.getMessageMap().removeFromErrorPath(KFSPropertyConstants.DOCUMENT);
            return false;
        }

        return true;
    }

    /**
     * Validates that a target accounting line does not duplicate any of the source lines
     *
     * @return false if a target accounting line duplicates a source line; otherwise, false
     */
    private boolean hasDuplicateAccountingLine() {
        AccountingDocument accountingDocument = getAccountingDocumentForValidation();
        SalaryExpenseTransferDocument salaryExpenseTransferDocument = (SalaryExpenseTransferDocument) accountingDocument;

        List<ExpenseTransferSourceAccountingLine> sourceAccountingLines =
                salaryExpenseTransferDocument.getSourceAccountingLines();
        List<ExpenseTransferTargetAccountingLine> targetAccountingLines =
                salaryExpenseTransferDocument.getTargetAccountingLines();

        final List<String> key = defaultKeyOfExpenseTransferAccountingLine();

        return targetAccountingLines.stream()
                .anyMatch(targetAccountingLine -> sourceAccountingLines.stream()
                        .anyMatch(sourceAccountingLine ->
                                ObjectUtil.equals(targetAccountingLine, sourceAccountingLine, key)));
    }

    private List<String> defaultKeyOfExpenseTransferAccountingLine() {
        List<String> defaultKey = new ArrayList<>();

        defaultKey.add(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        defaultKey.add(KFSPropertyConstants.ACCOUNT_NUMBER);
        defaultKey.add(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
        defaultKey.add(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
        defaultKey.add(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);
        defaultKey.add(KFSPropertyConstants.PROJECT_CODE);
        defaultKey.add(KFSPropertyConstants.ORGANIZATION_REFERENCE_ID);
        defaultKey.add(LaborPropertyConstants.PAYROLL_END_DATE_FISCAL_YEAR);
        defaultKey.add(LaborPropertyConstants.PAYROLL_END_DATE_FISCAL_PERIOD_CODE);
        defaultKey.add(KFSPropertyConstants.AMOUNT);

        return defaultKey;
    }

    public AccountingDocument getAccountingDocumentForValidation() {
        return accountingDocumentForValidation;
    }

    public void setAccountingDocumentForValidation(AccountingDocument accountingDocumentForValidation) {
        this.accountingDocumentForValidation = accountingDocumentForValidation;
    }
}
