/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.service.impl;

import org.apache.commons.collections4.IteratorUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.gl.OJBUtility;
import org.kuali.kfs.module.ld.businessobject.EmployeeFunding;
import org.kuali.kfs.module.ld.businessobject.LaborBalanceSummary;
import org.kuali.kfs.module.ld.businessobject.LaborTransaction;
import org.kuali.kfs.module.ld.businessobject.LedgerBalance;
import org.kuali.kfs.module.ld.businessobject.LedgerBalanceForYearEndBalanceForward;
import org.kuali.kfs.module.ld.dataaccess.LaborLedgerBalanceDao;
import org.kuali.kfs.module.ld.service.LaborCalculatedSalaryFoundationTrackerService;
import org.kuali.kfs.module.ld.service.LaborLedgerBalanceService;
import org.kuali.kfs.module.ld.util.DebitCreditUtil;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class LaborLedgerBalanceServiceImpl implements LaborLedgerBalanceService {

    private static final Logger LOG = LogManager.getLogger();

    protected LaborLedgerBalanceDao laborLedgerBalanceDao;
    protected LaborCalculatedSalaryFoundationTrackerService laborCalculatedSalaryFoundationTrackerService;

    @Override
    public Iterator<LedgerBalance> findBalancesForFiscalYear(Integer fiscalYear) {
        return laborLedgerBalanceDao.findBalancesForFiscalYear(fiscalYear);
    }

    @Override
    public Iterator<LedgerBalance> findBalancesForFiscalYear(Integer fiscalYear, Map<String, String> fieldValues,
            List<String> encumbranceBalanceTypes) {
        return laborLedgerBalanceDao.findBalancesForFiscalYear(fiscalYear, fieldValues, encumbranceBalanceTypes);
    }

    @Override
    public Iterator<LedgerBalanceForYearEndBalanceForward> findBalancesForFiscalYear(Integer fiscalYear,
            Map<String, String> fieldValues, Collection<String> subFundGroupCodes, Collection<String> fundGroupCodes) {
        List<String> fundGroupCodesList = new ArrayList<>(fundGroupCodes);
        Collections.sort(fundGroupCodesList);
        List<String> subFundGroupCodesList = new ArrayList<>(subFundGroupCodes);
        Collections.sort(subFundGroupCodesList);
        return laborLedgerBalanceDao.findBalancesForFiscalYear(fiscalYear, fieldValues, subFundGroupCodesList,
                fundGroupCodesList);
    }

    @Override
    public Iterator findBalance(Map fieldValues, boolean isConsolidated, List<String> encumbranceBalanceTypes,
            boolean noZeroAmounts) {
        LOG.debug("findBalance() started");
        return laborLedgerBalanceDao.findBalance(fieldValues, isConsolidated, encumbranceBalanceTypes, noZeroAmounts);
    }

    @Override
    @Deprecated
    public Iterator findBalance(Map fieldValues, boolean isConsolidated, List<String> encumbranceBalanceTypes) {
        LOG.debug("findBalance() started");
        return laborLedgerBalanceDao.findBalance(fieldValues, isConsolidated, encumbranceBalanceTypes);
    }

    @Override
    public Integer getBalanceRecordCount(Map fieldValues, boolean isConsolidated, List<String> encumbranceBalanceTypes,
            boolean noZeroAmounts) {
        LOG.debug("getBalanceRecordCount() started");

        int recordCount;
        if (!isConsolidated) {
            recordCount = OJBUtility.getResultSizeFromMap(fieldValues, new LedgerBalance()).intValue();
        } else {
            Iterator recordCountIterator = laborLedgerBalanceDao.getConsolidatedBalanceRecordCount(fieldValues,
                    encumbranceBalanceTypes, noZeroAmounts);
            List recordCountList = IteratorUtils.toList(recordCountIterator);
            recordCount = recordCountList.size();
        }
        return recordCount;
    }

    @Override
    @Deprecated
    public Integer getBalanceRecordCount(Map fieldValues, boolean isConsolidated,
            List<String> encumbranceBalanceTypes) {
        LOG.debug("getBalanceRecordCount() started");

        int recordCount;
        if (!isConsolidated) {
            recordCount = OJBUtility.getResultSizeFromMap(fieldValues, new LedgerBalance()).intValue();
        } else {
            Iterator recordCountIterator = laborLedgerBalanceDao.getConsolidatedBalanceRecordCount(fieldValues,
                    encumbranceBalanceTypes);
            List recordCountList = IteratorUtils.toList(recordCountIterator);
            recordCount = recordCountList.size();
        }
        return recordCount;
    }

    @Override
    public <T extends LedgerBalance> T findLedgerBalance(Collection<T> ledgerBalanceCollection,
            LaborTransaction transaction, List<String> keyList) {
        for (T ledgerBalance : ledgerBalanceCollection) {
            boolean found = ObjectUtil.equals(ledgerBalance, transaction, keyList);
            if (found) {
                return ledgerBalance;
            }
        }
        return null;
    }

    @Override
    public <T extends LedgerBalance> T findLedgerBalance(Collection<T> ledgerBalanceCollection,
            LaborTransaction transaction) {
        for (T ledgerBalance : ledgerBalanceCollection) {
            boolean found = ObjectUtil.equals(ledgerBalance, transaction, ledgerBalance.getPrimaryKeyList());
            if (found) {
                return ledgerBalance;
            }
        }
        return null;
    }

    @Override
    @Transactional
    public <T extends LedgerBalance> void updateLedgerBalance(T ledgerBalance, LaborTransaction transaction) {
        String debitCreditCode = transaction.getTransactionDebitCreditCode();
        KualiDecimal amount = transaction.getTransactionLedgerEntryAmount();
        amount = DebitCreditUtil.getNumericAmount(amount, debitCreditCode);
        ledgerBalance.addAmount(transaction.getUniversityFiscalPeriodCode(), amount);
    }

    @Override
    @Transactional
    public LedgerBalance addLedgerBalance(Collection<LedgerBalance> ledgerBalanceCollection,
            LaborTransaction transaction) {
        LedgerBalance ledgerBalance = this.findLedgerBalance(ledgerBalanceCollection, transaction);

        if (ledgerBalance == null) {
            LedgerBalance newLedgerBalance = new LedgerBalance();
            ObjectUtil.buildObject(newLedgerBalance, transaction);
            updateLedgerBalance(newLedgerBalance, transaction);

            ledgerBalanceCollection.add(newLedgerBalance);
            return newLedgerBalance;
        }
        return null;
    }

    @Override
    public List<EmployeeFunding> findEmployeeFunding(Map fieldValues, boolean isConsolidated) {
        List<EmployeeFunding> currentFundsCollection = laborLedgerBalanceDao.findCurrentEmployeeFunds(fieldValues);
        List<EmployeeFunding> encumbranceFundsCollection = laborLedgerBalanceDao.findEncumbranceEmployeeFunds(
                fieldValues);

        // merge encumbrance with the current funds
        for (EmployeeFunding encumbranceFunding : encumbranceFundsCollection) {
            KualiDecimal encumbrance = encumbranceFunding.getAccountLineAnnualBalanceAmount()
                    .add(encumbranceFunding.getContractsGrantsBeginningBalanceAmount());
            encumbranceFunding.setOutstandingEncumbrance(encumbrance);

            if (currentFundsCollection.contains(encumbranceFunding)) {
                int index = currentFundsCollection.indexOf(encumbranceFunding);
                currentFundsCollection.get(index).setOutstandingEncumbrance(encumbrance);
            } else if (encumbrance != null && encumbrance.isNonZero()) {
                currentFundsCollection.add(encumbranceFunding);
            }
        }

        // update the employee fundings
        for (EmployeeFunding employeeFunding : currentFundsCollection) {
            employeeFunding.setCurrentAmount(employeeFunding.getAccountLineAnnualBalanceAmount());
        }
        return currentFundsCollection;
    }

    @Override
    public List<EmployeeFunding> findEmployeeFundingWithCSFTracker(Map fieldValues, boolean isConsolidated) {
        List<EmployeeFunding> currentFundsCollection = this.findEmployeeFunding(fieldValues, isConsolidated);
        List<EmployeeFunding> CSFTrackersCollection = laborCalculatedSalaryFoundationTrackerService
                .findCSFTrackersAsEmployeeFunding(fieldValues, isConsolidated);

        for (EmployeeFunding CSFTrackerAsEmployeeFunding : CSFTrackersCollection) {
            if (currentFundsCollection.contains(CSFTrackerAsEmployeeFunding)) {
                int index = currentFundsCollection.indexOf(CSFTrackerAsEmployeeFunding);
                EmployeeFunding currentFunds = currentFundsCollection.get(index);

                currentFunds.setCsfDeleteCode(CSFTrackerAsEmployeeFunding.getCsfDeleteCode());
                currentFunds.setCsfTimePercent(CSFTrackerAsEmployeeFunding.getCsfTimePercent());
                currentFunds.setCsfFundingStatusCode(CSFTrackerAsEmployeeFunding.getCsfFundingStatusCode());
                currentFunds.setCsfAmount(CSFTrackerAsEmployeeFunding.getCsfAmount());
                currentFunds.setCsfFullTimeEmploymentQuantity(CSFTrackerAsEmployeeFunding
                        .getCsfFullTimeEmploymentQuantity());
            } else {
                /*KFSCNTRB-1534 It is possible for a CSF item to exist on its own without being related to
                  a current funds record.*/
                currentFundsCollection.add(CSFTrackerAsEmployeeFunding);
            }
        }

        return currentFundsCollection;
    }

    @Override
    public List<LaborBalanceSummary> findBalanceSummary(Integer fiscalYear, Collection<String> balanceTypes) {
        return laborLedgerBalanceDao.findBalanceSummary(fiscalYear, balanceTypes);
    }

    @Override
    public List<List<String>> findAccountsInFundGroups(Integer fiscalYear, Map<String, String> fieldValues,
            List<String> subFundGroupCodes, List<String> fundGroupCodes) {
        return laborLedgerBalanceDao.findAccountsInFundGroups(fiscalYear, fieldValues, subFundGroupCodes,
                fundGroupCodes);
    }

    @Override
    public Collection<LedgerBalance> findLedgerBalances(Map<String, List<String>> fieldValues, Map<String,
            List<String>> excludedFieldValues, Set<Integer> fiscalYears, List<String> balanceTypeList,
            List<String> positionObjectGroupCodes) {
        return laborLedgerBalanceDao.findLedgerBalances(fieldValues, excludedFieldValues, fiscalYears,
                balanceTypeList, positionObjectGroupCodes);
    }

    @Override
    @Transactional
    public void deleteLedgerBalancesPriorToYear(Integer fiscalYear, String chartOfAccountsCode) {
        laborLedgerBalanceDao.deleteLedgerBalancesPriorToYear(fiscalYear, chartOfAccountsCode);
    }

    public void setLaborLedgerBalanceDao(LaborLedgerBalanceDao laborLedgerBalanceDao) {
        this.laborLedgerBalanceDao = laborLedgerBalanceDao;
    }

    public void setLaborCalculatedSalaryFoundationTrackerService(
            LaborCalculatedSalaryFoundationTrackerService laborCalculatedSalaryFoundationTrackerService) {
        this.laborCalculatedSalaryFoundationTrackerService = laborCalculatedSalaryFoundationTrackerService;
    }
}
