/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document.validation.impl;

import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.ld.LaborKeyConstants;
import org.kuali.kfs.module.ld.businessobject.ExpenseTransferAccountingLine;
import org.kuali.kfs.module.ld.document.LaborExpenseTransferDocumentBase;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.document.validation.GenericValidation;
import org.kuali.kfs.sys.document.validation.event.AttributedDocumentEvent;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.util.List;

public class LaborExpenseTransferAccountLineTotalsMatchValidation extends GenericValidation {

    private Document documentForValidation;

    /**
     * Determine whether the given accounting line has already been in the given document.
     */
    public boolean validate(AttributedDocumentEvent event) {
        Document documentForValidation = getDocumentForValidation();

        LaborExpenseTransferDocumentBase expenseTransferDocument =
                (LaborExpenseTransferDocumentBase) documentForValidation;

        List sourceLines = expenseTransferDocument.getSourceAccountingLines();
        List targetLines = expenseTransferDocument.getTargetAccountingLines();

        // check to ensure totals of accounting lines in source and target sections match
        if (!isAccountingLineTotalsMatch(sourceLines, targetLines)) {
            GlobalVariables.getMessageMap().putError(KFSPropertyConstants.SOURCE_ACCOUNTING_LINES,
                    LaborKeyConstants.ACCOUNTING_LINE_TOTALS_MISMATCH_ERROR);
            return false;
        }

        return true;
    }

    /**
     * This method checks if the total sum amount of the source accounting line matches the total sum amount of the
     * target accounting line, return true if the totals match, false otherwise.
     *
     * @param sourceLines
     * @param targetLines
     * @return
     */
    public boolean isAccountingLineTotalsMatch(List sourceLines, List targetLines) {
        boolean isValid = true;

        AccountingLine line;

        // totals for the from and to lines.
        KualiDecimal sourceLinesAmount = KualiDecimal.ZERO;
        KualiDecimal targetLinesAmount = KualiDecimal.ZERO;

        // sum source lines
        for (Object sourceLine : sourceLines) {
            line = (ExpenseTransferAccountingLine) sourceLine;
            sourceLinesAmount = sourceLinesAmount.add(line.getAmount());
        }

        // sum target lines
        for (Object targetLine : targetLines) {
            line = (ExpenseTransferAccountingLine) targetLine;
            targetLinesAmount = targetLinesAmount.add(line.getAmount());
        }

        // if totals don't match, then add error message
        if (sourceLinesAmount.compareTo(targetLinesAmount) != 0) {
            isValid = false;
        }

        return isValid;
    }

    public Document getDocumentForValidation() {
        return documentForValidation;
    }

    public void setDocumentForValidation(Document documentForValidation) {
        this.documentForValidation = documentForValidation;
    }
}
