/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch.service.impl;

import org.apache.commons.io.filefilter.RegexFileFilter;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.gl.GLParameterConstants;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.gl.batch.PosterSummaryReportStep;
import org.kuali.kfs.gl.businessobject.GlSummary;
import org.kuali.kfs.gl.businessobject.OriginEntryInformation;
import org.kuali.kfs.gl.report.PosterOutputSummaryReport;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.batch.service.LaborBalanceSummaryReportService;
import org.kuali.kfs.module.ld.businessobject.LaborBalanceSummary;
import org.kuali.kfs.module.ld.service.LaborLedgerBalanceService;
import org.kuali.kfs.module.ld.util.LaborOriginEntryFileIterator;
import org.kuali.kfs.sys.FileUtil;
import org.kuali.kfs.sys.batch.service.WrappingBatchService;
import org.kuali.kfs.sys.businessobject.SystemOptions;
import org.kuali.kfs.sys.service.FiscalYearAwareReportWriterService;
import org.kuali.kfs.sys.service.OptionsService;
import org.kuali.kfs.sys.service.ReportWriterService;
import org.kuali.kfs.core.api.datetime.DateTimeService;

import java.io.File;
import java.sql.Date;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Locale;

/**
 * Implements a set of methods that can generate labor balance summary reports
 */
public class LaborBalanceSummaryReportServiceImpl implements LaborBalanceSummaryReportService {

    private static final Logger LOG = LogManager.getLogger();

    protected DateTimeService dateTimeService;
    protected OptionsService optionsService;
    protected ParameterService parameterService;

    protected LaborLedgerBalanceService laborLedgerBalanceService;
    protected ReportWriterService laborPosterOutputSummaryReportWriterService;
    protected FiscalYearAwareReportWriterService laborActualBalanceSummaryReportWriterService;
    protected FiscalYearAwareReportWriterService laborBudgetBalanceSummaryReportWriterService;
    protected FiscalYearAwareReportWriterService laborEncumbranceSummaryReportWriterService;

    protected String batchFileDirectoryName;

    @Override
    public void generateBalanceSummaryReports() {
        LOG.debug("generateBalanceSummaryReports() started");

        final Date runDate = dateTimeService.getCurrentSqlDate();
        generatePosterOutputSummaryReport(runDate);
        generateBalanceSummaryReports(runDate);
    }

    @Override
    public void generateBalanceSummaryReports(final Date runDate) {
        LOG.debug("generateBalanceSummaryReports(Date) started");

        final String yearEndPeriodLowerBound = parameterService.getParameterValueAsString(PosterSummaryReportStep.class,
                GLParameterConstants.FISCAL_YEAR_BEGIN
        );
        final String lastDayOfFiscalYear = parameterService.getParameterValueAsString(PosterSummaryReportStep.class,
                GLParameterConstants.FISCAL_YEAR_END
        );
        final String yearEndPeriodUpperBound = parameterService.getParameterValueAsString(PosterSummaryReportStep.class,
                GLParameterConstants.PRIOR_YEAR_END
        );

        final Integer currentYear = optionsService.getCurrentYearOptions().getUniversityFiscalYear();
        generateBalanceSummaryReports(currentYear, runDate);

        // if today is within the lower bound of the year end period, then generate reports for the next fiscal year
        if (isInYearEndLowerBound(runDate, yearEndPeriodLowerBound, lastDayOfFiscalYear)) {
            generateBalanceSummaryReports(currentYear + 1, runDate);
        }

        // if today is within the upper bound of the year end period, then generate reports for the last fiscal year
        if (isInYearEndUpperBound(runDate, yearEndPeriodUpperBound, lastDayOfFiscalYear)) {
            generateBalanceSummaryReports(currentYear - 1, runDate);
        }
    }

    // generate a set of balance summary reports for actual, budget and encumbrance balances
    protected void generateBalanceSummaryReports(final Integer fiscalYear, final Date runDate) {
        final SystemOptions options = optionsService.getOptions(fiscalYear);
        if (options == null) {
            LOG.fatal("The data for {}have NOT been setup.", fiscalYear);
            return;
        }

        final List<String> actualsBalanceTypes = getActualBalanceTypes(fiscalYear);
        writeSummaryReport(fiscalYear, actualsBalanceTypes, laborActualBalanceSummaryReportWriterService);

        final List<String> budgetBalanceTypes = getBudgetBalanceTypes(fiscalYear);
        writeSummaryReport(fiscalYear, budgetBalanceTypes, laborBudgetBalanceSummaryReportWriterService);

        final List<String> encumbranceBalanceTypes = getEncumbranceBalanceTypes(fiscalYear);
        writeSummaryReport(fiscalYear, encumbranceBalanceTypes, laborEncumbranceSummaryReportWriterService);
    }

    protected void writeSummaryReport(
            final Integer fiscalYear, final List<String> balanceTypes,
            final FiscalYearAwareReportWriterService reportWriterService) {
        final List<LaborBalanceSummary> balanceSummary = laborLedgerBalanceService.findBalanceSummary(fiscalYear, balanceTypes);
        final List<GlSummary> summaryList = new ArrayList<>(balanceSummary);

        final GlSummary totals = new LaborBalanceSummary();
        for (final GlSummary summaryLine : summaryList) {
            totals.add(summaryLine);
        }
        totals.setFundGroup("Total");

        try {
            reportWriterService.setFiscalYear(fiscalYear);
            ((WrappingBatchService) reportWriterService).initialize();
            reportWriterService.writeSubTitle("Balance Type of " + balanceTypes + " for Fiscal Year " + fiscalYear);
            reportWriterService.writeNewLines(1);

            reportWriterService.writeTableRowSeparationLine(totals);
            reportWriterService.writeTable(summaryList, true, false);

            reportWriterService.writeTableRowSeparationLine(totals);
            reportWriterService.writeTableRow(totals);
        } finally {
            ((WrappingBatchService) reportWriterService).destroy();
        }
    }

    /**
     * Generates reports about the output of a poster run.
     *
     * @param runDate the date the poster was run.
     */
    protected void generatePosterOutputSummaryReport(final Date runDate) {
        final PosterOutputSummaryReport posterOutputSummaryReport = new PosterOutputSummaryReport();

        // summarize all the entries for the main poster
        final File mainPosterFile = FileUtil.getNewestFile(new File(batchFileDirectoryName), new RegexFileFilter(
                LaborConstants.BatchFileSystem.POSTER_INPUT_FILE + "\\.[0-9_\\-]+\\" +
                        GeneralLedgerConstants.BatchFileSystem.EXTENSION));
        if (mainPosterFile != null && mainPosterFile.exists()) {
            final LaborOriginEntryFileIterator mainPosterIterator = new LaborOriginEntryFileIterator(mainPosterFile);
            while (mainPosterIterator.hasNext()) {
                final OriginEntryInformation originEntry = mainPosterIterator.next();
                posterOutputSummaryReport.summarize(originEntry);
            }
        } else {
            LOG.warn(
                    "Could not Main Poster Input file, {}, for tabulation in the Poster Output Summary Report",
                    LaborConstants.BatchFileSystem.POSTER_INPUT_FILE
            );
        }

        posterOutputSummaryReport.writeReport(laborPosterOutputSummaryReportWriterService);
    }

    /**
     * get the encumbrance balance type codes for the given fiscal year
     *
     * @param fiscalYear the given fiscal year
     * @return the encumbrance balance type codes for the given fiscal year
     */
    protected List<String> getEncumbranceBalanceTypes(final Integer fiscalYear) {
        final SystemOptions options = optionsService.getOptions(fiscalYear);

        final List<String> balanceTypes = new ArrayList<>();
        balanceTypes.add(options.getExtrnlEncumFinBalanceTypCd());
        balanceTypes.add(options.getIntrnlEncumFinBalanceTypCd());
        balanceTypes.add(options.getPreencumbranceFinBalTypeCd());
        balanceTypes.add(options.getCostShareEncumbranceBalanceTypeCd());
        return balanceTypes;
    }

    /**
     * get the actual balance type codes for the given fiscal year
     *
     * @param fiscalYear the given fiscal year
     * @return the actual balance type codes for the given fiscal year
     */
    protected List<String> getActualBalanceTypes(final Integer fiscalYear) {
        final SystemOptions options = optionsService.getOptions(fiscalYear);

        final List<String> balanceTypes = new ArrayList<>();
        balanceTypes.add(options.getActualFinancialBalanceTypeCd());
        return balanceTypes;
    }

    /**
     * get the budget balance type codes for the given fiscal year
     *
     * @param fiscalYear the given fiscal year
     * @return the budget balance type codes for the given fiscal year
     */
    protected List<String> getBudgetBalanceTypes(final Integer fiscalYear) {
        final SystemOptions options = optionsService.getOptions(fiscalYear);

        final List<String> balanceTypes = new ArrayList<>();
        balanceTypes.add(options.getBudgetCheckingBalanceTypeCd());
        balanceTypes.add(options.getBaseBudgetFinancialBalanceTypeCd());
        balanceTypes.add(options.getMonthlyBudgetFinancialBalanceTypeCd());
        return balanceTypes;
    }

    /**
     * determine if the given date is within the year end period
     *
     * @param runDate                 the given date
     * @param yearEndPeriodLowerBound the lower bound date of year end period
     * @param yearEndPeriodUpperBound the upper bound date of year end period
     * @param lastDayOfFiscalYear     the last day of the current fiscal year
     * @return true if the given date is within the lower bound of year end period; otherwise, false
     */
    protected boolean isInYearEndPeriod(
            final Date runDate, final String yearEndPeriodLowerBound, final String yearEndPeriodUpperBound,
            final String lastDayOfFiscalYear) {
        return isInYearEndLowerBound(runDate, yearEndPeriodLowerBound, lastDayOfFiscalYear)
                || isInYearEndUpperBound(runDate, yearEndPeriodUpperBound, lastDayOfFiscalYear);
    }

    /**
     * determine if the given date is within the lower bound of year end period
     *
     * @param runDate                 the given date
     * @param yearEndPeriodLowerBound the lower bound date of year end period
     * @param lastDayOfFiscalYear     the last day of the current fiscal year
     * @return true if the given date is within the lower bound of year end period; otherwise, false
     */
    protected boolean isInYearEndLowerBound(final Date runDate, final String yearEndPeriodLowerBound, final String lastDayOfFiscalYear) {
        final SimpleDateFormat sdf = new SimpleDateFormat("MMdd", Locale.US);
        final String today = sdf.format(runDate);
        return today.compareTo(yearEndPeriodLowerBound) >= 0 && today.compareTo(lastDayOfFiscalYear) <= 0;
    }

    /**
     * determine if the given date is within the upper bound of year end period
     *
     * @param runDate                 the given date
     * @param yearEndPeriodUpperBound the upper bound date of year end period
     * @param lastDayOfFiscalYear     the last day of the current fiscal year
     * @return true if the given date is within the upper bound of year end period; otherwise, false
     */
    protected boolean isInYearEndUpperBound(final Date runDate, final String yearEndPeriodUpperBound, final String lastDayOfFiscalYear) {
        final SimpleDateFormat sdf = new SimpleDateFormat("MMdd", Locale.US);
        final String today = sdf.format(runDate);

        final String month = StringUtils.mid(lastDayOfFiscalYear, 0, 2);
        final String date = StringUtils.mid(lastDayOfFiscalYear, 2, 2);

        final Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.MONTH, Integer.parseInt(month) - 1);
        calendar.set(Calendar.DATE, Integer.parseInt(date));
        calendar.add(Calendar.DATE, 1);
        final String firstDayOfNewFiscalYear = sdf.format(calendar.getTime());

        return today.compareTo(yearEndPeriodUpperBound) <= 0 && today.compareTo(firstDayOfNewFiscalYear) >= 0;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setOptionsService(final OptionsService optionsService) {
        this.optionsService = optionsService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setLaborLedgerBalanceService(final LaborLedgerBalanceService laborLedgerBalanceService) {
        this.laborLedgerBalanceService = laborLedgerBalanceService;
    }

    public void setLaborActualBalanceSummaryReportWriterService(
            final FiscalYearAwareReportWriterService laborActualBalanceSummaryReportWriterService) {
        this.laborActualBalanceSummaryReportWriterService = laborActualBalanceSummaryReportWriterService;
    }

    public void setLaborBudgetBalanceSummaryReportWriterService(
            final FiscalYearAwareReportWriterService laborBudgetBalanceSummaryReportWriterService) {
        this.laborBudgetBalanceSummaryReportWriterService = laborBudgetBalanceSummaryReportWriterService;
    }

    public void setLaborEncumbranceSummaryReportWriterService(
            final FiscalYearAwareReportWriterService laborEncumbranceSummaryReportWriterService) {
        this.laborEncumbranceSummaryReportWriterService = laborEncumbranceSummaryReportWriterService;
    }

    public void setLaborPosterOutputSummaryReportWriterService(
            final ReportWriterService laborPosterOutputSummaryReportWriterService) {
        this.laborPosterOutputSummaryReportWriterService = laborPosterOutputSummaryReportWriterService;
    }

    public void setBatchFileDirectoryName(final String batchFileDirectoryName) {
        this.batchFileDirectoryName = batchFileDirectoryName;
    }

}
