/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document.service;

import org.kuali.kfs.gl.businessobject.CorrectionChangeGroup;
import org.kuali.kfs.gl.document.GeneralLedgerCorrectionProcessDocument;
import org.kuali.kfs.gl.document.web.CorrectionDocumentEntryMetadata;
import org.kuali.kfs.kns.web.ui.Column;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.module.ld.document.LedgerCorrectionDocument;
import org.kuali.kfs.sys.batch.InitiateDirectory;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.List;

/**
 * Defines methods that must be implemented by classes providing a LaborCorrectionDocumentServiceImpl.
 */
public interface LaborCorrectionDocumentService extends InitiateDirectory {

    String CORRECTION_TYPE_MANUAL = "M";
    String CORRECTION_TYPE_CRITERIA = "C";
    String CORRECTION_TYPE_REMOVE_GROUP_FROM_PROCESSING = "R";

    /**
     * @param docId
     * @param i
     * @return return correctionChangeGroup with document number and Group Number
     */
    CorrectionChangeGroup findByDocumentNumberAndCorrectionChangeGroupNumber(String docId, int i);

    /**
     * @param docId
     * @param i
     * @return list of correctionChange with document number and Group Number
     */
    List findByDocumentHeaderIdAndCorrectionGroupNumber(String docId, int i);

    /**
     * @param docId
     * @param i
     * @return list of correctionCriteria with document number and Group Number
     */
    List findByDocumentNumberAndCorrectionGroupNumber(String docId, int i);

    /**
     * @param docId
     * @return laborCorrectionDocument with document number
     */
    LedgerCorrectionDocument findByCorrectionDocumentHeaderId(String docId);

    /**
     * @param docId
     * @return metadata to help render columns in the LLCP. Do not modify this list or the contents in this list.
     */
    List<Column> getTableRenderColumnMetadata(String docId);

    /**
     * This method persists an Iterator of input origin entries for a document that is in the initiated or saved state
     *
     * @param document an initiated or saved document
     * @param entries
     */
    void persistInputOriginEntriesForInitiatedOrSavedDocument(LedgerCorrectionDocument document,
            Iterator<LaborOriginEntry> entries);

    /**
     * @see #removePersistedInputOriginEntries(LedgerCorrectionDocument)
     */
    void removePersistedInputOriginEntries(LedgerCorrectionDocument document);

    /**
     * @see #removePersistedInputOriginEntries(LedgerCorrectionDocument)
     */
    void removePersistedInputOriginEntries(String docId);

    /**
     * Retrieves input origin entries that have been persisted for this document
     *
     * @param document       the document
     * @param abortThreshold if the file exceeds this number of rows, then null is returned.
     * @return the list, or null if there are too many origin entries
     * @throws RuntimeException several reasons, primarily relating to underlying persistence layer problems
     */
    List<LaborOriginEntry> retrievePersistedInputOriginEntries(LedgerCorrectionDocument document, int abortThreshold);

    /**
     * @param document
     * @return true if the system is storing input origin entries for this class. Note that this does not mean that
     *         there's at least one input origin entry record persisted for this document, but merely returns true if
     *         and only if the underlying persistence mechanism has a record of this document's origin entries. See
     *         the docs for the implementations of this method for more implementation specific details.
     */
    boolean areInputOriginEntriesPersisted(LedgerCorrectionDocument document);

    /**
     * Writes out the persisted input origin entries in an {@link OutputStream} in a flat file format
     *
     * @param document
     * @param out      an open and ready output stream
     * @throws IOException
     * @throws RuntimeException several reasons, including if the entries are not persisted
     */
    void writePersistedInputOriginEntriesToStream(LedgerCorrectionDocument document, OutputStream out) throws
            IOException;

    /**
     * This method persists an Iterator of input origin entries for a document that is in the initiated or saved state
     *
     * @param document an initiated or saved document
     * @param entries
     */
    void persistOutputLaborOriginEntriesForInitiatedOrSavedDocument(LedgerCorrectionDocument document,
            Iterator<LaborOriginEntry> entries);

    /**
     * @see #removePersistedOutputOriginEntries(LedgerCorrectionDocument)
     */
    void removePersistedOutputOriginEntries(LedgerCorrectionDocument document);

    /**
     * @see #removePersistedOutputOriginEntries(String)
     */
    void removePersistedOutputOriginEntries(String docId);

    /**
     * Retrieves output origin entries that have been persisted for this document
     *
     * @param document       the document
     * @param abortThreshold if the file exceeds this number of rows, then null is returned.
     * @return the list, or null if there are too many origin entries
     * @throws RuntimeException several reasons, primarily relating to underlying persistence layer problems
     */
    List<LaborOriginEntry> retrievePersistedOutputOriginEntries(LedgerCorrectionDocument document, int abortThreshold);

    /**
     * Retrieves input origin entries that have been persisted for this document in an iterator. Implementations of
     * this method may choose to implement this method in a way that consumes very little memory.
     *
     * @param document the document
     * @return the iterator
     * @throws RuntimeException several reasons, primarily relating to underlying persistence layer problems
     */
    Iterator<LaborOriginEntry> retrievePersistedInputOriginEntriesAsIterator(LedgerCorrectionDocument document);

    /**
     * Retrieves output origin entries that have been persisted for this document in an iterator. Implementations of
     * this method may choose to implement this method in a way that consumes very little memory.
     *
     * @param document the document
     * @return the iterator
     * @throws RuntimeException several reasons, primarily relating to underlying persistence layer problems
     */
    Iterator<LaborOriginEntry> retrievePersistedOutputOriginEntriesAsIterator(LedgerCorrectionDocument document);

    /**
     * @param document
     * @return true if the system is storing output origin entries for this class. Note that this does not mean that
     *         there's at least one output origin entry record persisted for this document, but merely returns true if
     *         and only if the underlying persistence mechanism has a record of this document's origin entries. See the
     *         docs for the implementations of this method for more implementation specific details.
     */
    boolean areOutputOriginEntriesPersisted(LedgerCorrectionDocument document);

    /**
     * Writes out the persisted output origin entries in an {@link OutputStream} in a flat file format\
     *
     * @param document
     * @param out      axn open and ready output stream
     * @throws IOException
     * @throws RuntimeException several reasons, including if the entries are not persisted
     */
    void writePersistedOutputOriginEntriesToStream(LedgerCorrectionDocument document, OutputStream out) throws
            IOException;

    /**
     * Saves the input and output origin entry groups for a document prior to saving the document
     *
     * @param document
     * @param correctionDocumentEntryMetadata
     */
    void persistOriginEntryGroupsForDocumentSave(LedgerCorrectionDocument document,
            CorrectionDocumentEntryMetadata correctionDocumentEntryMetadata);

    String generateOutputOriginEntryFileName(String docId);

    String createOutputFileForProcessing(String docId, java.util.Date today);

    String getBatchFileDirectoryName();

    String getLlcpDirectoryName();

    /**
     * Generate a text report for the given correction document
     *
     * @param document LLCP document to report on
     */
    void generateCorrectionReport(LedgerCorrectionDocument document);

    void aggregateCorrectionDocumentReports(GeneralLedgerCorrectionProcessDocument document);
}
