/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.service.ObjectCodeService;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.fp.document.YearEndDocument;
import org.kuali.kfs.fp.document.service.YearEndPendingEntryService;
import org.kuali.kfs.datadictionary.legacy.DataDictionaryService;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.LaborParameterConstants;
import org.kuali.kfs.module.ld.businessobject.BenefitsCalculation;
import org.kuali.kfs.module.ld.businessobject.ExpenseTransferAccountingLine;
import org.kuali.kfs.module.ld.businessobject.LaborLedgerPendingEntry;
import org.kuali.kfs.module.ld.document.LaborLedgerPostingDocument;
import org.kuali.kfs.module.ld.document.service.LaborPendingEntryConverterService;
import org.kuali.kfs.module.ld.service.LaborBenefitsCalculationService;
import org.kuali.kfs.module.ld.util.DebitCreditUtil;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySequenceHelper;
import org.kuali.kfs.sys.service.HomeOriginationService;
import org.kuali.kfs.sys.service.OptionsService;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;

import java.sql.Date;

public class LaborPendingEntryConverterServiceImpl implements LaborPendingEntryConverterService {

    protected HomeOriginationService homeOriginationService;
    protected LaborBenefitsCalculationService laborBenefitsCalculationService;
    protected OptionsService optionsService;
    protected ObjectCodeService objectCodeService;
    protected DataDictionaryService dataDictionaryService;
    protected DateTimeService dateTimeService;
    protected YearEndPendingEntryService yearEndPendingEntryService;
    protected ParameterService parameterService;

    @Override
    public LaborLedgerPendingEntry getBenefitA21PendingEntry(
            final LaborLedgerPostingDocument document,
            final ExpenseTransferAccountingLine accountingLine, final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final KualiDecimal benefitAmount, final String fringeBenefitObjectCode) {
        final LaborLedgerPendingEntry pendingEntry = getBenefitPendingEntry(document, accountingLine, sequenceHelper,
                benefitAmount, fringeBenefitObjectCode);

        pendingEntry.setFinancialBalanceTypeCode(KFSConstants.BALANCE_TYPE_A21);
        final String debitCreditCode = DebitCreditUtil.getReverseDebitCreditCode(pendingEntry.getTransactionDebitCreditCode());
        pendingEntry.setTransactionDebitCreditCode(debitCreditCode);

        return pendingEntry;
    }

    @Override
    public LaborLedgerPendingEntry getBenefitA21ReversalPendingEntry(
            final LaborLedgerPostingDocument document,
            final ExpenseTransferAccountingLine accountingLine, final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final KualiDecimal benefitAmount, final String fringeBenefitObjectCode) {
        final LaborLedgerPendingEntry pendingEntry = getBenefitA21PendingEntry(document, accountingLine, sequenceHelper,
                benefitAmount, fringeBenefitObjectCode);

        pendingEntry.setUniversityFiscalYear(accountingLine.getPayrollEndDateFiscalYear());
        pendingEntry.setUniversityFiscalPeriodCode(accountingLine.getPayrollEndDateFiscalPeriodCode());

        final String debitCreditCode = DebitCreditUtil.getReverseDebitCreditCode(pendingEntry.getTransactionDebitCreditCode());
        pendingEntry.setTransactionDebitCreditCode(debitCreditCode);

        return pendingEntry;
    }

    @Override
    public LaborLedgerPendingEntry getBenefitClearingPendingEntry(
            final LaborLedgerPostingDocument document,
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper, final String accountNumber, final String chartOfAccountsCode,
            final String benefitTypeCode, final KualiDecimal clearingAmount) {
        final LaborLedgerPendingEntry pendingEntry = getDefaultPendingEntry(document);

        pendingEntry.setChartOfAccountsCode(chartOfAccountsCode);
        pendingEntry.setAccountNumber(accountNumber);
        pendingEntry.setSubAccountNumber(KFSConstants.getDashSubAccountNumber());
        pendingEntry.setFinancialBalanceTypeCode(KFSConstants.BALANCE_TYPE_ACTUAL);

        final Integer fiscalYear = getOptionsService().getCurrentYearOptions().getUniversityFiscalYear();
        final BenefitsCalculation benefitsCalculation = getLaborBenefitsCalculationService().getBenefitsCalculation(
                fiscalYear, chartOfAccountsCode, benefitTypeCode);
        final String objectCode = benefitsCalculation.getPositionFringeBenefitObjectCode();
        pendingEntry.setFinancialObjectCode(objectCode);

        final ObjectCode oc = getObjectCodeService().getByPrimaryId(fiscalYear, chartOfAccountsCode, objectCode);
        pendingEntry.setFinancialObjectTypeCode(oc.getFinancialObjectTypeCode());

        pendingEntry.setFinancialSubObjectCode(KFSConstants.getDashFinancialSubObjectCode());
        pendingEntry.setTransactionLedgerEntrySequenceNumber(getNextSequenceNumber(sequenceHelper));

        final String debitCreditCode = DebitCreditUtil.getDebitCreditCode(clearingAmount, false);
        pendingEntry.setTransactionDebitCreditCode(debitCreditCode);
        pendingEntry.setTransactionLedgerEntryAmount(clearingAmount.abs());

        pendingEntry.setProjectCode(KFSConstants.getDashProjectCode());
        pendingEntry.setPositionNumber(LaborConstants.getDashPositionNumber());
        pendingEntry.setEmplid(LaborConstants.getDashEmplId());
        pendingEntry.setTransactionTotalHours(null);

        overrideEntryForYearEndIfNecessary(document, pendingEntry);

        return pendingEntry;
    }

    /**
     * Updates the given LLPE for year end documents
     *
     * @param document     the document to check if it is YearEnd
     * @param pendingEntry the pending entry to update
     */
    protected void overrideEntryForYearEndIfNecessary(
            final LaborLedgerPostingDocument document,
            final LaborLedgerPendingEntry pendingEntry) {
        // year end document should post to previous fiscal year and final period
        if (document instanceof YearEndDocument) {
            pendingEntry.setUniversityFiscalYear(getYearEndPendingEntryService().getPreviousFiscalYear());
            pendingEntry.setUniversityFiscalPeriodCode(getYearEndPendingEntryService().getFinalAccountingPeriod());
        }
    }

    @Override
    public LaborLedgerPendingEntry getBenefitPendingEntry(
            final LaborLedgerPostingDocument document,
            final ExpenseTransferAccountingLine accountingLine, final GeneralLedgerPendingEntrySequenceHelper sequenceHelper,
            final KualiDecimal benefitAmount, final String fringeBenefitObjectCode) {
        final LaborLedgerPendingEntry pendingEntry = getDefaultPendingEntry(document, accountingLine);

        // if account doesn't accept fringe charges, use reports to account
        if (!accountingLine.getAccount().isAccountsFringesBnftIndicator()) {
            pendingEntry.setChartOfAccountsCode(accountingLine.getAccount().getReportsToChartOfAccountsCode());
            pendingEntry.setAccountNumber(accountingLine.getAccount().getReportsToAccountNumber());
        }

        pendingEntry.setFinancialBalanceTypeCode(KFSConstants.BALANCE_TYPE_ACTUAL);
        pendingEntry.setFinancialObjectCode(pickValue(fringeBenefitObjectCode,
                KFSConstants.getDashFinancialObjectCode()));

        final ObjectCode fringeObjectCode = getObjectCodeService().getByPrimaryId(
                accountingLine.getPayrollEndDateFiscalYear(), accountingLine.getChartOfAccountsCode(),
                fringeBenefitObjectCode);
        pendingEntry.setFinancialObjectTypeCode(fringeObjectCode.getFinancialObjectTypeCode());

        setSubobjectCodeOnBenefitPendingEntry(accountingLine, pendingEntry);
        pendingEntry.setTransactionLedgerEntryAmount(benefitAmount.abs());
        pendingEntry.setPositionNumber(LaborConstants.getDashPositionNumber());
        pendingEntry.setEmplid(LaborConstants.getDashEmplId());
        pendingEntry.setTransactionLedgerEntrySequenceNumber(getNextSequenceNumber(sequenceHelper));

        // year end document should post to previous fiscal year and final period
        overrideEntryForYearEndIfNecessary(document, pendingEntry);

        return pendingEntry;
    }

    protected void setSubobjectCodeOnBenefitPendingEntry(
            final ExpenseTransferAccountingLine accountingLine,
            final LaborLedgerPendingEntry pendingEntry) {
        final boolean copySubobjectCode = getParameterService().getParameterValueAsBoolean(
                KfsParameterConstants.LABOR_DOCUMENT.class,
                LaborParameterConstants.COPY_SUB_OBJECT_TO_BENEFIT_ENTRIES, false);
        if (copySubobjectCode) {
            pendingEntry.setFinancialSubObjectCode(accountingLine.getFinancialSubObjectCode());
        } else {
            pendingEntry.setFinancialSubObjectCode(KFSConstants.getDashFinancialSubObjectCode());
        }
    }

    @Override
    public LaborLedgerPendingEntry getDefaultPendingEntry(
            final LaborLedgerPostingDocument document,
            final ExpenseTransferAccountingLine accountingLine) {
        final LaborLedgerPendingEntry pendingEntry = getDefaultPendingEntry(document);

        pendingEntry.setUniversityFiscalYear(accountingLine.getPostingYear());

        pendingEntry.setChartOfAccountsCode(accountingLine.getChartOfAccountsCode());
        pendingEntry.setAccountNumber(accountingLine.getAccountNumber());
        pendingEntry.setFinancialObjectCode(accountingLine.getFinancialObjectCode());

        final String subAccountNumber = pickValue(accountingLine.getSubAccountNumber(),
                KFSConstants.getDashSubAccountNumber());
        pendingEntry.setSubAccountNumber(subAccountNumber);

        final String subObjectCode = pickValue(accountingLine.getFinancialSubObjectCode(),
                KFSConstants.getDashFinancialSubObjectCode());
        pendingEntry.setFinancialSubObjectCode(subObjectCode);

        final String projectCode = pickValue(accountingLine.getProjectCode(), KFSConstants.getDashProjectCode());
        pendingEntry.setProjectCode(projectCode);

        accountingLine.refreshReferenceObject(KFSPropertyConstants.OBJECT_CODE);
        final String objectTypeCode = accountingLine.getObjectCode().getFinancialObjectTypeCode();
        pendingEntry.setFinancialObjectTypeCode(objectTypeCode);

        final KualiDecimal transactionAmount = accountingLine.getAmount();
        final String debitCreditCode = DebitCreditUtil.getDebitCreditCodeForExpenseDocument(accountingLine);
        pendingEntry.setTransactionDebitCreditCode(debitCreditCode);
        pendingEntry.setTransactionLedgerEntryAmount(transactionAmount.abs());

        pendingEntry.setPositionNumber(accountingLine.getPositionNumber());
        pendingEntry.setEmplid(accountingLine.getEmplid());
        pendingEntry.setPayrollEndDateFiscalYear(accountingLine.getPayrollEndDateFiscalYear());
        pendingEntry.setPayrollEndDateFiscalPeriodCode(accountingLine.getPayrollEndDateFiscalPeriodCode());
        pendingEntry.setTransactionTotalHours(accountingLine.getPayrollTotalHours());
        pendingEntry.setOrganizationReferenceId(accountingLine.getOrganizationReferenceId());

        return pendingEntry;
    }

    @Override
    public LaborLedgerPendingEntry getDefaultPendingEntry(final LaborLedgerPostingDocument document) {
        final LaborLedgerPendingEntry pendingEntry = getSimpleDefaultPendingEntry();
        final DocumentHeader documentHeader = document.getDocumentHeader();

        final String documentTypeCode = getDataDictionaryService().getDocumentTypeNameByClass(document.getClass());
        pendingEntry.setFinancialDocumentTypeCode(documentTypeCode);

        pendingEntry.setDocumentNumber(documentHeader.getDocumentNumber());
        pendingEntry.setTransactionLedgerEntryDescription(documentHeader.getDocumentDescription());
        pendingEntry.setOrganizationDocumentNumber(documentHeader.getOrganizationDocumentNumber());

        return pendingEntry;
    }

    @Override
    public LaborLedgerPendingEntry getExpenseA21PendingEntry(
            final LaborLedgerPostingDocument document,
            final ExpenseTransferAccountingLine accountingLine, final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        final LaborLedgerPendingEntry pendingEntry = getExpensePendingEntry(document, accountingLine, sequenceHelper);

        pendingEntry.setFinancialBalanceTypeCode(KFSConstants.BALANCE_TYPE_A21);
        final String debitCreditCode = DebitCreditUtil.getReverseDebitCreditCode(
                pendingEntry.getTransactionDebitCreditCode());
        pendingEntry.setTransactionDebitCreditCode(debitCreditCode);

        return pendingEntry;
    }

    @Override
    public LaborLedgerPendingEntry getExpenseA21ReversalPendingEntry(
            final LaborLedgerPostingDocument document,
            final ExpenseTransferAccountingLine accountingLine, final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        final LaborLedgerPendingEntry pendingEntry = getExpenseA21PendingEntry(document, accountingLine, sequenceHelper);

        pendingEntry.setUniversityFiscalYear(accountingLine.getPayrollEndDateFiscalYear());
        pendingEntry.setUniversityFiscalPeriodCode(accountingLine.getPayrollEndDateFiscalPeriodCode());

        final String debitCreditCode = DebitCreditUtil.getReverseDebitCreditCode(
                pendingEntry.getTransactionDebitCreditCode());
        pendingEntry.setTransactionDebitCreditCode(debitCreditCode);

        return pendingEntry;
    }

    @Override
    public LaborLedgerPendingEntry getExpensePendingEntry(
            final LaborLedgerPostingDocument document,
            final ExpenseTransferAccountingLine accountingLine, final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        final LaborLedgerPendingEntry pendingEntry = getDefaultPendingEntry(document, accountingLine);

        pendingEntry.setFinancialBalanceTypeCode(KFSConstants.BALANCE_TYPE_ACTUAL);
        pendingEntry.setTransactionLedgerEntrySequenceNumber(getNextSequenceNumber(sequenceHelper));

        // year end document should post to previous fiscal year and final period
        overrideEntryForYearEndIfNecessary(document, pendingEntry);

        return pendingEntry;
    }

    @Override
    public LaborLedgerPendingEntry getSimpleDefaultPendingEntry() {
        final LaborLedgerPendingEntry pendingEntry = new LaborLedgerPendingEntry();

        pendingEntry.setUniversityFiscalYear(null);
        pendingEntry.setUniversityFiscalPeriodCode(null);

        final String originationCode = getHomeOriginationService().getHomeOrigination().getFinSystemHomeOriginationCode();
        pendingEntry.setFinancialSystemOriginationCode(originationCode);

        final Date transactionDate = getDateTimeService().getCurrentSqlDate();
        pendingEntry.setTransactionDate(transactionDate);

        pendingEntry.setFinancialDocumentReversalDate(null);
        pendingEntry.setReferenceFinancialSystemOriginationCode(null);
        pendingEntry.setReferenceFinancialDocumentNumber(null);
        pendingEntry.setReferenceFinancialDocumentTypeCode(null);

        return pendingEntry;
    }

    /**
     * Pick one from target and backup values based on the availability of target value
     *
     * @param targetValue the given target value
     * @param backupValue the backup value of the target value
     * @return target value if it is not null; otherwise, return its backup
     */
    protected String pickValue(final String targetValue, final String backupValue) {
        return StringUtils.isNotBlank(targetValue) ? targetValue : backupValue;
    }

    /**
     * This method gets the next sequence number and increments.
     *
     * @param sequenceHelper the given sequence helper
     * @return the next sequence number and increments
     */
    protected Integer getNextSequenceNumber(final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        final Integer nextSequenceNumber = sequenceHelper.getSequenceCounter();
        sequenceHelper.increment();
        return nextSequenceNumber;
    }

    public HomeOriginationService getHomeOriginationService() {
        return homeOriginationService;
    }

    public void setHomeOriginationService(final HomeOriginationService homeOriginationService) {
        this.homeOriginationService = homeOriginationService;
    }

    public LaborBenefitsCalculationService getLaborBenefitsCalculationService() {
        return laborBenefitsCalculationService;
    }

    public void setLaborBenefitsCalculationService(final LaborBenefitsCalculationService laborBenefitsCalculationService) {
        this.laborBenefitsCalculationService = laborBenefitsCalculationService;
    }

    public OptionsService getOptionsService() {
        return optionsService;
    }

    public void setOptionsService(final OptionsService optionsService) {
        this.optionsService = optionsService;
    }

    public ObjectCodeService getObjectCodeService() {
        return objectCodeService;
    }

    public void setObjectCodeService(final ObjectCodeService objectCodeService) {
        this.objectCodeService = objectCodeService;
    }

    public DataDictionaryService getDataDictionaryService() {
        return dataDictionaryService;
    }

    public void setDataDictionaryService(final DataDictionaryService dataDictionaryService) {
        this.dataDictionaryService = dataDictionaryService;
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public YearEndPendingEntryService getYearEndPendingEntryService() {
        return yearEndPendingEntryService;
    }

    public void setYearEndPendingEntryService(final YearEndPendingEntryService yearEndPendingEntryService) {
        this.yearEndPendingEntryService = yearEndPendingEntryService;
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }
}
