/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document.web.struts;

import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.module.ld.LaborPropertyConstants;
import org.kuali.kfs.module.ld.businessobject.LaborAccountingLineOverride;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.businessobject.SourceAccountingLine;
import org.kuali.kfs.sys.businessobject.TargetAccountingLine;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.OptionsService;

import java.util.Map;

/**
 * Base Struts Action Form for all expense transfer documents.
 */
public abstract class ExpenseTransferDocumentFormBase extends LaborDocumentFormBase implements
        MultipleValueLookupBroker {

    // Indicates which result set we are using when refreshing/returning from a multi-value lookup.
    protected String lookupResultsSequenceNumber;
    // Type of result returned by the multi-value lookup. ?to be persisted in the lookup results service instead?
    protected String lookupResultsBOClassName;
    // The name of the collection looked up (by a multiple value lookup)
    protected String lookedUpCollectionName;
    protected Integer universityFiscalYear;

    public ExpenseTransferDocumentFormBase() {
        super();
        setUniversityFiscalYear(SpringContext.getBean(OptionsService.class).getCurrentYearOptions()
                .getUniversityFiscalYear());
    }

    @Override
    public void addRequiredNonEditableProperties() {
        super.addRequiredNonEditableProperties();
        registerRequiredNonEditableProperty(KRADConstants.LOOKUP_RESULTS_SEQUENCE_NUMBER);
    }

    @Override
    public String getLookupResultsSequenceNumber() {
        return lookupResultsSequenceNumber;
    }

    @Override
    public void setLookupResultsSequenceNumber(final String lookupResultsSequenceNumber) {
        this.lookupResultsSequenceNumber = lookupResultsSequenceNumber;
    }

    @Override
    public String getLookupResultsBOClassName() {
        return lookupResultsBOClassName;
    }

    @Override
    public void setLookupResultsBOClassName(final String lookupResultsBOClassName) {
        this.lookupResultsBOClassName = lookupResultsBOClassName;
    }

    @Override
    public String getLookedUpCollectionName() {
        return lookedUpCollectionName;
    }

    @Override
    public void setLookedUpCollectionName(final String lookedUpCollectionName) {
        this.lookedUpCollectionName = lookedUpCollectionName;
    }

    public Integer getUniversityFiscalYear() {
        return universityFiscalYear;
    }

    public void setUniversityFiscalYear(final Integer universityFiscalYear) {
        this.universityFiscalYear = universityFiscalYear;
    }

    /**
     * Populate the search fields with the last search criteria
     */
    public abstract void populateSearchFields();

    /**
     * Gets forced read only source fields
     *
     * @return a Map of read only source fields
     */
    public Map getForcedReadOnlySourceFields() {
        final Map map = super.getForcedReadOnlyFields();
        map.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, Boolean.TRUE);
        map.put(KFSPropertyConstants.ACCOUNT_NUMBER, Boolean.TRUE);
        map.put(KFSPropertyConstants.SUB_ACCOUNT_NUMBER, Boolean.TRUE);
        map.put(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, Boolean.TRUE);
        map.put(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE, Boolean.TRUE);
        map.put(KFSPropertyConstants.PROJECT_CODE, Boolean.TRUE);
        map.put(KFSPropertyConstants.ORGANIZATION_REFERENCE_ID, Boolean.TRUE);
        map.put(KFSPropertyConstants.POSITION_NUMBER, Boolean.TRUE);
        map.put(LaborPropertyConstants.PAYROLL_END_DATE_FISCAL_PERIOD_CODE, Boolean.TRUE);
        map.put(LaborPropertyConstants.PAYROLL_END_DATE_FISCAL_YEAR, Boolean.TRUE);
        return map;
    }

    /**
     * @return forced read only target fields (i.e read only fields plus payroll end date fiscal period code and
     * payroll end date fiscal year)
     *
     * @see org.kuali.kfs.kns.web.struts.form.KualiTransactionalDocumentFormBase#getForcedReadOnlyFields()
     */
    public Map getForcedReadOnlyTargetFields() {
        final Map map = super.getForcedReadOnlyFields();
        map.put(LaborPropertyConstants.PAYROLL_END_DATE_FISCAL_PERIOD_CODE, Boolean.TRUE);
        map.put(LaborPropertyConstants.PAYROLL_END_DATE_FISCAL_YEAR, Boolean.TRUE);
        return map;
    }

    /**
     * Deal with the labor-specific override code to the given source accounting line
     */
    @Override
    public void populateSourceAccountingLine(
            final SourceAccountingLine sourceLine, final String accountingLinePropertyName,
            final Map parameterMap) {
        super.populateSourceAccountingLine(sourceLine, accountingLinePropertyName, parameterMap);
        LaborAccountingLineOverride.populateFromInput(sourceLine);
    }

    /**
     * Deal with the labor-specific override code to the given target accounting line
     */
    @Override
    public void populateTargetAccountingLine(
            final TargetAccountingLine targetLine, final String accountingLinePropertyName,
            final Map parameterMap) {
        super.populateTargetAccountingLine(targetLine, accountingLinePropertyName, parameterMap);
        LaborAccountingLineOverride.populateFromInput(targetLine);
    }

    @Override
    protected void repopulateOverrides(final AccountingLine line, final String accountingLinePropertyName, final Map parameterMap) {
        super.repopulateOverrides(line, accountingLinePropertyName, parameterMap);
        LaborAccountingLineOverride.determineNeededOverrides(getFinancialDocument(), line);
        if (line.getNonFringeAccountOverrideNeeded()) {
            if (parameterMap.containsKey(accountingLinePropertyName + ".nonFringeAccountOverride.present")) {
                line.setNonFringeAccountOverride(parameterMap.containsKey(accountingLinePropertyName +
                        ".nonFringeAccountOverride"));
            }
        } else {
            line.setNonFringeAccountOverride(false);
        }
    }

}
