/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.gl.businessobject.Entry;
import org.kuali.kfs.integration.ld.LaborLedgerBalance;
import org.kuali.kfs.integration.ld.LaborLedgerExpenseTransferAccountingLine;
import org.kuali.kfs.integration.ld.LaborLedgerObject;
import org.kuali.kfs.integration.ld.LaborLedgerPositionObjectBenefit;
import org.kuali.kfs.integration.ld.LaborLedgerPositionObjectGroup;
import org.kuali.kfs.integration.ld.LaborModuleService;
import org.kuali.kfs.kew.api.KewApiConstants;
import org.kuali.kfs.kew.api.document.WorkflowDocumentService;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.lookup.HtmlData.AnchorHtmlData;
import org.kuali.kfs.krad.bo.AdHocRoutePerson;
import org.kuali.kfs.krad.bo.AdHocRouteRecipient;
import org.kuali.kfs.krad.rules.rule.event.SaveDocumentEvent;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.service.SessionDocumentService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.ld.LaborParameterConstants;
import org.kuali.kfs.module.ld.LaborPropertyConstants;
import org.kuali.kfs.module.ld.businessobject.LaborLedgerPendingEntry;
import org.kuali.kfs.module.ld.businessobject.LedgerBalance;
import org.kuali.kfs.module.ld.businessobject.LedgerEntryGLSummary;
import org.kuali.kfs.module.ld.document.SalaryExpenseTransferDocument;
import org.kuali.kfs.module.ld.service.LaborBenefitsCalculationService;
import org.kuali.kfs.module.ld.service.LaborLedgerBalanceService;
import org.kuali.kfs.module.ld.service.LaborLedgerEntryService;
import org.kuali.kfs.module.ld.service.LaborLedgerPendingEntryService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.businessobject.AccountingLineOverride;
import org.kuali.kfs.sys.businessobject.AccountingLineOverride.COMPONENT;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.document.AccountingDocument;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This implements the service methods that may be used by outside of labor module
 */
@Transactional
public class LaborModuleServiceImpl implements LaborModuleService {

    private static final Logger LOG = LogManager.getLogger();

    private static final String GL_LABOR_ENTRY_SUMMARIZATION_INQUIRY_BASE_URL =
            "laborGLLaborEntrySummarizationInquiry.do";
    private static final String GL_LABOR_ENTRY_SUMMARIZATION_INQUIRY_METHOD = "viewResults";

    private BusinessObjectService businessObjectService;
    private DocumentService documentService;
    private KualiModuleService kualiModuleService;
    private LaborBenefitsCalculationService laborBenefitsCalculationService;
    private LaborLedgerBalanceService laborLedgerBalanceService;
    private LaborLedgerEntryService laborLedgerEntryService;
    private LaborLedgerPendingEntryService laborLedgerPendingEntryService;
    private ParameterService parameterService;
    private SessionDocumentService sessionDocumentService;
    private WorkflowDocumentService workflowDocumentService;

    @Override
    public KualiDecimal calculateFringeBenefit(
            final Integer fiscalYear, final String chartCode, final String objectCode,
            final KualiDecimal salaryAmount, final String accountNumber, final String subAccountNumber) {
        return laborBenefitsCalculationService.calculateFringeBenefit(fiscalYear, chartCode, objectCode,
                salaryAmount, accountNumber, subAccountNumber);
    }

    @Override
    public void createAndBlankApproveSalaryExpenseTransferDocument(
            final String documentDescription, final String explanation,
            final String annotation, final List<String> adHocRecipients,
            final List<LaborLedgerExpenseTransferAccountingLine> sourceAccountingLines,
            final List<LaborLedgerExpenseTransferAccountingLine> targetAccountingLines) {
        LOG.debug("createSalaryExpenseTransferDocument() start");

        if (sourceAccountingLines == null || sourceAccountingLines.isEmpty()) {
            LOG.info("Cannot create a salary expense document when the given source accounting line is empty.");
            return;
        }

        if (targetAccountingLines == null || targetAccountingLines.isEmpty()) {
            LOG.info("Cannot create a salary expense document when the given target accounting line is empty.");
            return;
        }

        final SalaryExpenseTransferDocument document = (SalaryExpenseTransferDocument) documentService
                .getNewDocument(SalaryExpenseTransferDocument.class);

        document.setEmplid(sourceAccountingLines.get(0).getEmplid());
        document.setSourceAccountingLines(sourceAccountingLines);
        document.setTargetAccountingLines(targetAccountingLines);

        final DocumentHeader documentHeader = document.getDocumentHeader();
        documentHeader.setDocumentDescription(documentDescription);
        documentHeader.setExplanation(explanation);

        document.prepareForSave(new SaveDocumentEvent(document));
        document.populateDocumentForRouting();

        final String documentTitle = document.getDocumentTitle();
        if (StringUtils.isNotBlank(documentTitle)) {
            document.getDocumentHeader().getWorkflowDocument().setTitle(documentTitle);
        }

        final String organizationDocumentNumber = document.getDocumentHeader().getOrganizationDocumentNumber();
        if (StringUtils.isNotBlank(organizationDocumentNumber)) {
            document.getDocumentHeader().getWorkflowDocument().setApplicationDocumentId(organizationDocumentNumber);
        }

        businessObjectService.save(document);

        final List<AdHocRouteRecipient> adHocRecipientList = new ArrayList<>();

        for (final String adHocRouteRecipient : adHocRecipients) {
            adHocRecipientList.add(buildApprovePersonRecipient(adHocRouteRecipient));
        }

        // blanket approve salary expense transfer doc bypassing all rules
        workflowDocumentService.blanketApprove(document.getDocumentHeader().getWorkflowDocument(), annotation,
                adHocRecipientList);
        sessionDocumentService.addDocumentToUserSession(GlobalVariables.getUserSession(),
                document.getDocumentHeader().getWorkflowDocument());

    }

    protected AdHocRouteRecipient buildApprovePersonRecipient(final String userId) {
        final AdHocRouteRecipient adHocRouteRecipient = new AdHocRoutePerson();
        adHocRouteRecipient.setActionRequested(KewApiConstants.ACTION_REQUEST_APPROVE_REQ);
        adHocRouteRecipient.setId(userId);
        return adHocRouteRecipient;
    }

    @Override
    public int countPendingSalaryExpenseTransfer(final String emplid) {
        final Map<String, Object> positiveFieldValues = new HashMap<>();
        positiveFieldValues.put(KFSPropertyConstants.EMPLID, emplid);
        positiveFieldValues.put(KFSPropertyConstants.FINANCIAL_DOCUMENT_TYPE_CODE,
                KFSConstants.FinancialDocumentTypeCodes.SALARY_EXPENSE_TRANSFER);

        final List<String> approvedCodes = Arrays.asList(KFSConstants.PENDING_ENTRY_APPROVED_STATUS_CODE.APPROVED,
                KFSConstants.PENDING_ENTRY_APPROVED_STATUS_CODE.PROCESSED);
        final Map<String, Object> negativeFieldValues = new HashMap<>();
        negativeFieldValues.put(KFSPropertyConstants.FINANCIAL_DOCUMENT_APPROVED_CODE, approvedCodes);

        return businessObjectService.countMatching(LaborLedgerPendingEntry.class, positiveFieldValues,
                negativeFieldValues);
    }

    @Override
    public List<String> findEmployeesWithPayType(
            final Map<Integer, Set<String>> payPeriods, final List<String> balanceTypes,
            final Map<String, Set<String>> earnCodePayGroupMap) {
        return laborLedgerEntryService.findEmployeesWithPayType(payPeriods, balanceTypes, earnCodePayGroupMap);
    }

    @Override
    public boolean isEmployeeWithPayType(
            final String emplid, final Map<Integer, Set<String>> payPeriods,
            final List<String> balanceTypes, final Map<String, Set<String>> earnCodePayGroupMap) {
        return laborLedgerEntryService.isEmployeeWithPayType(emplid, payPeriods, balanceTypes,
                earnCodePayGroupMap);
    }

    @Override
    public Collection<LaborLedgerBalance> findLedgerBalances(
            final Map<String, Collection<String>> fieldValues,
            final Map<String, Collection<String>> excludedFieldValues, final Set<Integer> fiscalYears, final List<String> balanceTypes,
            final List<String> positionObjectGroupCodes) {

        final Map<String, List<String>> excludedFieldValueList = new HashMap<>();
        for (final Map.Entry<String, Collection<String>> e : excludedFieldValues.entrySet()) {
            // convert collection to list
            final List<String> list = new ArrayList<>(e.getValue());
            Collections.sort(list);
            excludedFieldValueList.put(e.getKey(), list);
        }
        final Map<String, List<String>> fieldValueList = new HashMap<>();
        for (final Map.Entry<String, Collection<String>> e : fieldValues.entrySet()) {
            // convert collection to list
            final List<String> list = new ArrayList<>(e.getValue());
            Collections.sort(list);
            fieldValueList.put(e.getKey(), list);
        }
        final Collection<LedgerBalance> ledgerBalances = laborLedgerBalanceService.findLedgerBalances(fieldValueList,
                excludedFieldValueList, fiscalYears, balanceTypes, positionObjectGroupCodes);
        return new ArrayList<>(ledgerBalances);
    }

    public LaborLedgerPositionObjectGroup getLaborLedgerPositionObjectGroup(final String positionObjectGroupCode) {
        final Map<String, Object> primaryKeys = new HashMap<>();
        primaryKeys.put(LaborPropertyConstants.POSITION_OBJECT_GROUP_CODE, positionObjectGroupCode);

        return kualiModuleService.getResponsibleModuleService(LaborLedgerPositionObjectGroup.class)
                .getExternalizableBusinessObject(LaborLedgerPositionObjectGroup.class, primaryKeys);
    }

    @Override
    public LaborLedgerObject retrieveLaborLedgerObject(
            final Integer fiscalYear, final String chartOfAccountsCode,
            final String objectCode) {
        final Map<String, Object> searchCriteria = new HashMap<>();
        searchCriteria.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);
        searchCriteria.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        searchCriteria.put(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, objectCode);

        return kualiModuleService.getResponsibleModuleService(LaborLedgerObject.class)
                .getExternalizableBusinessObject(LaborLedgerObject.class, searchCriteria);
    }

    @Override
    public LaborLedgerObject retrieveLaborLedgerObject(final ObjectCode financialObject) {
        if (financialObject == null) {
            throw new IllegalArgumentException("The given financial object cannot be null.");
        }

        final Integer fiscalYear = financialObject.getUniversityFiscalYear();
        final String chartOfAccountsCode = financialObject.getChartOfAccountsCode();
        final String financialObjectCode = financialObject.getFinancialObjectCode();

        return retrieveLaborLedgerObject(fiscalYear, chartOfAccountsCode, financialObjectCode);
    }

    @Override
    public boolean hasPendingLaborLedgerEntry(final String chartOfAccountsCode, final String accountNumber) {
        return laborLedgerPendingEntryService.hasPendingLaborLedgerEntry(chartOfAccountsCode, accountNumber);
    }

    @Override
    public List<LaborLedgerPositionObjectBenefit> retrieveActiveLaborPositionObjectBenefits(
            final Integer fiscalYear,
            final String chartOfAccountsCode, final String objectCode) {
        final Map<String, Object> searchCriteria = new HashMap<>();

        searchCriteria.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);
        searchCriteria.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        searchCriteria.put(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, objectCode);
        searchCriteria.put(KFSPropertyConstants.ACTIVE, KFSConstants.ACTIVE_INDICATOR);

        return kualiModuleService.getResponsibleModuleService(LaborLedgerPositionObjectBenefit.class)
                .getExternalizableBusinessObjectsList(LaborLedgerPositionObjectBenefit.class, searchCriteria);
    }

    @Override
    public boolean hasFringeBenefitProducingObjectCodes(
            final Integer fiscalYear, final String chartOfAccountsCode,
            final String financialObjectCode) {
        final List<LaborLedgerPositionObjectBenefit> objectBenefits = retrieveActiveLaborPositionObjectBenefits(
                fiscalYear, chartOfAccountsCode, financialObjectCode);
        return objectBenefits != null && !objectBenefits.isEmpty();
    }

    @Override
    public List<LaborLedgerPositionObjectBenefit> retrieveLaborPositionObjectBenefits(
            final Integer fiscalYear,
                                                                                      final String chartOfAccountsCode,
                                                                                      final String objectCode) {
        final Map<String, Object> searchCriteria = new HashMap<>();

        searchCriteria.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);
        searchCriteria.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        searchCriteria.put(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, objectCode);
        return kualiModuleService.getResponsibleModuleService(LaborLedgerPositionObjectBenefit.class)
                .getExternalizableBusinessObjectsList(LaborLedgerPositionObjectBenefit.class, searchCriteria);
    }

    @Override
    public Collection<String> getLaborLedgerGLOriginCodes() {
        return parameterService.getParameterValuesAsString(Entry.class,
                LaborParameterConstants.LABOR_ORIGINATION_CODES
        );
    }

    /**
     * Builds the url for the given GL entry to go to inquiry screen for related LD entries
     */
    @Override
    public HtmlData getInquiryUrlForGeneralLedgerEntryDocumentNumber(final Entry entry) {
        final Map<String, String> params = new HashMap<>();
        params.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, GL_LABOR_ENTRY_SUMMARIZATION_INQUIRY_METHOD);
        params.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, entry.getUniversityFiscalYear().toString());
        params.put(KFSPropertyConstants.UNIVERSITY_FISCAL_PERIOD_CODE, entry.getUniversityFiscalPeriodCode());
        params.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, entry.getChartOfAccountsCode());
        params.put(KFSPropertyConstants.ACCOUNT_NUMBER, entry.getAccountNumber());
        params.put(KFSPropertyConstants.SUB_ACCOUNT_NUMBER, entry.getSubAccountNumber());
        params.put(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, entry.getFinancialObjectCode());
        params.put(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE, entry.getFinancialSubObjectCode());
        params.put(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, entry.getFinancialBalanceTypeCode());
        params.put(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE, entry.getFinancialObjectTypeCode());
        params.put(KFSPropertyConstants.FINANCIAL_DOCUMENT_TYPE_CODE, entry.getFinancialDocumentTypeCode());
        params.put(KFSPropertyConstants.FINANCIAL_SYSTEM_ORIGINATION_CODE, entry.getFinancialSystemOriginationCode());
        params.put(KFSPropertyConstants.DOCUMENT_NUMBER, entry.getDocumentNumber());
        params.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, LedgerEntryGLSummary.class.getName());
        return new AnchorHtmlData(UrlFactory.parameterizeUrl(GL_LABOR_ENTRY_SUMMARIZATION_INQUIRY_BASE_URL, params),
                entry.getDocumentNumber());
    }

    @Override
    public String getBenefitRateCategoryCode(
            final String chartOfAccountsCode, final String accountNumber,
            final String subAccountNumber) {
        return laborBenefitsCalculationService.getBenefitRateCategoryCode(chartOfAccountsCode, accountNumber,
                subAccountNumber);
    }

    @Override
    public String getCostSharingSourceAccountNumber() {
        return laborBenefitsCalculationService.getCostSharingSourceAccountNumber();
    }

    @Override
    public String getCostSharingSourceSubAccountNumber() {
        return laborBenefitsCalculationService.getCostSharingSourceSubAccountNumber();
    }

    @Override
    public String getCostSharingSourceChartOfAccountsCode() {
        return laborBenefitsCalculationService.getCostSharingSourceAccountChartOfAccountsCode();
    }

    @Override
    public AccountingLineOverride determineNeededOverrides(final AccountingDocument document, final AccountingLine line) {
        boolean isDocumentFinalOrProcessed = false;
        if (ObjectUtils.isNotNull(document)) {
            final AccountingDocument accountingDocument = document;
            isDocumentFinalOrProcessed = accountingDocument.isDocumentFinalOrProcessed();
        }
        final Set<Integer> neededOverrideComponents = new HashSet<>();
        if (AccountingLineOverride.needsExpiredAccountOverride(line, isDocumentFinalOrProcessed)) {
            neededOverrideComponents.add(COMPONENT.EXPIRED_ACCOUNT);
        }
        if (AccountingLineOverride.needsObjectBudgetOverride(line.getAccount(), line.getObjectCode())) {
            neededOverrideComponents.add(COMPONENT.NON_BUDGETED_OBJECT);
        }
        if (AccountingLineOverride.needsNonFringAccountOverride(line.getAccount())) {
            neededOverrideComponents.add(COMPONENT.NON_FRINGE_ACCOUNT_USED);
        }
        final Integer[] inputComponentArray = neededOverrideComponents.toArray(new Integer[neededOverrideComponents.size()]);

        return AccountingLineOverride.valueOf(inputComponentArray);
    }

    @Override
    @Deprecated
    public AccountingLineOverride determineNeededOverrides(final AccountingLine line) {
        final Set<Integer> neededOverrideComponents = new HashSet<>();
        if (AccountingLineOverride.needsExpiredAccountOverride(line.getAccount())) {
            neededOverrideComponents.add(COMPONENT.EXPIRED_ACCOUNT);
        }
        if (AccountingLineOverride.needsObjectBudgetOverride(line.getAccount(), line.getObjectCode())) {
            neededOverrideComponents.add(COMPONENT.NON_BUDGETED_OBJECT);
        }
        if (AccountingLineOverride.needsNonFringAccountOverride(line.getAccount())) {
            neededOverrideComponents.add(COMPONENT.NON_FRINGE_ACCOUNT_USED);
        }
        final Integer[] inputComponentArray = neededOverrideComponents.toArray(new Integer[neededOverrideComponents.size()]);

        return AccountingLineOverride.valueOf(inputComponentArray);
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setKualiModuleService(final KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }

    public void setLaborBenefitsCalculationService(final LaborBenefitsCalculationService laborBenefitsCalculationService) {
        this.laborBenefitsCalculationService = laborBenefitsCalculationService;
    }

    public void setLaborLedgerBalanceService(final LaborLedgerBalanceService laborLedgerBalanceService) {
        this.laborLedgerBalanceService = laborLedgerBalanceService;
    }

    public void setLaborLedgerEntryService(final LaborLedgerEntryService laborLedgerEntryService) {
        this.laborLedgerEntryService = laborLedgerEntryService;
    }

    public void setLaborLedgerPendingEntryService(final LaborLedgerPendingEntryService laborLedgerPendingEntryService) {
        this.laborLedgerPendingEntryService = laborLedgerPendingEntryService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setSessionDocumentService(final SessionDocumentService sessionDocumentService) {
        this.sessionDocumentService = sessionDocumentService;
    }

    public void setWorkflowDocumentService(final WorkflowDocumentService workflowDocumentService) {
        this.workflowDocumentService = workflowDocumentService;
    }
}
