/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.service.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.gl.businessobject.OriginEntryStatistics;
import org.kuali.kfs.gl.service.impl.OriginEntryGroupServiceImpl;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.module.ld.service.LaborOriginEntryService;
import org.kuali.kfs.module.ld.util.LaborOriginEntryFileIterator;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.Message;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.springframework.transaction.annotation.Transactional;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

@Transactional
public class LaborOriginEntryServiceImpl extends OriginEntryGroupServiceImpl implements LaborOriginEntryService {

    private static final Logger LOG = LogManager.getLogger();

    @Override
    public OriginEntryStatistics getStatistics(final String fileName) {
        LOG.debug("getStatistics() started");
        final OriginEntryStatistics oes = new OriginEntryStatistics();
        final KualiDecimal totalCredit = KualiDecimal.ZERO;
        final KualiDecimal totalDebit = KualiDecimal.ZERO;
        Integer rowCount = 0;
        final FileReader INPUT_FILE;
        final BufferedReader INPUT_FILE_br;
        try {
            INPUT_FILE = new FileReader(fileName, StandardCharsets.UTF_8);
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }
        INPUT_FILE_br = new BufferedReader(INPUT_FILE);

        try {
            String currentLine = INPUT_FILE_br.readLine();
            while (currentLine != null) {
                KualiDecimal amount = KualiDecimal.ZERO;
                final String amountString = currentLine.substring(109, 126);
                if (!amountString.trim().equals(GeneralLedgerConstants.EMPTY_CODE)) {
                    try {
                        amount = new KualiDecimal(amountString.trim());

                        // TODO:- Check with FIS (Row count should be all rows?)
                        rowCount++;
                    } catch (final NumberFormatException e) {
                        LOG.warn("getStatistics() - amount={} is not a valid numeric format", amountString);
                    }
                } else {
                    amount = KualiDecimal.ZERO;
                }
                final String debitOrCreditCode = currentLine.substring(126, 127);

                if (debitOrCreditCode.equals(KFSConstants.GL_CREDIT_CODE)) {
                    totalCredit.add(amount);
                } else if (debitOrCreditCode.equals(KFSConstants.GL_DEBIT_CODE)) {
                    totalDebit.add(amount);
                }
                currentLine = INPUT_FILE_br.readLine();
            }
            INPUT_FILE_br.close();
        } catch (final IOException e) {
            // FIXME: do whatever should be done here
            throw new RuntimeException(e);
        }

        oes.setCreditTotalAmount(totalCredit);
        oes.setDebitTotalAmount(totalDebit);
        oes.setRowCount(rowCount);

        return oes;
    }

    @Override
    public Map getEntriesByGroupIdWithPath(final String fileNameWithPath, final List<LaborOriginEntry> originEntryList) {
        final FileReader INPUT_GLE_FILE;
        final BufferedReader INPUT_GLE_FILE_br;
        try {
            INPUT_GLE_FILE = new FileReader(fileNameWithPath, StandardCharsets.UTF_8);
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }
        INPUT_GLE_FILE_br = new BufferedReader(INPUT_GLE_FILE);

        final Map returnMessageMap = getEntriesByBufferedReader(INPUT_GLE_FILE_br, originEntryList);

        try {
            INPUT_GLE_FILE_br.close();
            INPUT_GLE_FILE.close();
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }

        return returnMessageMap;
    }

    @Override
    public Map getEntriesByBufferedReader(final BufferedReader inputBufferedReader, final List<LaborOriginEntry> originEntryList) {
        String line;
        int lineNumber = 0;
        final Map returnMessageMap = new HashMap();
        try {
            List<Message> tmperrors;
            while ((line = inputBufferedReader.readLine()) != null) {
                lineNumber++;
                final LaborOriginEntry laborOriginEntry = new LaborOriginEntry();
                tmperrors = laborOriginEntry.setFromTextFileForBatch(line, lineNumber);
                laborOriginEntry.setEntryId(lineNumber);
                if (tmperrors.size() > 0) {
                    returnMessageMap.put(lineNumber, tmperrors);
                } else {
                    originEntryList.add(laborOriginEntry);
                }
            }
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }

        return returnMessageMap;
    }

    @Override
    public Integer getGroupCount(final String fileNameWithPath) {
        final File file = new File(fileNameWithPath);
        final Iterator<LaborOriginEntry> fileIterator = new LaborOriginEntryFileIterator(file);
        int count = 0;

        while (fileIterator.hasNext()) {
            count++;
            fileIterator.next();
        }
        return count;
    }

    @Override
    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    @Override
    public void setBatchFileDirectoryName(final String batchFileDirectoryName) {
        this.batchFileDirectoryName = batchFileDirectoryName;
    }
}
