/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.LineIterator;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.batch.service.LaborEncumbranceAdjustmentService;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.sys.batch.AbstractStep;

import java.io.File;
import java.io.IOException;
import java.util.Date;

/**
 *  Builds a balance file from the current encumbrance balances in the labor ledger.
 */
public class LaborBuildEncumbranceBalanceFileStep extends AbstractStep {
    private static final Logger LOG = LogManager.getLogger();

    private String batchFileDirectoryName;
    private LaborEncumbranceAdjustmentService laborEncumbranceAdjustmentService;

    @Override
    public boolean execute(final String jobName, final Date jobRunDate) {
        LOG.debug("execute() started");

        final File inputFile = new File(batchFileDirectoryName + File.separator
                                        + LaborConstants.Encumbrance.SORTED_ENCUMBRANCE_INPUT_FILE
                                        + GeneralLedgerConstants.BatchFileSystem.EXTENSION);
        if (inputFile.exists()) {
            final File outputFile = new File(
                    batchFileDirectoryName + File.separator + LaborConstants.Encumbrance.ENCUMBRANCE_BALANCE_FILE
                    + GeneralLedgerConstants.BatchFileSystem.EXTENSION);
            Integer fiscalYear = null;

            try (LineIterator inputEntries = FileUtils.lineIterator(inputFile)) {
                if (inputEntries.hasNext()) {
                    final String firstLine = inputEntries.nextLine();
                    final LaborOriginEntry oe = new LaborOriginEntry(firstLine);
                    fiscalYear = oe.getUniversityFiscalYear();
                }
            } catch (final IOException ex) {
                LOG.error("Error encountered trying to read first line of labor encumbrance file", ex);
                throw new RuntimeException("Error encountered trying to read first line of labor encumbrance file", ex);
            }

            if (fiscalYear != null) {
                laborEncumbranceAdjustmentService.buildBalanceFile(fiscalYear, outputFile);
            }
        }
        return true;
    }

    public void setBatchFileDirectoryName(final String batchFileDirectoryName) {
        this.batchFileDirectoryName = batchFileDirectoryName;
    }

    public void setLaborEncumbranceAdjustmentService(
            final LaborEncumbranceAdjustmentService laborEncumbranceAdjustmentService
    ) {
        this.laborEncumbranceAdjustmentService = laborEncumbranceAdjustmentService;
    }
}
