/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.batch.service.LaborEncumbranceAdjustmentService;
import org.kuali.kfs.sys.batch.AbstractStep;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

/**
 * Batch step to create the encumbrance adjustment file.
 */

public class LaborCalculateEncumbranceAdjustmentsStep extends AbstractStep {
    private static final Logger LOG = LogManager.getLogger();

    private String batchFileDirectoryName;
    private String enterpriseFeedDirectoryName;
    private LaborEncumbranceAdjustmentService laborEncumbranceAdjustmentService;

    @Override
    public boolean execute(final String jobName, final Date jobRunDate) {
        LOG.debug("execute() started");

        final String inputFileName = batchFileDirectoryName + File.separator
                                     + LaborConstants.Encumbrance.SORTED_ENCUMBRANCE_INPUT_FILE
                                     + GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        final File inputFile = new File(inputFileName);
        final String balanceFileName =
                batchFileDirectoryName + File.separator + LaborConstants.Encumbrance.ENCUMBRANCE_BALANCE_FILE
                + GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        final File balanceFile = new File(balanceFileName);
        if (inputFile.exists() && balanceFile.exists()) {
            final String outputFileName =
                    batchFileDirectoryName + File.separator + LaborConstants.Encumbrance.ENCUMBRANCE_OUTPUT_FILE
                    + "-" + dateTimeService.toDateTimeStringForFilename(jobRunDate)
                    + GeneralLedgerConstants.BatchFileSystem.EXTENSION;
            final File outputFile = new File(outputFileName);
            final String errorFileName =
                    batchFileDirectoryName + File.separator + LaborConstants.Encumbrance.ENCUMBRANCE_ERROR_FILE
                    + "-" + dateTimeService.toDateTimeStringForFilename(jobRunDate)
                    + GeneralLedgerConstants.BatchFileSystem.EXTENSION;
            final File errorFile = new File(errorFileName);

            laborEncumbranceAdjustmentService.buildEncumbranceDifferenceFile(inputFile,
                    balanceFile,
                    outputFile,
                    errorFile
            );

            final String reconFileName =
                    batchFileDirectoryName + File.separator + LaborConstants.Encumbrance.ENCUMBRANCE_OUTPUT_FILE
                    + "-" + dateTimeService.toDateTimeStringForFilename(jobRunDate)
                    + GeneralLedgerConstants.BatchFileSystem.RECON_FILE_EXTENSION;
            final File reconFile = new File(reconFileName);

            final String doneFileName =
                    batchFileDirectoryName + File.separator + LaborConstants.Encumbrance.ENCUMBRANCE_OUTPUT_FILE
                    + "-" + dateTimeService.toDateTimeStringForFilename(jobRunDate)
                    + GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION;
            final File doneFile = new File(doneFileName);

            copyFile(outputFile, enterpriseFeedDirectoryName);
            copyFile(errorFile, enterpriseFeedDirectoryName);
            copyFile(reconFile, enterpriseFeedDirectoryName);
            copyFile(doneFile, enterpriseFeedDirectoryName);

            removeFilesfromBatchFileDirectory(List.of(inputFile, balanceFile, doneFile));

            final File batchFileDirectory = new File(batchFileDirectoryName);
            final List<File> doneFiles = Arrays.asList(batchFileDirectory.listFiles((batchFileDirectory1, name) ->
                    name.startsWith(LaborConstants.Encumbrance.ENCUMBRANCE_INPUT_FILE) && name.endsWith(
                            GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION)));

            for (final File file: doneFiles) {
                file.delete();
            }
        }
        return true;
    }

    private void removeFilesfromBatchFileDirectory(final List<File> files) {
        for (final File file : files) {
            if (file.exists()) {
                file.delete();
            }
        }
    }

    /**
     * Copy a specified file into the specified directory location
     *
     * @param aFile             The file that will be copied
     * @param fileDirectoryName Location of where the file needs to be copied to
     * @return Returns boolean to indicate if copy was successful or not
     */
    private void copyFile(final File aFile, final String fileDirectoryName) {
        final int length;
        final String bFileName = fileDirectoryName + File.separator + aFile.getName();
        final Path outPath = Paths.get(bFileName);

        try {
            Files.copy(aFile.toPath(), outPath, StandardCopyOption.REPLACE_EXISTING);
        } catch (final IOException e) {
            LOG.error("Unable to copy {} into {}", aFile::getName, () -> fileDirectoryName);
            throw new RuntimeException("Caught exception trying to archive files.", e);
        }
    }

    public void setBatchFileDirectoryName(final String batchFileDirectoryName) {
        this.batchFileDirectoryName = batchFileDirectoryName;
    }

    public void setLaborEncumbranceAdjustmentService(
            final LaborEncumbranceAdjustmentService laborEncumbranceAdjustmentService
    ) {
        this.laborEncumbranceAdjustmentService = laborEncumbranceAdjustmentService;
    }

    public void setEnterpriseFeedDirectoryName(final String enterpriseFeedDirectoryName) {
        this.enterpriseFeedDirectoryName = enterpriseFeedDirectoryName;
    }
}
