/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.businessobject;

import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.core.api.mo.common.active.MutableInactivatable;
import org.kuali.kfs.core.api.util.type.KualiPercent;
import org.kuali.kfs.integration.ld.LaborLedgerBenefitsCalculation;
import org.kuali.kfs.integration.ld.LaborLedgerPositionObjectBenefit;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.module.ld.service.LaborBenefitsCalculationService;
import org.kuali.kfs.sys.businessobject.FiscalYearBasedBusinessObject;
import org.kuali.kfs.sys.businessobject.SystemOptions;
import org.kuali.kfs.sys.context.SpringContext;

/**
 * Labor business object for Modeling of Position Object Benefit
 */
public class PositionObjectBenefit extends PersistableBusinessObjectBase implements LaborLedgerPositionObjectBenefit,
        MutableInactivatable, FiscalYearBasedBusinessObject {

    protected Integer universityFiscalYear;
    protected String chartOfAccountsCode;
    protected String financialObjectCode;
    protected String financialObjectBenefitsTypeCode;
    protected Chart chartOfAccounts;
    protected boolean active;

    protected ObjectCode financialObject;
    protected transient SystemOptions universityFiscal;
    protected BenefitsCalculation benefitsCalculation;
    protected BenefitsType financialObjectBenefitsType;
    protected LaborObject laborObject;
    private String laborBenefitRateCategoryCode;
    private LaborBenefitRateCategory laborBenefitRateCategory;

    public PositionObjectBenefit() {

    }

    @Override
    public Integer getUniversityFiscalYear() {
        return universityFiscalYear;
    }

    @Override
    public void setUniversityFiscalYear(final Integer universityFiscalYear) {
        this.universityFiscalYear = universityFiscalYear;
    }

    @Override
    public String getChartOfAccountsCode() {
        return chartOfAccountsCode;
    }

    @Override
    public void setChartOfAccountsCode(final String chartOfAccountsCode) {
        this.chartOfAccountsCode = chartOfAccountsCode;
    }

    @Override
    public String getFinancialObjectCode() {
        return financialObjectCode;
    }

    @Override
    public void setFinancialObjectCode(final String financialObjectCode) {
        this.financialObjectCode = financialObjectCode;
    }

    @Override
    public String getFinancialObjectBenefitsTypeCode() {
        return financialObjectBenefitsTypeCode;
    }

    @Override
    public void setFinancialObjectBenefitsTypeCode(final String financialObjectBenefitsTypeCode) {
        this.financialObjectBenefitsTypeCode = financialObjectBenefitsTypeCode;
    }

    @Override
    public ObjectCode getFinancialObject() {
        return financialObject;
    }

    @Override
    @Deprecated
    public void setFinancialObject(final ObjectCode financialObject) {
        this.financialObject = financialObject;
    }

    @Override
    public Chart getChartOfAccounts() {
        return chartOfAccounts;
    }

    @Override
    @Deprecated
    public void setChartOfAccounts(final Chart chartOfAccounts) {
        this.chartOfAccounts = chartOfAccounts;
    }

    @Override
    public SystemOptions getUniversityFiscal() {
        return universityFiscal;
    }

    @Override
    @Deprecated
    public void setUniversityFiscal(final SystemOptions universityFiscal) {
        this.universityFiscal = universityFiscal;
    }

    public BenefitsCalculation getBenefitsCalculation(final String laborBenefitRateCategoryCode) {
        BenefitsCalculation bc = SpringContext.getBean(LaborBenefitsCalculationService.class).getBenefitsCalculation(
                universityFiscalYear, chartOfAccountsCode, financialObjectBenefitsTypeCode, laborBenefitRateCategoryCode);
        //if we can't find a match, create a new benefit calculation with a 0.00 benefit percent
        if (bc == null) {
            bc = new BenefitsCalculation();
            bc.setPositionFringeBenefitPercent(new KualiPercent(0));
        }
        return bc;
    }

    public BenefitsCalculation getBenefitsCalculation() {
        if (benefitsCalculation == null) {
            benefitsCalculation = getBenefitsCalculation(getLaborBenefitRateCategoryCode());
        }
        return benefitsCalculation;
    }

    @Deprecated
    public void setBenefitsCalculation(final BenefitsCalculation benefitsCalculation) {
        this.benefitsCalculation = benefitsCalculation;
    }

    public BenefitsType getFinancialObjectBenefitsType() {
        return financialObjectBenefitsType;
    }

    @Deprecated
    public void setFinancialObjectBenefitsType(final BenefitsType financialObjectBenefitsType) {
        this.financialObjectBenefitsType = financialObjectBenefitsType;
    }

    public LaborObject getLaborObject() {
        return laborObject;
    }

    @Deprecated
    public void setLaborObject(final LaborObject laborObject) {
        this.laborObject = laborObject;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public void setActive(final boolean active) {
        this.active = active;
    }

    /**
     * This method (a hack by any other name...) returns a string so that an Labor Object Code Benefits can have a link
     * to view its own inquiry page after a look up
     *
     * @return the String "View Labor Object Code Benefits"
     */
    public String getLaborObjectCodeBenefitsViewer() {
        return "View Labor Object Code Benefits";
    }

    @Override
    public LaborLedgerBenefitsCalculation getLaborLedgerBenefitsCalculation() {
        return SpringContext.getBean(LaborBenefitsCalculationService.class)
                .getBenefitsCalculation(universityFiscalYear, chartOfAccountsCode, financialObjectBenefitsTypeCode);
    }

    @Override
    public LaborLedgerBenefitsCalculation getLaborLedgerBenefitsCalculation(final String laborBenefitRateCategoryCode) {
        return SpringContext.getBean(LaborBenefitsCalculationService.class).getBenefitsCalculation(
                universityFiscalYear,
                chartOfAccountsCode,
                financialObjectBenefitsTypeCode,
                laborBenefitRateCategoryCode
        );
    }

    @Override
    public void setLaborLedgerBenefitsCalculation(final LaborLedgerBenefitsCalculation laborLedgerBenefitsCalculation) {
        benefitsCalculation = (BenefitsCalculation) laborLedgerBenefitsCalculation;
    }

    public String getLaborBenefitRateCategoryCode() {
        return laborBenefitRateCategoryCode;
    }

    public void setLaborBenefitRateCategoryCode(final String laborBenefitRateCategoryCode) {
        this.laborBenefitRateCategoryCode = laborBenefitRateCategoryCode;
    }

    public LaborBenefitRateCategory getLaborBenefitRateCategory() {
        return laborBenefitRateCategory;
    }

    public void setLaborBenefitRateCategory(final LaborBenefitRateCategory laborBenefitRateCategory) {
        this.laborBenefitRateCategory = laborBenefitRateCategory;
    }
}
