/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.businessobject.lookup;

import org.kuali.kfs.kns.lookup.AbstractLookupableHelperServiceImpl;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.lookup.CollectionIncomplete;
import org.kuali.kfs.krad.util.BeanPropertyComparator;
import org.kuali.kfs.module.ld.businessobject.AccountStatusBaseFunds;
import org.kuali.kfs.module.ld.businessobject.inquiry.BaseFundsInquirableImpl;
import org.kuali.kfs.module.ld.service.LaborBaseFundsService;
import org.kuali.kfs.module.ld.service.LaborInquiryOptionsService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.krad.bo.BusinessObject;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * The BaseFundsLookupableHelperServiceImpl class is the front-end for all Base Fund balance inquiry processing.
 */
public class BaseFundsLookupableHelperServiceImpl extends AbstractLookupableHelperServiceImpl {

    private LaborBaseFundsService laborBaseFundsService;
    private LaborInquiryOptionsService laborInquiryOptionsService;

    @Override
    public HtmlData getInquiryUrl(final BusinessObject bo, final String propertyName) {
        return new BaseFundsInquirableImpl().getInquiryUrl(bo, propertyName);
    }

    @Override
    public List<? extends BusinessObject> getSearchResults(final Map<String, String> fieldValues) {
        setBackLocation(fieldValues.get(KFSConstants.BACK_LOCATION));
        setDocFormKey(fieldValues.get(KFSConstants.DOC_FORM_KEY));

        // test if the consolidation option is selected or not
        final boolean isConsolidated = laborInquiryOptionsService.isConsolidationSelected(fieldValues, getRows());

        final List<AccountStatusBaseFunds> searchResultsCollection =
                laborBaseFundsService.findAccountStatusBaseFundsWithCSFTracker(fieldValues, isConsolidated);

        // get the actual size of all qualified search results
        final Long actualSize = (long) searchResultsCollection.size();

        return buildSearchResultList(searchResultsCollection, actualSize);
    }

    /**
     * build the search result list from the given collection and the number of all qualified search results
     *
     * @param searchResultsCollection the given search results, which may be a subset of the qualified search results
     * @param actualSize              the number of all qualified search results
     * @return the search result list with the given results and actual size
     */
    protected List buildSearchResultList(final Collection searchResultsCollection, final Long actualSize) {
        final CollectionIncomplete results = new CollectionIncomplete(searchResultsCollection, actualSize);

        // sort list if default sort column given
        final List searchResults = results;
        final List<String> defaultSortColumns = getDefaultSortColumns();
        if (defaultSortColumns.size() > 0) {
            results.sort(new BeanPropertyComparator(defaultSortColumns, true));
        }
        return searchResults;
    }

    public void setLaborInquiryOptionsService(final LaborInquiryOptionsService laborInquiryOptionsService) {
        this.laborInquiryOptionsService = laborInquiryOptionsService;
    }

    public void setLaborBaseFundsService(final LaborBaseFundsService laborBaseFundsService) {
        this.laborBaseFundsService = laborBaseFundsService;
    }
}
