/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.businessobject.lookup;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.gl.businessobject.inquiry.EntryInquirableImpl;
import org.kuali.kfs.gl.businessobject.inquiry.InquirableFinancialDocument;
import org.kuali.kfs.integration.ld.businessobject.inquiry.AbstractPositionDataDetailsInquirableImpl;
import org.kuali.kfs.kns.lookup.AbstractLookupableHelperServiceImpl;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.lookup.HtmlData.AnchorHtmlData;
import org.kuali.kfs.kns.lookup.CollectionIncomplete;
import org.kuali.kfs.krad.util.BeanPropertyComparator;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.businessobject.LedgerEntry;
import org.kuali.kfs.module.ld.businessobject.inquiry.PositionDataDetailsInquirableImpl;
import org.kuali.kfs.module.ld.service.LaborInquiryOptionsService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.krad.bo.BusinessObject;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The class is the front-end for all Ledger Entry inquiry processing.
 *
 * @see org.kuali.kfs.module.ld.businessobject.LedgerEntry
 */
public class LedgerEntryLookupableHelperServiceImpl extends AbstractLookupableHelperServiceImpl {

    private LaborInquiryOptionsService laborInquiryOptionsService;

    @Override
    public HtmlData getInquiryUrl(final BusinessObject businessObject, final String propertyName) {
        if (KFSPropertyConstants.DOCUMENT_NUMBER.equals(propertyName)) {
            if (businessObject instanceof LedgerEntry) {
                final LedgerEntry entry = (LedgerEntry) businessObject;
                return new AnchorHtmlData(new InquirableFinancialDocument().getInquirableDocumentUrl(entry),
                        KFSConstants.EMPTY_STRING, "view ledger entry");
            }
        } else if (KFSPropertyConstants.POSITION_NUMBER.equals(propertyName)) {
            final LedgerEntry entry = (LedgerEntry) businessObject;
            final AbstractPositionDataDetailsInquirableImpl positionDataDetailsInquirable =
                    new PositionDataDetailsInquirableImpl();

            final Map<String, String> fieldValues = new HashMap<>();
            fieldValues.put(propertyName, entry.getPositionNumber());

            final BusinessObject positionData = positionDataDetailsInquirable.getBusinessObject(fieldValues);

            return positionData == null ? new AnchorHtmlData(KFSConstants.EMPTY_STRING, KFSConstants.EMPTY_STRING) :
                    positionDataDetailsInquirable.getInquiryUrl(positionData, propertyName);
        } else if (KFSPropertyConstants.EMPLOYEE_NAME.equals(propertyName)) {
            return new HtmlData.AnchorHtmlData();
        }
        return new EntryInquirableImpl().getInquiryUrl(businessObject, propertyName);
    }

    @Override
    public List<? extends BusinessObject> getSearchResults(final Map<String, String> fieldValues) {
        setBackLocation(fieldValues.get(KFSConstants.BACK_LOCATION));
        setDocFormKey(fieldValues.get(KFSConstants.DOC_FORM_KEY));

        // get the pending entry option. This method must be prior to the get search results
        final String pendingEntryOption = laborInquiryOptionsService.getSelectedPendingEntryOption(fieldValues);

        // get the input balance type code
        final String balanceTypeCode = fieldValues.get(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE);
        final boolean isA21Balance = StringUtils.isNotEmpty(balanceTypeCode)
                                     && LaborConstants.BalanceInquiries.BALANCE_TYPE_AC_AND_A21.equals(balanceTypeCode.trim());

        if (isA21Balance) {
            fieldValues.put(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, KFSConstants.BALANCE_TYPE_ACTUAL);
        }

        final Collection<LedgerEntry> ledgerEntries = getLookupService().findCollectionBySearch(LedgerEntry.class,
                fieldValues);
        laborInquiryOptionsService.updateLedgerEntryByPendingLedgerEntry(ledgerEntries, fieldValues,
                pendingEntryOption);

        // add the ledger entries into the search results if the searching balance type code is A21
        if (isA21Balance) {
            fieldValues.put(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, KFSConstants.BALANCE_TYPE_A21);
            final Collection<LedgerEntry> effortLedgerEntries = getLookupService().findCollectionBySearch(LedgerEntry.class,
                    fieldValues);
            laborInquiryOptionsService.updateLedgerEntryByPendingLedgerEntry(effortLedgerEntries, fieldValues,
                    pendingEntryOption);

            ledgerEntries.addAll(effortLedgerEntries);
        }

        // get the actual size of all qualified search results
        final Long actualSize = (long) ledgerEntries.size();
        return buildSearchResultList(ledgerEntries, actualSize);
    }

    /**
     * build the search result list from the given collection and the number of all qualified search results
     *
     * @param searchResultsCollection the given search results, which may be a subset of the qualified search results
     * @param actualSize              the number of all qualified search results
     * @return the search result list with the given results and actual size
     */
    protected List buildSearchResultList(final Collection searchResultsCollection, final Long actualSize) {
        final CollectionIncomplete results = new CollectionIncomplete(searchResultsCollection, actualSize);

        // sort list if default sort column given
        final List searchResults = results;
        final List<String> defaultSortColumns = getDefaultSortColumns();
        if (defaultSortColumns.size() > 0) {
            results.sort(new BeanPropertyComparator(defaultSortColumns, true));
        }
        return searchResults;
    }

    public void setLaborInquiryOptionsService(final LaborInquiryOptionsService laborInquiryOptionsService) {
        this.laborInquiryOptionsService = laborInquiryOptionsService;
    }
}
