/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.document.Copyable;
import org.kuali.kfs.module.ld.businessobject.ExpenseTransferSourceAccountingLine;
import org.kuali.kfs.module.ld.businessobject.ExpenseTransferTargetAccountingLine;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.AmountTotaling;
import org.kuali.kfs.sys.document.Correctable;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kim.api.identity.PersonService;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Labor Base class for Expense Transfer Documents
 */
public abstract class LaborExpenseTransferDocumentBase extends LaborLedgerPostingDocumentBase implements
        AmountTotaling, Copyable, Correctable, LaborExpenseTransferDocument {

    protected String emplid;
    protected Person user;

    public LaborExpenseTransferDocumentBase() {
        super();
    }

    /**
     * @return true if target accounting lines have the same amounts as source accounting lines for each object code;
     *         otherwise, false
     */
    public Map<String, KualiDecimal> getUnbalancedObjectCodes() {
        final Map<String, KualiDecimal> amountsFromSourceLine = summerizeByObjectCode(getSourceAccountingLines());
        final Map<String, KualiDecimal> amountsFromTargetLine = summerizeByObjectCode(getTargetAccountingLines());

        final Map<String, KualiDecimal> unbalancedAmounts = new HashMap<>();
        for (final String objectCode : amountsFromSourceLine.keySet()) {
            final KualiDecimal sourceAmount = amountsFromSourceLine.get(objectCode);

            if (!amountsFromTargetLine.containsKey(objectCode)) {
                unbalancedAmounts.put(objectCode, sourceAmount.negated());
            } else {
                final KualiDecimal targetAmount = amountsFromTargetLine.get(objectCode);
                final KualiDecimal amountDifference = targetAmount.subtract(sourceAmount);
                if (amountDifference.isNonZero()) {
                    unbalancedAmounts.put(objectCode, amountDifference);
                }
            }
        }

        for (final String objectCode : amountsFromTargetLine.keySet()) {
            if (!amountsFromSourceLine.containsKey(objectCode)) {
                final KualiDecimal targetAmount = amountsFromTargetLine.get(objectCode);
                unbalancedAmounts.put(objectCode, targetAmount);
            }
        }

        return unbalancedAmounts;
    }

    /**
     * summarize the amounts of accounting lines by object codes
     *
     * @param accountingLines the given accounting line list
     * @return the summarized amounts by object codes
     */
    protected Map<String, KualiDecimal> summerizeByObjectCode(final List accountingLines) {
        final Map<String, KualiDecimal> amountByObjectCode = new HashMap<>();

        for (final Object accountingLine : accountingLines) {
            final AccountingLine line = (AccountingLine) accountingLine;
            final String objectCode = line.getFinancialObjectCode();
            KualiDecimal amount = line.getAmount();

            if (amountByObjectCode.containsKey(objectCode)) {
                amount = amount.add(amountByObjectCode.get(objectCode));
            }
            amountByObjectCode.put(objectCode, amount);
        }

        return amountByObjectCode;
    }

    @Override
    public String getEmplid() {
        return emplid;
    }

    @Override
    public void setEmplid(final String emplid) {
        this.emplid = emplid;
    }

    public Person getUser() {
        if (user == null || !StringUtils.equals(user.getEmployeeId(), emplid)) {
            user = SpringContext.getBean(PersonService.class).getPersonByEmployeeId(emplid);
        }
        if (user == null && emplid == null) {
            user = new Person();
        }
        return user;
    }

    public void setUser(final Person user) {
        this.user = user;
    }

    /**
     * Overrides the base implementation to return "From".
     */
    @Override
    public String getSourceAccountingLinesSectionTitle() {
        return KFSConstants.FROM;
    }

    /**
     * Overrides the base implementation to return "To".
     */
    @Override
    public String getTargetAccountingLinesSectionTitle() {
        return KFSConstants.TO;
    }

    @Override
    public Class getSourceAccountingLineClass() {
        return ExpenseTransferSourceAccountingLine.class;
    }

    @Override
    public Class getTargetAccountingLineClass() {
        return ExpenseTransferTargetAccountingLine.class;
    }

}
