/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.service.impl;

import org.apache.commons.collections4.IteratorUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.gl.OJBUtility;
import org.kuali.kfs.module.ld.businessobject.EmployeeFunding;
import org.kuali.kfs.module.ld.businessobject.LaborBalanceSummary;
import org.kuali.kfs.module.ld.businessobject.LaborTransaction;
import org.kuali.kfs.module.ld.businessobject.LedgerBalance;
import org.kuali.kfs.module.ld.businessobject.LedgerBalanceForYearEndBalanceForward;
import org.kuali.kfs.module.ld.dataaccess.LaborLedgerBalanceDao;
import org.kuali.kfs.module.ld.service.LaborCalculatedSalaryFoundationTrackerService;
import org.kuali.kfs.module.ld.service.LaborLedgerBalanceService;
import org.kuali.kfs.module.ld.util.DebitCreditUtil;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class LaborLedgerBalanceServiceImpl implements LaborLedgerBalanceService {

    private static final Logger LOG = LogManager.getLogger();

    protected LaborLedgerBalanceDao laborLedgerBalanceDao;
    protected LaborCalculatedSalaryFoundationTrackerService laborCalculatedSalaryFoundationTrackerService;

    @Override
    public Iterator<LedgerBalance> findBalancesForFiscalYear(final Integer fiscalYear) {
        return laborLedgerBalanceDao.findBalancesForFiscalYear(fiscalYear);
    }

    @Override
    public Iterator<LedgerBalance> findBalancesForFiscalYear(
            final Integer fiscalYear, final Map<String, String> fieldValues,
            final List<String> encumbranceBalanceTypes) {
        return laborLedgerBalanceDao.findBalancesForFiscalYear(fiscalYear, fieldValues, encumbranceBalanceTypes);
    }

    @Override
    public Iterator<LedgerBalanceForYearEndBalanceForward> findBalancesForFiscalYear(
            final Integer fiscalYear,
            final Map<String, String> fieldValues, final Collection<String> subFundGroupCodes, final Collection<String> fundGroupCodes) {
        final List<String> fundGroupCodesList = new ArrayList<>(fundGroupCodes);
        Collections.sort(fundGroupCodesList);
        final List<String> subFundGroupCodesList = new ArrayList<>(subFundGroupCodes);
        Collections.sort(subFundGroupCodesList);
        return laborLedgerBalanceDao.findBalancesForFiscalYear(fiscalYear, fieldValues, subFundGroupCodesList,
                fundGroupCodesList);
    }

    @Override
    public Iterator findBalance(
            final Map fieldValues, final boolean isConsolidated, final List<String> encumbranceBalanceTypes,
            final boolean noZeroAmounts) {
        LOG.debug("findBalance() started");
        return laborLedgerBalanceDao.findBalance(fieldValues, isConsolidated, encumbranceBalanceTypes, noZeroAmounts);
    }

    @Override
    @Deprecated
    public Iterator findBalance(final Map fieldValues, final boolean isConsolidated, final List<String> encumbranceBalanceTypes) {
        LOG.debug("findBalance() started");
        return laborLedgerBalanceDao.findBalance(fieldValues, isConsolidated, encumbranceBalanceTypes);
    }

    @Override
    public Integer getBalanceRecordCount(
            final Map fieldValues, final boolean isConsolidated, final List<String> encumbranceBalanceTypes,
            final boolean noZeroAmounts) {
        LOG.debug("getBalanceRecordCount() started");

        final int recordCount;
        if (!isConsolidated) {
            recordCount = OJBUtility.getResultSizeFromMap(fieldValues, new LedgerBalance()).intValue();
        } else {
            final Iterator recordCountIterator = laborLedgerBalanceDao.getConsolidatedBalanceRecordCount(fieldValues,
                    encumbranceBalanceTypes, noZeroAmounts);
            final List recordCountList = IteratorUtils.toList(recordCountIterator);
            recordCount = recordCountList.size();
        }
        return recordCount;
    }

    @Override
    @Deprecated
    public Integer getBalanceRecordCount(
            final Map fieldValues, final boolean isConsolidated,
            final List<String> encumbranceBalanceTypes) {
        LOG.debug("getBalanceRecordCount() started");

        final int recordCount;
        if (!isConsolidated) {
            recordCount = OJBUtility.getResultSizeFromMap(fieldValues, new LedgerBalance()).intValue();
        } else {
            final Iterator recordCountIterator = laborLedgerBalanceDao.getConsolidatedBalanceRecordCount(fieldValues,
                    encumbranceBalanceTypes);
            final List recordCountList = IteratorUtils.toList(recordCountIterator);
            recordCount = recordCountList.size();
        }
        return recordCount;
    }

    @Override
    public <T extends LedgerBalance> T findLedgerBalance(
            final Collection<T> ledgerBalanceCollection,
            final LaborTransaction transaction, final List<String> keyList) {
        for (final T ledgerBalance : ledgerBalanceCollection) {
            final boolean found = ObjectUtil.equals(ledgerBalance, transaction, keyList);
            if (found) {
                return ledgerBalance;
            }
        }
        return null;
    }

    @Override
    public <T extends LedgerBalance> T findLedgerBalance(
            final Collection<T> ledgerBalanceCollection,
            final LaborTransaction transaction) {
        for (final T ledgerBalance : ledgerBalanceCollection) {
            final boolean found = ObjectUtil.equals(ledgerBalance, transaction, ledgerBalance.getPrimaryKeyList());
            if (found) {
                return ledgerBalance;
            }
        }
        return null;
    }

    @Override
    @Transactional
    public <T extends LedgerBalance> void updateLedgerBalance(final T ledgerBalance, final LaborTransaction transaction) {
        final String debitCreditCode = transaction.getTransactionDebitCreditCode();
        KualiDecimal amount = transaction.getTransactionLedgerEntryAmount();
        amount = DebitCreditUtil.getNumericAmount(amount, debitCreditCode);
        ledgerBalance.addAmount(transaction.getUniversityFiscalPeriodCode(), amount);
    }

    @Override
    @Transactional
    public LedgerBalance addLedgerBalance(
            final Collection<LedgerBalance> ledgerBalanceCollection,
            final LaborTransaction transaction) {
        final LedgerBalance ledgerBalance = findLedgerBalance(ledgerBalanceCollection, transaction);

        if (ledgerBalance == null) {
            final LedgerBalance newLedgerBalance = new LedgerBalance();
            ObjectUtil.buildObject(newLedgerBalance, transaction);
            updateLedgerBalance(newLedgerBalance, transaction);

            ledgerBalanceCollection.add(newLedgerBalance);
            return newLedgerBalance;
        }
        return null;
    }

    @Override
    public List<EmployeeFunding> findEmployeeFunding(final Map fieldValues, final boolean isConsolidated) {
        final List<EmployeeFunding> currentFundsCollection = laborLedgerBalanceDao.findCurrentEmployeeFunds(fieldValues);
        final List<EmployeeFunding> encumbranceFundsCollection =
                laborLedgerBalanceDao.findEncumbranceEmployeeFunds(fieldValues);

        // update the employee fundings
        for (final EmployeeFunding employeeFunding : currentFundsCollection) {
            employeeFunding.setCurrentAmount(employeeFunding.getAccountLineAnnualBalanceAmount());
        }

        // merge encumbrance with the current funds
        for (final EmployeeFunding encumbranceFunding : encumbranceFundsCollection) {
            final KualiDecimal encumbrance = encumbranceFunding.getAccountLineAnnualBalanceAmount()
                    .add(encumbranceFunding.getContractsGrantsBeginningBalanceAmount());
            encumbranceFunding.setOutstandingEncumbrance(encumbrance);

            if (currentFundsCollection.contains(encumbranceFunding)) {
                final int index = currentFundsCollection.indexOf(encumbranceFunding);
                currentFundsCollection.get(index).setOutstandingEncumbrance(encumbrance);
            } else if (encumbrance != null && encumbrance.isNonZero()) {
                currentFundsCollection.add(encumbranceFunding);
            }
        }

        return currentFundsCollection;
    }

    @Override
    public List<EmployeeFunding> findEmployeeFundingWithCSFTracker(final Map fieldValues, final boolean isConsolidated) {
        final List<EmployeeFunding> currentFundsCollection = findEmployeeFunding(fieldValues, isConsolidated);
        final List<EmployeeFunding> CSFTrackersCollection = laborCalculatedSalaryFoundationTrackerService
                .findCSFTrackersAsEmployeeFunding(fieldValues, isConsolidated);

        for (final EmployeeFunding CSFTrackerAsEmployeeFunding : CSFTrackersCollection) {
            if (currentFundsCollection.contains(CSFTrackerAsEmployeeFunding)) {
                final int index = currentFundsCollection.indexOf(CSFTrackerAsEmployeeFunding);
                final EmployeeFunding currentFunds = currentFundsCollection.get(index);

                currentFunds.setCsfDeleteCode(CSFTrackerAsEmployeeFunding.getCsfDeleteCode());
                currentFunds.setCsfTimePercent(CSFTrackerAsEmployeeFunding.getCsfTimePercent());
                currentFunds.setCsfFundingStatusCode(CSFTrackerAsEmployeeFunding.getCsfFundingStatusCode());
                currentFunds.setCsfAmount(CSFTrackerAsEmployeeFunding.getCsfAmount());
                currentFunds.setCsfFullTimeEmploymentQuantity(CSFTrackerAsEmployeeFunding
                        .getCsfFullTimeEmploymentQuantity());
            } else {
                /*KFSCNTRB-1534 It is possible for a CSF item to exist on its own without being related to
                  a current funds record.*/
                currentFundsCollection.add(CSFTrackerAsEmployeeFunding);
            }
        }

        return currentFundsCollection;
    }

    @Override
    public List<LaborBalanceSummary> findBalanceSummary(final Integer fiscalYear, final Collection<String> balanceTypes) {
        return laborLedgerBalanceDao.findBalanceSummary(fiscalYear, balanceTypes);
    }

    @Override
    public List<List<String>> findAccountsInFundGroups(
            final Integer fiscalYear, final Map<String, String> fieldValues,
            final List<String> subFundGroupCodes, final List<String> fundGroupCodes) {
        return laborLedgerBalanceDao.findAccountsInFundGroups(fiscalYear, fieldValues, subFundGroupCodes,
                fundGroupCodes);
    }

    @Override
    public Collection<LedgerBalance> findLedgerBalances(
            final Map<String, List<String>> fieldValues, final Map<String,
            List<String>> excludedFieldValues, final Set<Integer> fiscalYears, final List<String> balanceTypeList,
            final List<String> positionObjectGroupCodes) {
        return laborLedgerBalanceDao.findLedgerBalances(fieldValues, excludedFieldValues, fiscalYears,
                balanceTypeList, positionObjectGroupCodes);
    }

    @Override
    @Transactional
    public void deleteLedgerBalancesPriorToYear(final Integer fiscalYear, final String chartOfAccountsCode) {
        laborLedgerBalanceDao.deleteLedgerBalancesPriorToYear(fiscalYear, chartOfAccountsCode);
    }

    public void setLaborLedgerBalanceDao(final LaborLedgerBalanceDao laborLedgerBalanceDao) {
        this.laborLedgerBalanceDao = laborLedgerBalanceDao;
    }

    public void setLaborCalculatedSalaryFoundationTrackerService(
            final LaborCalculatedSalaryFoundationTrackerService laborCalculatedSalaryFoundationTrackerService) {
        this.laborCalculatedSalaryFoundationTrackerService = laborCalculatedSalaryFoundationTrackerService;
    }
}
