/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch.service.impl;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.LineIterator;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.A21SubAccount;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.BalanceType;
import org.kuali.kfs.coa.service.ObjectCodeService;
import org.kuali.kfs.coa.service.OffsetDefinitionService;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.parameter.ParameterEvaluator;
import org.kuali.kfs.core.api.parameter.ParameterEvaluatorService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.gl.GLParameterConstants;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.gl.batch.BatchSortUtil;
import org.kuali.kfs.gl.batch.ScrubberStep;
import org.kuali.kfs.gl.businessobject.DemergerReportData;
import org.kuali.kfs.gl.businessobject.OriginEntryStatistics;
import org.kuali.kfs.gl.businessobject.ScrubberProcessUnitOfWork;
import org.kuali.kfs.gl.report.LedgerSummaryReport;
import org.kuali.kfs.gl.report.PreScrubberReport;
import org.kuali.kfs.gl.report.PreScrubberReportData;
import org.kuali.kfs.gl.report.TransactionListingReport;
import org.kuali.kfs.gl.service.OriginEntryGroupService;
import org.kuali.kfs.gl.service.PreScrubberService;
import org.kuali.kfs.gl.service.ScrubberReportData;
import org.kuali.kfs.gl.service.ScrubberValidator;
import org.kuali.kfs.krad.service.PersistenceService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.LaborParameterConstants;
import org.kuali.kfs.module.ld.batch.LaborDemergerStep;
import org.kuali.kfs.module.ld.batch.LaborScrubberSortComparator;
import org.kuali.kfs.module.ld.batch.service.LaborAccountingCycleCachingService;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntryFieldUtil;
import org.kuali.kfs.module.ld.service.LaborOriginEntryService;
import org.kuali.kfs.module.ld.util.FilteringLaborOriginEntryFileIterator;
import org.kuali.kfs.module.ld.util.FilteringLaborOriginEntryFileIterator.LaborOriginEntryFilter;
import org.kuali.kfs.module.ld.util.LaborOriginEntryFileIterator;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.Message;
import org.kuali.kfs.sys.batch.service.WrappingBatchService;
import org.kuali.kfs.sys.businessobject.UniversityDate;
import org.kuali.kfs.sys.dataaccess.UniversityDateDao;
import org.kuali.kfs.sys.service.DocumentNumberAwareReportWriterService;
import org.kuali.kfs.sys.service.FlexibleOffsetAccountService;
import org.kuali.kfs.sys.service.ReportWriterService;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.charset.StandardCharsets;
import java.sql.Date;
import java.text.NumberFormat;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.StringTokenizer;

/**
 * This class has the logic for the scrubber. It is required because the scrubber process needs instance variables.
 * Instance variables in a spring service are shared between all code calling the service. This will make sure each
 * run of the scrubber has it's own instance variables instead of being shared.
 */
public class LaborScrubberProcess {

    private static final Logger LOG = LogManager.getLogger();

    // 40 spaces - used for filling in descriptions with spaces
    protected static String SPACES = "                                        ";

    /* Services required */
    protected FlexibleOffsetAccountService flexibleOffsetAccountService;
    protected LaborOriginEntryService laborOriginEntryService;
    protected OriginEntryGroupService originEntryGroupService;
    protected DateTimeService dateTimeService;
    protected OffsetDefinitionService offsetDefinitionService;
    protected ObjectCodeService objectCodeService;
    protected ConfigurationService kualiConfigurationService;
    protected UniversityDateDao universityDateDao;
    protected PersistenceService persistenceService;
    protected ScrubberValidator scrubberValidator;
    protected LaborAccountingCycleCachingService laborAccountingCycleCachingService;
    protected PreScrubberService laborPreScrubberService;

    protected DocumentNumberAwareReportWriterService laborMainReportWriterService;
    protected DocumentNumberAwareReportWriterService laborLedgerReportWriterService;
    protected ReportWriterService laborBadBalanceTypeReportWriterService;
    protected ReportWriterService laborErrorListingReportWriterService;
    protected DocumentNumberAwareReportWriterService laborGeneratedTransactionsReportWriterService;
    protected ReportWriterService laborDemergerReportWriterService;
    protected DocumentNumberAwareReportWriterService laborPreScrubberReportWriterService;
    protected ParameterService parameterService;
    private final ParameterEvaluatorService parameterEvaluatorService;

    protected String batchFileDirectoryName;

    enum GROUP_TYPE {
        VALID, ERROR, EXPIRED
    }

    // These are all different forms of the run date for this job
    protected Date runDate;
    protected UniversityDate universityRunDate;
    protected String offsetString;

    // These fields are used to control whether the job was run before some set time, if so, the rundate of the job
    // will be set to 11:59 PM of the previous day
    protected Integer cutoffHour;
    protected Integer cutoffMinute;
    protected Integer cutoffSecond;

    // Unit Of Work info
    protected ScrubberProcessUnitOfWork scrubberProcessUnitOfWork;
    protected KualiDecimal scrubCostShareAmount;
    protected ScrubberReportData scrubberReport;

    // Description names
    protected String offsetDescription;
    protected String capitalizationDescription;
    protected String liabilityDescription;
    protected String transferDescription;
    protected String costShareDescription;

    protected String inputFile;
    protected String validFile;
    protected String errorFile;
    protected String expiredFile;

    private LocalDateTime runLocalDateTime;

    /**
     * These parameters are all the dependencies.
     */
    public LaborScrubberProcess(
            final FlexibleOffsetAccountService flexibleOffsetAccountService,
            final LaborAccountingCycleCachingService laborAccountingCycleCachingService,
            final LaborOriginEntryService laborOriginEntryService, final OriginEntryGroupService originEntryGroupService,
            final DateTimeService dateTimeService, final OffsetDefinitionService offsetDefinitionService,
            final ObjectCodeService objectCodeService, final ConfigurationService kualiConfigurationService,
            final UniversityDateDao universityDateDao, final PersistenceService persistenceService,
            final ScrubberValidator scrubberValidator, final String batchFileDirectoryName,
            final DocumentNumberAwareReportWriterService laborMainReportWriterService,
            final DocumentNumberAwareReportWriterService laborLedgerReportWriterService,
            final ReportWriterService laborBadBalanceTypeReportWriterService,
            final ReportWriterService laborErrorListingReportWriterService,
            final DocumentNumberAwareReportWriterService laborGeneratedTransactionsReportWriterService,
            final ReportWriterService laborDemergerReportWriterService, final PreScrubberService laborPreScrubberService,
            final DocumentNumberAwareReportWriterService laborPreScrubberReportWriterService,
            final ParameterService parameterService, final ParameterEvaluatorService parameterEvaluatorService) {
        super();
        this.flexibleOffsetAccountService = flexibleOffsetAccountService;
        this.laborAccountingCycleCachingService = laborAccountingCycleCachingService;
        this.laborOriginEntryService = laborOriginEntryService;
        this.originEntryGroupService = originEntryGroupService;
        this.dateTimeService = dateTimeService;
        this.offsetDefinitionService = offsetDefinitionService;
        this.objectCodeService = objectCodeService;
        this.kualiConfigurationService = kualiConfigurationService;
        this.universityDateDao = universityDateDao;
        this.persistenceService = persistenceService;
        this.scrubberValidator = scrubberValidator;
        this.batchFileDirectoryName = batchFileDirectoryName;
        this.laborMainReportWriterService = laborMainReportWriterService;
        this.laborLedgerReportWriterService = laborLedgerReportWriterService;
        this.laborBadBalanceTypeReportWriterService = laborBadBalanceTypeReportWriterService;
        this.laborErrorListingReportWriterService = laborErrorListingReportWriterService;
        this.laborGeneratedTransactionsReportWriterService = laborGeneratedTransactionsReportWriterService;
        this.laborDemergerReportWriterService = laborDemergerReportWriterService;
        this.laborPreScrubberService = laborPreScrubberService;
        this.laborPreScrubberReportWriterService = laborPreScrubberReportWriterService;
        this.parameterService = parameterService;
        this.parameterEvaluatorService = parameterEvaluatorService;

        cutoffHour = null;
        cutoffMinute = null;
        cutoffSecond = null;

        initCutoffTime();
    }

    /**
     * Scrub this single group read only. This will only output the scrubber report. It won't output any other groups.
     *
     * @param fileName
     * @param documentNumber
     */
    public void scrubGroupReportOnly(final String fileName, final String documentNumber) {
        inputFile = fileName + ".sort";
        validFile = batchFileDirectoryName + File.separator +
                LaborConstants.BatchFileSystem.SCRUBBER_VALID_OUTPUT_FILE +
                GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        errorFile = batchFileDirectoryName + File.separator +
                LaborConstants.BatchFileSystem.SCRUBBER_ERROR_OUTPUT_FILE +
                GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        expiredFile = batchFileDirectoryName + File.separator +
                LaborConstants.BatchFileSystem.SCRUBBER_EXPIRED_OUTPUT_FILE +
                GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        final String prescrubOutput = batchFileDirectoryName + File.separator +
                                      LaborConstants.BatchFileSystem.PRE_SCRUBBER_FILE + GeneralLedgerConstants.BatchFileSystem.EXTENSION;

        final PreScrubberReportData preScrubberReportData;
        // run pre-scrubber on the raw input into the sort process
        try (LineIterator inputEntries = FileUtils.lineIterator(new File(fileName))) {
            preScrubberReportData = laborPreScrubberService.preprocessOriginEntries(inputEntries, prescrubOutput);
        } catch (final IOException e1) {
            LOG.error("Error encountered trying to prescrub GLCP/LLCP document", e1);
            throw new RuntimeException("Error encountered trying to prescrub GLCP/LLCP document", e1);
        }
        if (preScrubberReportData != null) {
            laborPreScrubberReportWriterService.setDocumentNumber(documentNumber);
            ((WrappingBatchService) laborPreScrubberReportWriterService).initialize();
            try {
                new PreScrubberReport().generateReport(preScrubberReportData, laborPreScrubberReportWriterService);
            } finally {
                ((WrappingBatchService) laborPreScrubberReportWriterService).destroy();
            }
        }
        BatchSortUtil.sortTextFileWithFields(prescrubOutput, inputFile, new LaborScrubberSortComparator());

        scrubEntries(true, documentNumber);

        final File deleteSortFile = new File(inputFile);
        final File deleteValidFile = new File(validFile);
        final File deleteErrorFile = new File(errorFile);
        final File deleteExpiredFile = new File(expiredFile);
        try {
            deleteSortFile.delete();
            deleteValidFile.delete();
            deleteErrorFile.delete();
            deleteExpiredFile.delete();
        } catch (final Exception e) {
            LOG.error("scrubGroupReportOnly delete output files process Stopped: {}", e::getMessage);
            throw new RuntimeException("scrubGroupReportOnly delete output files process Stopped: " + e.getMessage(), e);
        }
    }

    public void scrubEntries() {
        inputFile = batchFileDirectoryName + File.separator +
                LaborConstants.BatchFileSystem.SCRUBBER_INPUT_FILE + GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        validFile = batchFileDirectoryName + File.separator +
                LaborConstants.BatchFileSystem.SCRUBBER_VALID_OUTPUT_FILE +
                GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        errorFile = batchFileDirectoryName + File.separator +
                LaborConstants.BatchFileSystem.SCRUBBER_ERROR_OUTPUT_FILE +
                GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        expiredFile = batchFileDirectoryName + File.separator +
                LaborConstants.BatchFileSystem.SCRUBBER_EXPIRED_OUTPUT_FILE +
                GeneralLedgerConstants.BatchFileSystem.EXTENSION;

        scrubEntries(false, null);
    }

    /**
     * Scrub all entries that need it in origin entry. Put valid scrubbed entries in a scrubber valid group, put
     * errors in a scrubber error group, and transactions with an expired account in the scrubber expired account
     * group.
     */
    public void scrubEntries(final boolean reportOnlyMode, final String documentNumber) {
        LOG.debug("scrubEntries() started");

        if (reportOnlyMode) {
            laborMainReportWriterService.setDocumentNumber(documentNumber);
            laborLedgerReportWriterService.setDocumentNumber(documentNumber);
            laborGeneratedTransactionsReportWriterService.setDocumentNumber(documentNumber);
        }

        // setup an object to hold the "default" date information
        runDate = calculateRunDate(dateTimeService.getCurrentDate());
        runLocalDateTime = dateTimeService.getLocalDateTime(runDate);

        universityRunDate = laborAccountingCycleCachingService.getUniversityDate(runDate);
        if (universityRunDate == null) {
            throw new IllegalStateException(kualiConfigurationService.getPropertyValueAsString(
                    KFSKeyConstants.ERROR_UNIV_DATE_NOT_FOUND));
        }
        setOffsetString();
        setDescriptions();

        try {
            ((WrappingBatchService) laborMainReportWriterService).initialize();
            ((WrappingBatchService) laborLedgerReportWriterService).initialize();
            if (reportOnlyMode) {
                ((WrappingBatchService) laborGeneratedTransactionsReportWriterService).initialize();
            }

            scrubberReport = new ScrubberReportData();
            processGroup();

            // Run the reports
            if (reportOnlyMode) {
                generateScrubberTransactionsOnline();
            } else {
                generateScrubberBadBalanceTypeListingReport();
            }
        } finally {
            ((WrappingBatchService) laborMainReportWriterService).destroy();
            ((WrappingBatchService) laborLedgerReportWriterService).destroy();
            if (reportOnlyMode) {
                ((WrappingBatchService) laborGeneratedTransactionsReportWriterService).destroy();
            }
        }
    }

    /**
     * Determine the type of the transaction by looking at attributes
     *
     * @param transaction Transaction to identify
     * @return CE (Cost share encumbrance, O (Offset), C (Capitalization), L (Liability), T (Transfer),
     *         CS (Cost Share), X (Other)
     */
    protected String getTransactionType(final LaborOriginEntry transaction) {
        if ("CE".equals(transaction.getFinancialBalanceTypeCode())) {
            return "CE";
        }
        final String desc = transaction.getTransactionLedgerEntryDescription();

        if (desc == null) {
            return "X";
        }

        if (desc.startsWith(offsetDescription) && desc.contains("***")) {
            return "CS";
        }
        if (desc.startsWith(costShareDescription) && desc.contains("***")) {
            return "CS";
        }
        if (desc.startsWith(offsetDescription)) {
            return "O";
        }
        if (desc.startsWith(capitalizationDescription)) {
            return "C";
        }
        if (desc.startsWith(liabilityDescription)) {
            return "L";
        }
        if (desc.startsWith(transferDescription)) {
            return "T";
        }
        return "X";
    }

    /**
     * This will process a group of origin entries. The COBOL code was refactored a lot to get this so there isn't a
     * 1 to 1 section of Cobol relating to this.
     */
    protected void processGroup() {
        scrubCostShareAmount = KualiDecimal.ZERO;
        scrubberProcessUnitOfWork = new ScrubberProcessUnitOfWork();
        final FileReader INPUT_GLE_FILE;
        final BufferedReader INPUT_GLE_FILE_br;
        final PrintStream OUTPUT_GLE_FILE_ps;
        final PrintStream OUTPUT_ERR_FILE_ps;
        final PrintStream OUTPUT_EXP_FILE_ps;
        try {
            INPUT_GLE_FILE = new FileReader(inputFile, StandardCharsets.UTF_8);
        } catch (final IOException e) {
            throw new RuntimeException("Unable to find input file: " + inputFile, e);
        }
        try {
            OUTPUT_GLE_FILE_ps = new PrintStream(validFile);
            OUTPUT_ERR_FILE_ps = new PrintStream(errorFile);
            OUTPUT_EXP_FILE_ps = new PrintStream(expiredFile);
        } catch (final IOException e) {
            throw new RuntimeException("Problem opening output files", e);
        }

        INPUT_GLE_FILE_br = new BufferedReader(INPUT_GLE_FILE);
        LOG.debug("Starting Scrubber Process process group...");

        int lineNumber = 0;
        int loadedCount = 0;

        final LedgerSummaryReport laborLedgerSummaryReport = new LedgerSummaryReport();
        LaborOriginEntry unscrubbedEntry = new LaborOriginEntry();
        List<Message> tmperrors;
        try {
            String currentLine = INPUT_GLE_FILE_br.readLine();

            while (currentLine != null) {
                boolean saveErrorTransaction;
                boolean saveValidTransaction;
                final LaborOriginEntry scrubbedEntry = new LaborOriginEntry();
                try {
                    lineNumber++;

                    if (StringUtils.isNotEmpty(currentLine) && StringUtils.isNotBlank(currentLine.trim())) {
                        unscrubbedEntry = new LaborOriginEntry();
                        tmperrors = unscrubbedEntry.setFromTextFileForBatch(currentLine, lineNumber);
                        loadedCount++;

                        // just test entry with the entry loaded above
                        scrubberReport.incrementUnscrubbedRecordsRead();
                        final List<Message> transactionErrors = new ArrayList<>();

                        // This is done so if the code modifies this row, then saves it, it will be an insert,
                        // and it won't touch the original. The Scrubber never modifies input rows/groups.
                        unscrubbedEntry.setGroup(null);
                        unscrubbedEntry.setVersionNumber(null);
                        unscrubbedEntry.setEntryId(null);
                        saveErrorTransaction = false;
                        saveValidTransaction = false;

                        // Build a scrubbed entry
                        // Labor has more fields
                        buildScrubbedEntry(unscrubbedEntry, scrubbedEntry);

                        laborLedgerSummaryReport.summarizeEntry(unscrubbedEntry);

                        try {
                            tmperrors.addAll(scrubberValidator.validateTransaction(unscrubbedEntry, scrubbedEntry,
                                    universityRunDate, true, laborAccountingCycleCachingService));
                        } catch (final Exception e) {
                            transactionErrors.add(new Message(e.toString() + " occurred for this record.", Message.TYPE_FATAL));
                            saveValidTransaction = false;
                        }
                        transactionErrors.addAll(tmperrors);

                        // Expired account?
                        final Account unscrubbedEntryAccount = laborAccountingCycleCachingService.getAccount(
                                unscrubbedEntry.getChartOfAccountsCode(), unscrubbedEntry.getAccountNumber());
                        if (ObjectUtils.isNotNull(unscrubbedEntry.getAccount())
                                && (scrubberValidator.isAccountExpired(unscrubbedEntryAccount, universityRunDate)
                                || unscrubbedEntryAccount.isClosed())) {
                            // Make a copy of it so OJB doesn't just update the row in the original
                            // group. It needs to make a new one in the expired group
                            final LaborOriginEntry expiredEntry = new LaborOriginEntry(scrubbedEntry);

                            createOutputEntry(expiredEntry, OUTPUT_EXP_FILE_ps);
                            scrubberReport.incrementExpiredAccountFound();
                        }

                        if (!isFatal(transactionErrors)) {
                            saveValidTransaction = true;

                            // See if unit of work has changed
                            if (!scrubberProcessUnitOfWork.isSameUnitOfWork(scrubbedEntry)) {
                                // Generate offset for last unit of work
                                scrubberProcessUnitOfWork = new ScrubberProcessUnitOfWork(scrubbedEntry);
                            }
                            final KualiDecimal transactionAmount = scrubbedEntry.getTransactionLedgerEntryAmount();
                            final ParameterEvaluator offsetFiscalPeriods = parameterEvaluatorService
                                    .getParameterEvaluator(ScrubberStep.class,
                                            GLParameterConstants.FISCAL_PERIODS_EXCLUDED,
                                            scrubbedEntry.getUniversityFiscalPeriodCode());
                            final BalanceType scrubbedEntryBalanceType = laborAccountingCycleCachingService.getBalanceType(
                                    scrubbedEntry.getFinancialBalanceTypeCode());
                            if (scrubbedEntryBalanceType.isFinancialOffsetGenerationIndicator()
                                    && offsetFiscalPeriods.evaluationSucceeds()) {
                                if (scrubbedEntry.isDebit()) {
                                    scrubberProcessUnitOfWork.setOffsetAmount(
                                            scrubberProcessUnitOfWork.getOffsetAmount().add(transactionAmount));
                                } else {
                                    scrubberProcessUnitOfWork.setOffsetAmount(
                                            scrubberProcessUnitOfWork.getOffsetAmount().subtract(transactionAmount));
                                }
                            }

                            // The sub account type code will only exist if there is a valid sub account
                            // TODO: GLConstants.getSpaceSubAccountTypeCode();
                            String subAccountTypeCode = "  ";

                            final A21SubAccount scrubbedEntryA21SubAccount = laborAccountingCycleCachingService.
                                    getA21SubAccount(scrubbedEntry.getChartOfAccountsCode(),
                                            scrubbedEntry.getAccountNumber(), scrubbedEntry.getSubAccountNumber());
                            if (ObjectUtils.isNotNull(scrubbedEntryA21SubAccount)) {
                                subAccountTypeCode = scrubbedEntryA21SubAccount.getSubAccountTypeCode();
                            }

                            if (transactionErrors.size() > 0) {
                                laborMainReportWriterService.writeError(unscrubbedEntry, transactionErrors);
                            }
                        } else {
                            // Error transaction
                            saveErrorTransaction = true;
                            laborMainReportWriterService.writeError(unscrubbedEntry, transactionErrors);
                        }

                        if (saveValidTransaction) {
                            scrubbedEntry.setTransactionScrubberOffsetGenerationIndicator(false);
                            createOutputEntry(scrubbedEntry, OUTPUT_GLE_FILE_ps);
                            scrubberReport.incrementScrubbedRecordWritten();
                        }

                        if (saveErrorTransaction) {
                            // Make a copy of it so OJB doesn't just update the row in the original
                            // group. It needs to make a new one in the error group
                            final LaborOriginEntry errorEntry = new LaborOriginEntry(unscrubbedEntry);
                            errorEntry.setTransactionScrubberOffsetGenerationIndicator(false);
                            createOutputEntry(currentLine, OUTPUT_ERR_FILE_ps);
                            scrubberReport.incrementErrorRecordWritten();
                        }
                    }
                    currentLine = INPUT_GLE_FILE_br.readLine();

                } catch (final IOException ioe) {
                    // catch here again, it should be from postSingleEntryIntoLaborLedger
                    final int loggableLoadedCount = loadedCount;
                    LOG.error(
                            "processGroup() stopped due to: {} on line number : {}",
                            ioe::getMessage,
                            () -> loggableLoadedCount,
                            () -> ioe
                    );
                    throw new RuntimeException("processGroup() stopped due to: " + ioe.getMessage() +
                            " on line number : " + loadedCount, ioe);
                }
            }
            INPUT_GLE_FILE_br.close();
            INPUT_GLE_FILE.close();
            OUTPUT_GLE_FILE_ps.close();
            OUTPUT_ERR_FILE_ps.close();
            OUTPUT_EXP_FILE_ps.close();

            laborMainReportWriterService.writeStatisticLine("UNSCRUBBED RECORDS READ              %,9d",
                    scrubberReport.getNumberOfUnscrubbedRecordsRead());
            laborMainReportWriterService.writeStatisticLine("SCRUBBED RECORDS WRITTEN             %,9d",
                    scrubberReport.getNumberOfScrubbedRecordsWritten());
            laborMainReportWriterService.writeStatisticLine("ERROR RECORDS WRITTEN                %,9d",
                    scrubberReport.getNumberOfErrorRecordsWritten());
            laborMainReportWriterService.writeStatisticLine("TOTAL OUTPUT RECORDS WRITTEN         %,9d",
                    scrubberReport.getTotalNumberOfRecordsWritten());
            laborMainReportWriterService.writeStatisticLine("EXPIRED ACCOUNTS FOUND               %,9d",
                    scrubberReport.getNumberOfExpiredAccountsFound());

            laborLedgerSummaryReport.writeReport(laborLedgerReportWriterService);
        } catch (final IOException ioe) {
            LOG.error("processGroup() stopped due to: {}", ioe::getMessage, () -> ioe);
            throw new RuntimeException("processGroup() stopped due to: " + ioe.getMessage(), ioe);
        }
    }

    protected boolean isFatal(final List<Message> errors) {
        for (final Message element : errors) {
            if (element.getType() == Message.TYPE_FATAL) {
                return true;
            }
        }
        return false;
    }

    /**
     * Get all the transaction descriptions from the param table
     */
    protected void setDescriptions() {
        offsetDescription = kualiConfigurationService.getPropertyValueAsString(
                KFSKeyConstants.MSG_GENERATED_OFFSET);
        capitalizationDescription = kualiConfigurationService.getPropertyValueAsString(
                KFSKeyConstants.MSG_GENERATED_CAPITALIZATION);
        liabilityDescription = kualiConfigurationService.getPropertyValueAsString(
                KFSKeyConstants.MSG_GENERATED_LIABILITY);
        costShareDescription = kualiConfigurationService.getPropertyValueAsString(
                KFSKeyConstants.MSG_GENERATED_COST_SHARE);
        transferDescription = kualiConfigurationService.getPropertyValueAsString(
                KFSKeyConstants.MSG_GENERATED_TRANSFER);
    }

    /**
     * Generate the flag for the end of specific descriptions. This will be used in the demerger step
     */
    protected void setOffsetString() {
        final NumberFormat nf = NumberFormat.getInstance(Locale.US);
        nf.setMaximumFractionDigits(0);
        nf.setMaximumIntegerDigits(2);
        nf.setMinimumFractionDigits(0);
        nf.setMinimumIntegerDigits(2);

        offsetString = "***" + nf.format(runLocalDateTime.getMonthValue()) +
                nf.format(runLocalDateTime.getDayOfMonth());
    }

    /**
     * Generate the offset message with the flag at the end
     *
     * @return Offset message
     */
    protected String getOffsetMessage() {
        final String msg = offsetDescription + SPACES;
        return msg.substring(0, 33) + offsetString;
    }

    protected void setCutoffTimeForPreviousDay(final int hourOfDay, final int minuteOfDay, final int secondOfDay) {
        cutoffHour = hourOfDay;
        cutoffMinute = minuteOfDay;
        cutoffSecond = secondOfDay;

        LOG.info("Setting cutoff time to hour: {}, minute: {}, second: {}", hourOfDay, minuteOfDay, secondOfDay);
    }

    protected void setCutoffTime(String cutoffTime) {
        if (StringUtils.isBlank(cutoffTime)) {
            LOG.debug("Cutoff time is blank");
            unsetCutoffTimeForPreviousDay();
        } else {
            cutoffTime = cutoffTime.trim();
            LOG.debug("Cutoff time value found: {}", cutoffTime);
            final StringTokenizer st = new StringTokenizer(cutoffTime, ":", false);

            try {
                final String hourStr = st.nextToken();
                final String minuteStr = st.nextToken();
                final String secondStr = st.nextToken();

                final int hourInt = Integer.parseInt(hourStr, 10);
                final int minuteInt = Integer.parseInt(minuteStr, 10);
                final int secondInt = Integer.parseInt(secondStr, 10);

                if (hourInt < 0 || hourInt > 23 || minuteInt < 0 || minuteInt > 59 || secondInt < 0 || secondInt > 59) {
                    throw new IllegalArgumentException("Cutoff time must be in the format \"HH:mm:ss\", where " +
                            "HH, mm, ss are defined in the java.text.SimpleDateFormat class.  In particular, " +
                            "0 <= hour <= 23, 0 <= minute <= 59, and 0 <= second <= 59");
                }
                setCutoffTimeForPreviousDay(hourInt, minuteInt, secondInt);
            } catch (final Exception e) {
                throw new IllegalArgumentException("Cutoff time should either be null, or in the format " +
                        "\"HH:mm:ss\", where HH, mm, ss are defined in the java.text.SimpleDateFormat class.", e);
            }
        }
    }

    public void unsetCutoffTimeForPreviousDay() {
        cutoffHour = null;
        cutoffMinute = null;
        cutoffSecond = null;
    }

    /**
     * This method modifies the run date if it is before the cutoff time specified by calling the
     * setCutoffTimeForPreviousDay method. See KULRNE-70 This method is public to facilitate unit testing
     *
     * @param currentDate
     * @return
     */
    public java.sql.Date calculateRunDate(final java.util.Date currentDate) {
        final LocalDateTime currentLocalDateTime = dateTimeService.getLocalDateTime(currentDate);

        if (isCurrentDateBeforeCutoff(currentLocalDateTime)) {
            // time to set the date to the previous day's last minute/second
            // per old COBOL code (see KULRNE-70),
            // the time is set to 23:59:59 (assuming 0 ms)
            final LocalDateTime endOfDayBefore = currentLocalDateTime.minusDays(1)
                    .withHour(23)
                    .withMinute(59)
                    .withSecond(59)
                    .withNano(0);
            return new java.sql.Date(dateTimeService.getLocalDateTimeMillis(endOfDayBefore));
        }
        return new java.sql.Date(currentDate.getTime());
    }

    protected boolean isCurrentDateBeforeCutoff(final LocalDateTime currentLocalDateTime) {
        if (cutoffHour != null && cutoffMinute != null && cutoffSecond != null) {
            // if cutoff date is not properly defined 24 hour clock (i.e. hour is 0 - 23)

            final LocalDateTime cutoffTime = currentLocalDateTime
                    .withHour(cutoffHour)
                    .withMinute(cutoffMinute)
                    .withSecond(cutoffSecond)
                    .withNano(0);

            return currentLocalDateTime.isBefore(cutoffTime);
        }
        // if cutoff date is not properly defined, then it is considered to be after the cutoff
        return false;
    }

    protected void initCutoffTime() {
        final String cutoffTime = parameterService.getParameterValueAsString(ScrubberStep.class,
                GLParameterConstants.SCRUBBER_CUTOFF_TIME);
        if (StringUtils.isBlank(cutoffTime)) {
            LOG.debug("Cutoff time system parameter not found");
            unsetCutoffTimeForPreviousDay();
            return;
        }
        setCutoffTime(cutoffTime);
    }

    protected void buildScrubbedEntry(final LaborOriginEntry unscrubbedEntry, final LaborOriginEntry scrubbedEntry) {
        scrubbedEntry.setDocumentNumber(unscrubbedEntry.getDocumentNumber());
        scrubbedEntry.setOrganizationDocumentNumber(unscrubbedEntry.getOrganizationDocumentNumber());
        scrubbedEntry.setOrganizationReferenceId(unscrubbedEntry.getOrganizationReferenceId());
        scrubbedEntry.setReferenceFinancialDocumentNumber(unscrubbedEntry.getReferenceFinancialDocumentNumber());

        final Integer transactionNumber = unscrubbedEntry.getTransactionLedgerEntrySequenceNumber();
        scrubbedEntry.setTransactionLedgerEntrySequenceNumber(null == transactionNumber ? Integer.valueOf(0) :
                transactionNumber);
        scrubbedEntry.setTransactionLedgerEntryDescription(unscrubbedEntry.getTransactionLedgerEntryDescription());
        scrubbedEntry.setTransactionLedgerEntryAmount(unscrubbedEntry.getTransactionLedgerEntryAmount());
        scrubbedEntry.setTransactionDebitCreditCode(unscrubbedEntry.getTransactionDebitCreditCode());

        // For Labor's more fields
        // It might be changed based on Labor Scrubber's business rule
        scrubbedEntry.setPositionNumber(unscrubbedEntry.getPositionNumber());
        scrubbedEntry.setTransactionPostingDate(unscrubbedEntry.getTransactionPostingDate());
        scrubbedEntry.setPayPeriodEndDate(unscrubbedEntry.getPayPeriodEndDate());
        scrubbedEntry.setTransactionTotalHours(unscrubbedEntry.getTransactionTotalHours());
        scrubbedEntry.setPayrollEndDateFiscalYear(unscrubbedEntry.getPayrollEndDateFiscalYear());
        scrubbedEntry.setPayrollEndDateFiscalPeriodCode(unscrubbedEntry.getPayrollEndDateFiscalPeriodCode());
        scrubbedEntry.setFinancialDocumentApprovedCode(unscrubbedEntry.getFinancialDocumentApprovedCode());
        scrubbedEntry.setTransactionEntryOffsetCode(unscrubbedEntry.getTransactionEntryOffsetCode());
        scrubbedEntry.setTransactionEntryProcessedTimestamp(unscrubbedEntry.getTransactionEntryProcessedTimestamp());
        scrubbedEntry.setEmplid(unscrubbedEntry.getEmplid());
        scrubbedEntry.setEmployeeRecord(unscrubbedEntry.getEmployeeRecord());
        scrubbedEntry.setEarnCode(unscrubbedEntry.getEarnCode());
        scrubbedEntry.setPayGroup(unscrubbedEntry.getPayGroup());
        scrubbedEntry.setSalaryAdministrationPlan(unscrubbedEntry.getSalaryAdministrationPlan());
        scrubbedEntry.setGrade(unscrubbedEntry.getGrade());
        scrubbedEntry.setRunIdentifier(unscrubbedEntry.getRunIdentifier());
        scrubbedEntry.setLaborLedgerOriginalChartOfAccountsCode(
                unscrubbedEntry.getLaborLedgerOriginalChartOfAccountsCode());
        scrubbedEntry.setLaborLedgerOriginalAccountNumber(unscrubbedEntry.getLaborLedgerOriginalAccountNumber());
        scrubbedEntry.setLaborLedgerOriginalSubAccountNumber(unscrubbedEntry.getLaborLedgerOriginalSubAccountNumber());
        scrubbedEntry.setLaborLedgerOriginalFinancialObjectCode(
                unscrubbedEntry.getLaborLedgerOriginalFinancialObjectCode());
        scrubbedEntry.setLaborLedgerOriginalFinancialSubObjectCode(
                unscrubbedEntry.getLaborLedgerOriginalFinancialSubObjectCode());
        scrubbedEntry.setHrmsCompany(unscrubbedEntry.getHrmsCompany());
        scrubbedEntry.setSetid(unscrubbedEntry.getSetid());
        scrubbedEntry.setTransactionDateTimeStamp(unscrubbedEntry.getTransactionDateTimeStamp());
        scrubbedEntry.setReferenceFinancialDocumentTypeCode(unscrubbedEntry.getReferenceFinancialDocumentTypeCode());
        scrubbedEntry.setReferenceFinancialSystemOrigination(unscrubbedEntry.getReferenceFinancialSystemOrigination());
        scrubbedEntry.setPayrollEndDateFiscalPeriod(unscrubbedEntry.getPayrollEndDateFiscalPeriod());
    }

    /**
     * The demerger process reads all of the documents in the error group, then moves all of the original entries for
     * that document from the valid group to the error group. It does not move generated entries to the error group.
     * Those are deleted. It also modifies the doc number and origin code of cost share transfers.
     */
    public void performDemerger() {
        LOG.debug("performDemerger() started");
        final LaborOriginEntryFieldUtil loefu = new LaborOriginEntryFieldUtil();
        final Map<String, Integer> pMap = loefu.getFieldBeginningPositionMap();

        final String validOutputFilename = batchFileDirectoryName + File.separator +
                                           LaborConstants.BatchFileSystem.SCRUBBER_VALID_OUTPUT_FILE +
                                           GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        final String errorOutputFilename = batchFileDirectoryName + File.separator +
                                           LaborConstants.BatchFileSystem.SCRUBBER_ERROR_SORTED_FILE +
                                           GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        runDate = calculateRunDate(dateTimeService.getCurrentDate());

        final DemergerReportData demergerReport = new DemergerReportData();

        final OriginEntryStatistics eOes = laborOriginEntryService.getStatistics(errorOutputFilename);
        demergerReport.setErrorTransactionsRead(eOes.getRowCount());

        // Get the list of document type codes from the parameter.  If the current document type matches any of
        // parameter defined type codes, then demerge all other entries for this document; otherwise, only pull the
        // entry with the error and don't demerge anything else.
        final Collection<String> demergeDocumentTypes = parameterService.getParameterValuesAsString(LaborDemergerStep.class,
                LaborParameterConstants.DOCUMENT_TYPES
        );

        // Read all the documents from the error group and move all non-generated transactions for these documents
        // from the valid group into the error group

        final String demergerValidOutputFilename = batchFileDirectoryName + File.separator +
                                                   LaborConstants.BatchFileSystem.DEMERGER_VALID_OUTPUT_FILE +
                                                   GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        final String demergerErrorOutputFilename = batchFileDirectoryName + File.separator +
                                                   LaborConstants.BatchFileSystem.DEMERGER_ERROR_OUTPUT_FILE +
                                                   GeneralLedgerConstants.BatchFileSystem.EXTENSION;

        int validRead = 0;
        int errorRead = 0;

        int validSaved = 0;
        int errorSaved = 0;

        try (
                BufferedReader inputGleFile =
                        new BufferedReader(new FileReader(validOutputFilename, StandardCharsets.UTF_8));
                BufferedReader inputErrFile =
                        new BufferedReader(new FileReader(errorOutputFilename, StandardCharsets.UTF_8));
                PrintStream outputDemergerGleFile = new PrintStream(demergerValidOutputFilename);
                PrintStream outputDemergerErrFile = new PrintStream(demergerErrorOutputFilename)
        ) {
            String currentValidLine = inputGleFile.readLine();
            String currentErrorLine = inputErrFile.readLine();

            while (currentValidLine != null || currentErrorLine != null) {
                // validLine is null means that errorLine is not null
                if (StringUtils.isEmpty(currentValidLine)) {
                    createOutputEntry(currentErrorLine, outputDemergerErrFile);
                    currentErrorLine = inputErrFile.readLine();
                    errorRead++;
                    errorSaved++;
                    continue;
                }

                // errorLine is null means that validLine is not null
                if (StringUtils.isEmpty(currentErrorLine)) {
                    createOutputEntry(currentValidLine, outputDemergerGleFile);
                    currentValidLine = inputGleFile.readLine();
                    validRead++;
                    validSaved++;
                    continue;
                }

                String documentTypeCode = currentErrorLine.substring(
                        pMap.get(KFSPropertyConstants.FINANCIAL_DOCUMENT_TYPE_CODE),
                        pMap.get(KFSPropertyConstants.FINANCIAL_SYSTEM_ORIGINATION_CODE));
                if (documentTypeCode != null) {
                    documentTypeCode = documentTypeCode.trim();
                }

                if (demergeDocumentTypes.contains(documentTypeCode)) {
                    final String compareStringFromValidEntry = currentValidLine.substring(
                            pMap.get(KFSPropertyConstants.FINANCIAL_DOCUMENT_TYPE_CODE),
                            pMap.get(KFSPropertyConstants.TRANSACTION_ENTRY_SEQUENCE_NUMBER));
                    final String compareStringFromErrorEntry = currentErrorLine.substring(
                            pMap.get(KFSPropertyConstants.FINANCIAL_DOCUMENT_TYPE_CODE),
                            pMap.get(KFSPropertyConstants.TRANSACTION_ENTRY_SEQUENCE_NUMBER));

                    if (compareStringFromValidEntry.compareTo(compareStringFromErrorEntry) < 0) {
                        createOutputEntry(currentValidLine, outputDemergerGleFile);
                        currentValidLine = inputGleFile.readLine();
                        validRead++;
                        validSaved++;
                    } else if (compareStringFromValidEntry.compareTo(compareStringFromErrorEntry) > 0) {
                        createOutputEntry(currentErrorLine, outputDemergerErrFile);
                        currentErrorLine = inputErrFile.readLine();
                        errorRead++;
                        errorSaved++;
                    } else {
                        createOutputEntry(currentValidLine, outputDemergerErrFile);
                        currentValidLine = inputGleFile.readLine();
                        validRead++;
                        errorSaved++;
                    }
                    continue;
                }
                createOutputEntry(currentErrorLine, outputDemergerErrFile);
                currentErrorLine = inputErrFile.readLine();
                errorRead++;
                errorSaved++;
            }

        } catch (final Exception e) {
            LOG.error("performDemerger() stopped due to: {}", e::getMessage, () -> e);
            throw new RuntimeException("performDemerger() stopped due to: " + e.getMessage(), e);
        }

        demergerReport.setValidTransactionsRead(validRead);
        demergerReport.setValidTransactionsSaved(validSaved);
        demergerReport.setErrorTransactionsRead(errorRead);
        demergerReport.setErrorTransactionWritten(errorSaved);

        laborDemergerReportWriterService.writeStatisticLine("SCRUBBER ERROR TRANSACTIONS READ       %,9d",
                demergerReport.getErrorTransactionsRead());
        laborDemergerReportWriterService.writeStatisticLine("SCRUBBER VALID TRANSACTIONS READ       %,9d",
                demergerReport.getValidTransactionsRead());
        laborDemergerReportWriterService.writeStatisticLine("DEMERGER ERRORS SAVED                  %,9d",
                demergerReport.getErrorTransactionsSaved());
        laborDemergerReportWriterService.writeStatisticLine("DEMERGER VALID TRANSACTIONS SAVED      %,9d",
                demergerReport.getValidTransactionsSaved());

        generateScrubberErrorListingReport(demergerErrorOutputFilename);
    }

    protected void createOutputEntry(final LaborOriginEntry entry, final PrintStream ps) throws IOException {
        try {
            ps.printf("%s\n", entry.getLine());
        } catch (final Exception e) {
            throw new IOException(e.toString(), e);
        }
    }

    protected void createOutputEntry(final String line, final PrintStream ps) throws IOException {
        try {
            ps.printf("%s\n", line);
        } catch (final Exception e) {
            throw new IOException(e.toString(), e);
        }
    }

    protected boolean checkEntry(final LaborOriginEntry validEntry, final LaborOriginEntry errorEntry, final String documentTypeCode) {
        final String documentNumber = errorEntry.getDocumentNumber();
        final String originationCode = errorEntry.getFinancialSystemOriginationCode();

        return validEntry.getDocumentNumber().equals(documentNumber)
                && validEntry.getFinancialDocumentTypeCode().equals(documentTypeCode)
                && validEntry.getFinancialSystemOriginationCode().equals(originationCode);
    }

    /**
     * Generates a transaction listing report for labor origin entries that were valid
     */
    protected void generateScrubberTransactionsOnline() {
        try {
            final Iterator<LaborOriginEntry> generatedTransactions = new LaborOriginEntryFileIterator(new File(inputFile));

            ((WrappingBatchService) laborGeneratedTransactionsReportWriterService).initialize();
            new TransactionListingReport().generateReport(laborGeneratedTransactionsReportWriterService,
                    generatedTransactions);
        } finally {
            ((WrappingBatchService) laborGeneratedTransactionsReportWriterService).destroy();
        }
    }

    /**
     * Generates a transaction listing report for labor origin entries with bad balance types
     */
    protected void generateScrubberBadBalanceTypeListingReport() {
        final LaborOriginEntryFilter blankBalanceTypeFilter = originEntry -> {
            final BalanceType originEntryBalanceType =
                    laborAccountingCycleCachingService.getBalanceType(originEntry.getFinancialBalanceTypeCode());
            return ObjectUtils.isNull(originEntryBalanceType);
        };
        try {
            ((WrappingBatchService) laborBadBalanceTypeReportWriterService).initialize();
            final Iterator<LaborOriginEntry> blankBalanceOriginEntries = new FilteringLaborOriginEntryFileIterator(
                    new File(inputFile), blankBalanceTypeFilter);
            new TransactionListingReport().generateReport(laborBadBalanceTypeReportWriterService,
                    blankBalanceOriginEntries);
        } finally {
            ((WrappingBatchService) laborBadBalanceTypeReportWriterService).destroy();
        }
    }

    /**
     * Generates a transaction listing report for labor origin entries with errors
     */
    protected void generateScrubberErrorListingReport(final String errorFileName) {
        try {
            ((WrappingBatchService) laborErrorListingReportWriterService).initialize();
            final Iterator<LaborOriginEntry> removedTransactions = new LaborOriginEntryFileIterator(new File(errorFileName));
            new TransactionListingReport().generateReport(laborErrorListingReportWriterService, removedTransactions);
        } finally {
            ((WrappingBatchService) laborErrorListingReportWriterService).destroy();
        }
    }
}
