/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.coa.businessobject.SubAccount;
import org.kuali.kfs.coa.service.AccountService;
import org.kuali.kfs.coa.service.BalanceTypeService;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.gl.batch.service.AccountingCycleCachingService;
import org.kuali.kfs.gl.businessobject.OriginEntryFull;
import org.kuali.kfs.gl.businessobject.OriginEntryInformation;
import org.kuali.kfs.gl.service.ScrubberValidator;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.PersistenceService;
import org.kuali.kfs.krad.service.PersistenceStructureService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.LaborKeyConstants;
import org.kuali.kfs.module.ld.LaborParameterConstants;
import org.kuali.kfs.module.ld.batch.LaborScrubberStep;
import org.kuali.kfs.module.ld.batch.service.LaborAccountingCycleCachingService;
import org.kuali.kfs.module.ld.businessobject.LaborObject;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.Message;
import org.kuali.kfs.sys.service.MessageBuilderService;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntry;
import org.kuali.kfs.sys.businessobject.SystemOptions;
import org.kuali.kfs.sys.businessobject.UniversityDate;
import org.kuali.kfs.sys.service.OptionsService;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Service implementation of ScrubberValidator.
 */
public class ScrubberValidatorImpl implements ScrubberValidator {

    private static final Logger LOG = LogManager.getLogger();

    protected ConfigurationService kualiConfigurationService;
    protected BusinessObjectService businessObjectService;
    protected ParameterService parameterService;
    protected AccountService accountService;
    protected BalanceTypeService balanceTypService;
    protected OptionsService optionsService;

    protected PersistenceService persistenceService;
    protected ScrubberValidator scrubberValidator;
    protected PersistenceStructureService persistenceStructureService;
    private MessageBuilderService messageBuilderService;
    protected boolean continuationAccountIndicator;

    @Override
    public List<Message> validateTransaction(
            final OriginEntryInformation originEntry, final OriginEntryInformation scrubbedEntry,
            final UniversityDate universityRunDate, final boolean laborIndicator,
            final AccountingCycleCachingService laborAccountingCycleCachingService) {
        LOG.debug("validateTransaction() started");
        continuationAccountIndicator = false;

        final LaborOriginEntry laborOriginEntry = (LaborOriginEntry) originEntry;
        final LaborOriginEntry laborScrubbedEntry = (LaborOriginEntry) scrubbedEntry;

        // gl scrubber validation
        final List<Message> errors = scrubberValidator.validateTransaction(laborOriginEntry, laborScrubbedEntry,
                universityRunDate, laborIndicator, laborAccountingCycleCachingService);

        refreshOriginEntryReferences(laborOriginEntry);
        refreshOriginEntryReferences(laborScrubbedEntry);

        if (StringUtils.isBlank(laborOriginEntry.getEmplid())) {
            laborScrubbedEntry.setEmplid(LaborConstants.getDashEmplId());
        }

        if (StringUtils.isBlank(laborOriginEntry.getPositionNumber())) {
            laborScrubbedEntry.setPositionNumber(LaborConstants.getDashPositionNumber());
        }

        Message err = validatePayrollEndFiscalYear(laborOriginEntry, laborScrubbedEntry, universityRunDate,
                (LaborAccountingCycleCachingService) laborAccountingCycleCachingService);
        if (err != null) {
            errors.add(err);
        }

        err = validatePayrollEndFiscalPeriodCode(laborOriginEntry, laborScrubbedEntry, universityRunDate,
                (LaborAccountingCycleCachingService) laborAccountingCycleCachingService);
        if (err != null) {
            errors.add(err);
        }

        err = validateAccount(laborOriginEntry, laborScrubbedEntry, universityRunDate,
                (LaborAccountingCycleCachingService) laborAccountingCycleCachingService);
        if (err != null) {
            errors.add(err);
        }

        err = validateSubAccount(laborOriginEntry, laborScrubbedEntry,
                (LaborAccountingCycleCachingService) laborAccountingCycleCachingService);
        if (err != null) {
            errors.add(err);
        }

        return errors;
    }

    /**
     * This method is for refreshing References of Origin Entry
     */
    protected void refreshOriginEntryReferences(final OriginEntryFull originEntry) {
        final Map<String, Class> referenceClasses =
                persistenceStructureService.listReferenceObjectFields(originEntry.getClass());
        for (final String reference : referenceClasses.keySet()) {
            if (KFSPropertyConstants.PROJECT.equals(reference)) {
                if (KFSConstants.getDashProjectCode().equals(originEntry.getProjectCode())) {
                    originEntry.setProject(null);
                } else {
                    persistenceService.retrieveReferenceObject(originEntry, reference);
                }
            } else if (KFSPropertyConstants.FINANCIAL_SUB_OBJECT.equals(reference)) {
                if (KFSConstants.getDashFinancialSubObjectCode().equals(originEntry.getFinancialSubObjectCode())) {
                    originEntry.setFinancialSubObject(null);
                } else {
                    persistenceService.retrieveReferenceObject(originEntry, reference);
                }
            } else if (KFSPropertyConstants.SUB_ACCOUNT.equals(reference)) {
                if (KFSConstants.getDashSubAccountNumber().equals(originEntry.getSubAccountNumber())) {
                    originEntry.setSubAccount(null);
                } else {
                    persistenceService.retrieveReferenceObject(originEntry, reference);
                }
            } else {
                persistenceService.retrieveReferenceObject(originEntry, reference);
            }
        }
    }

    /**
     * This method is for validation of payrollEndFiscalYear
     */
    protected Message validatePayrollEndFiscalYear(
            final LaborOriginEntry laborOriginEntry,
            final LaborOriginEntry laborWorkingEntry, final UniversityDate universityRunDate,
            final LaborAccountingCycleCachingService laborAccountingCycleCachingService) {
        LOG.debug("validatePayrollEndFiscalYear() started");
        final SystemOptions scrubbedEntryOption;
        if (laborOriginEntry.getPayrollEndDateFiscalYear() != null) {
            scrubbedEntryOption = laborAccountingCycleCachingService.getSystemOptions(
                    laborOriginEntry.getPayrollEndDateFiscalYear());

            if (scrubbedEntryOption == null) {
                return messageBuilderService.buildMessage(
                        LaborKeyConstants.ERROR_PAYROLL_END_DATE_FISCAL_YEAR,
                        "" + laborOriginEntry.getPayrollEndDateFiscalYear(),
                        Message.TYPE_FATAL
                );
            }
        }

        return null;
    }

    /**
     * This method is for validation of PayrollEndFiscalPeriodCode
     */
    protected Message validatePayrollEndFiscalPeriodCode(
            final LaborOriginEntry laborOriginEntry,
            final LaborOriginEntry laborWorkingEntry, final UniversityDate universityRunDate,
            final LaborAccountingCycleCachingService laborAccountingCycleCachingService) {
        LOG.debug("validateUniversityFiscalPeriodCode() started");

        final AccountingPeriod accountingPeriod;
        final Integer tempPayrollFiscalYear;
        if (laborOriginEntry.getPayrollEndDateFiscalYear() == null) {
            tempPayrollFiscalYear = universityRunDate.getUniversityFiscalYear();
        } else {
            tempPayrollFiscalYear = laborOriginEntry.getPayrollEndDateFiscalYear();
        }

        if (!laborOriginEntry.getPayrollEndDateFiscalPeriodCode().equals("")) {
            accountingPeriod = laborAccountingCycleCachingService.getAccountingPeriod(tempPayrollFiscalYear,
                    laborOriginEntry.getPayrollEndDateFiscalPeriodCode());
            if (accountingPeriod == null) {
                return messageBuilderService.buildMessage(
                        KFSKeyConstants.ERROR_PAYROLL_END_DATE_FISCAL_PERIOD,
                        laborOriginEntry.getPayrollEndDateFiscalPeriodCode(),
                        Message.TYPE_FATAL
                );
            }
        }

        return null;
    }

    /**
     * Performs Account Validation.
     */
    protected Message validateAccount(
            final LaborOriginEntry laborOriginEntry, final LaborOriginEntry laborWorkingEntry,
            final UniversityDate universityRunDate, final LaborAccountingCycleCachingService laborAccountingCycleCachingService) {
        LOG.debug("validateAccount() started");

        final Account account = laborOriginEntry.getAccount();
        final boolean suspenseAccountLogicInd = parameterService.getParameterValueAsBoolean(LaborScrubberStep.class,
                LaborParameterConstants.SUSPENSE_ACCOUNT_IND
        );
        if (ObjectUtils.isNull(account)) {
            if (suspenseAccountLogicInd) {
                return useSuspenseAccount(laborWorkingEntry);
            }
            return messageBuilderService.buildMessage(
                    KFSKeyConstants.ERROR_ACCOUNT_NOT_FOUND,
                    laborOriginEntry.getChartOfAccountsCode() + "-" + laborOriginEntry.getAccountNumber(),
                    Message.TYPE_FATAL
            );
        }

        // default
        laborWorkingEntry.setAccount(account);
        laborWorkingEntry.setChartOfAccountsCode(account.getChartOfAccountsCode());
        laborWorkingEntry.setAccountNumber(account.getAccountNumber());

        // no further validation for gl annual doc type
        final String glAnnualClosingType = parameterService.getParameterValueAsString(
                KfsParameterConstants.GENERAL_LEDGER_BATCH.class,
                KFSConstants.SystemGroupParameterNames.GL_ANNUAL_CLOSING_DOC_TYPE);
        if (glAnnualClosingType.equals(laborOriginEntry.getFinancialDocumentTypeCode())) {
            return null;
        }

        // Sub-Fund Wage Exclusion
        final String originationCode = laborOriginEntry.getFinancialSystemOriginationCode();
        final List<String> nonWageSubfundBypassOriginationCodes = new ArrayList<>(
                parameterService.getParameterValuesAsString(
                        KFSConstants.OptionalModuleNamespaces.LABOR_DISTRIBUTION,
                        LaborParameterConstants.Components.LABOR_SCRUBBER_WAGE_EXCLUSION,
                        LaborParameterConstants.ORIGINATION_CODES_EXCLUDE
                ));
        final boolean subfundWageExclusionInd = parameterService.getParameterValueAsBoolean(
                KFSConstants.OptionalModuleNamespaces.LABOR_DISTRIBUTION,
                LaborParameterConstants.Components.LABOR_SCRUBBER_WAGE_EXCLUSION,
                LaborParameterConstants.WAGE_EXCLUSION_IND
        );

        if (subfundWageExclusionInd && !account.getSubFundGroup().isSubFundGroupWagesIndicator()
                && !nonWageSubfundBypassOriginationCodes.contains(originationCode)) {
            if (suspenseAccountLogicInd) {
                return useSuspenseAccount(laborWorkingEntry);
            }

            return messageBuilderService.buildMessage(
                    LaborKeyConstants.ERROR_SUN_FUND_NOT_ACCEPT_WAGES,
                    Message.TYPE_FATAL
            );
        }

        // Account Fringe Validation
        final Collection<String> nonFringeAccountBypassOriginationCodes =
                parameterService.getParameterValuesAsString(
                        KFSConstants.OptionalModuleNamespaces.LABOR_DISTRIBUTION,
                        LaborParameterConstants.Components.LABOR_SCRUBBER_BENEFITS_ACCOUNT,
                        LaborParameterConstants.ORIGINATION_CODES
                );
        final boolean accountFringeExclusionInd = parameterService.getParameterValueAsBoolean(
                KFSConstants.OptionalModuleNamespaces.LABOR_DISTRIBUTION,
                LaborParameterConstants.Components.LABOR_SCRUBBER_BENEFITS_ACCOUNT,
                LaborParameterConstants.BENEFITS_ACCOUNT_IND
        );

        if (accountFringeExclusionInd && !nonFringeAccountBypassOriginationCodes.contains(originationCode)) {
            return checkAccountFringeIndicator(laborOriginEntry, laborWorkingEntry, account, universityRunDate,
                    laborAccountingCycleCachingService);
        }

        // Expired/Closed Validation
        return handleExpiredClosedAccount(laborOriginEntry.getAccount(), laborOriginEntry, laborWorkingEntry,
                universityRunDate);
    }

    /**
     * Checks the continuation account system indicator. If on checks whether the account is expired or closed, and if
     * so calls the continuation logic.
     */
    protected Message handleExpiredClosedAccount(
            final Account account, final LaborOriginEntry laborOriginEntry,
            final LaborOriginEntry laborWorkingEntry, final UniversityDate universityRunDate) {
        final List<String> continuationAccountBypassBalanceTypeCodes = balanceTypService
                .getContinuationAccountBypassBalanceTypeCodes(universityRunDate.getUniversityFiscalYear());
        final List<String> continuationAccountBypassOriginationCodes = new ArrayList<>(
                parameterService.getParameterValuesAsString(KFSConstants.OptionalModuleNamespaces.LABOR_DISTRIBUTION,
                        LaborParameterConstants.Components.LABOR_SCRUBBER_CONTINUATION_ACCOUNT,
                        LaborParameterConstants.ORIGINATION_CODES
                ));
        final List<String> continuationAccountBypassDocumentTypeCodes = new ArrayList<>(
                parameterService.getParameterValuesAsString(KFSConstants.OptionalModuleNamespaces.LABOR_DISTRIBUTION,
                        LaborParameterConstants.Components.LABOR_SCRUBBER_CONTINUATION_ACCOUNT,
                        LaborParameterConstants.DOCUMENT_TYPES
                ));

        final boolean isAccountExpiredOrClosed = account.getAccountExpirationDate() != null
                                                 && isAccountExpired(account, universityRunDate) || !account.isActive();
        final boolean continuationAccountLogicInd = parameterService.getParameterValueAsBoolean(
                KFSConstants.OptionalModuleNamespaces.LABOR_DISTRIBUTION,
                LaborParameterConstants.Components.LABOR_SCRUBBER_CONTINUATION_ACCOUNT,
                LaborParameterConstants.CONTINUATION_ACCOUNT_IND
        );

        if (continuationAccountLogicInd && isAccountExpiredOrClosed) {
            // special checks for origination codes that have override ability
            final boolean isOverrideOriginCode = continuationAccountBypassOriginationCodes.contains(
                    laborOriginEntry.getFinancialSystemOriginationCode());
            if (isOverrideOriginCode && !account.isActive()) {
                return messageBuilderService.buildMessage(
                        KFSKeyConstants.ERROR_ORIGIN_CODE_CANNOT_HAVE_CLOSED_ACCOUNT,
                        laborOriginEntry.getChartOfAccountsCode() + "-" + laborOriginEntry.getAccountNumber(),
                        Message.TYPE_FATAL
                );
            }

            final boolean canBypass = isOverrideOriginCode
                                      || continuationAccountBypassBalanceTypeCodes.contains(
                            laborOriginEntry.getFinancialBalanceTypeCode())
                                      || continuationAccountBypassDocumentTypeCodes.contains(
                            laborOriginEntry.getFinancialDocumentTypeCode().trim());
            if (account.isActive() && canBypass) {
                return null;
            }

            return continuationAccountLogic(account, laborOriginEntry, laborWorkingEntry, universityRunDate);
        }

        return null;
    }

    /**
     * Loops through continuation accounts for 10 tries or until it finds an account that is not expired.
     */
    protected Message continuationAccountLogic(
            final Account expiredClosedAccount, final LaborOriginEntry laborOriginEntry,
            final LaborOriginEntry laborWorkingEntry, final UniversityDate universityRunDate) {
        String chartCode = expiredClosedAccount.getContinuationFinChrtOfAcctCd();
        String accountNumber = expiredClosedAccount.getContinuationAccountNumber();

        final List<String> checkedAccountNumbers = new ArrayList<>();
        for (int i = 0; i < 10; ++i) {
            if (checkedAccountNumbers.contains(chartCode + accountNumber)) {
                // Something is really wrong with the data because this account has already been evaluated.
                return messageBuilderService.buildMessage(
                        KFSKeyConstants.ERROR_CIRCULAR_DEPENDENCY_IN_CONTINUATION_ACCOUNT_LOGIC,
                        Message.TYPE_FATAL
                );
            }

            checkedAccountNumbers.add(chartCode + accountNumber);

            if (chartCode == null || accountNumber == null) {
                return messageBuilderService.buildMessage(
                        KFSKeyConstants.ERROR_CONTINUATION_ACCOUNT_NOT_FOUND,
                        Message.TYPE_FATAL
                );
            }

            // Lookup the account
            final Account account = accountService.getByPrimaryId(chartCode, accountNumber);
            if (ObjectUtils.isNull(account)) {
                return messageBuilderService.buildMessage(
                        KFSKeyConstants.ERROR_CONTINUATION_ACCOUNT_NOT_FOUND,
                        Message.TYPE_FATAL
                );
            }

            // check account expiration
            if (ObjectUtils.isNotNull(account.getAccountExpirationDate())
                    && isAccountExpired(account, universityRunDate)) {
                chartCode = account.getContinuationFinChrtOfAcctCd();
                accountNumber = account.getContinuationAccountNumber();
            } else {
                // set continuationAccountLogicIndi
                continuationAccountIndicator = true;

                laborWorkingEntry.setAccount(account);
                laborWorkingEntry.setAccountNumber(accountNumber);
                laborWorkingEntry.setChartOfAccountsCode(chartCode);
                laborWorkingEntry.setSubAccountNumber(KFSConstants.getDashSubAccountNumber());
                laborWorkingEntry.setTransactionLedgerEntryDescription(
                        kualiConfigurationService.getPropertyValueAsString(KFSKeyConstants.MSG_AUTO_FORWARD) + " " +
                                expiredClosedAccount.getChartOfAccountsCode() + expiredClosedAccount.getAccountNumber() +
                                laborOriginEntry.getTransactionLedgerEntryDescription());

                return messageBuilderService.buildMessage(
                        KFSKeyConstants.MSG_ACCOUNT_CLOSED_TO,
                        laborWorkingEntry.getChartOfAccountsCode() + "-" + laborWorkingEntry.getAccountNumber(),
                        Message.TYPE_WARNING
                );
            }
        }

        // We failed to find a valid continuation account.
        final boolean suspenseAccountLogicInd = parameterService.getParameterValueAsBoolean(LaborScrubberStep.class,
                LaborParameterConstants.SUSPENSE_ACCOUNT_IND
        );
        if (suspenseAccountLogicInd) {
            return useSuspenseAccount(laborWorkingEntry);
        } else {
            return messageBuilderService.buildMessage(
                    KFSKeyConstants.ERROR_CONTINUATION_ACCOUNT_LIMIT_REACHED,
                    Message.TYPE_FATAL
            );
        }
    }

    /**
     * For fringe transaction types checks if the account accepts fringe benefits. If not, retrieves the alternative
     * account, then calls expiration checking on either the alternative account or the account passed in.
     */
    protected Message checkAccountFringeIndicator(
            final LaborOriginEntry laborOriginEntry,
            final LaborOriginEntry laborWorkingEntry, final Account account, final UniversityDate universityRunDate,
            final LaborAccountingCycleCachingService laborAccountingCycleCachingService) {
        // check for fringe transaction type
        final LaborObject laborObject = laborAccountingCycleCachingService.getLaborObject(
                laborOriginEntry.getUniversityFiscalYear(), laborOriginEntry.getChartOfAccountsCode(),
                laborOriginEntry.getFinancialObjectCode());
        final boolean isFringeTransaction = laborObject != null
                                            && StringUtils.equals(LaborConstants.BenefitExpenseTransfer.LABOR_LEDGER_BENEFIT_CODE,
                    laborObject.getFinancialObjectFringeOrSalaryCode());

        // alternative account handling for non fringe accounts
        if (isFringeTransaction && !account.isAccountsFringesBnftIndicator()) {
            final Account altAccount = accountService.getByPrimaryId(
                    laborOriginEntry.getAccount().getReportsToChartOfAccountsCode(),
                    laborOriginEntry.getAccount().getReportsToAccountNumber());
            if (ObjectUtils.isNotNull(altAccount)) {
                laborWorkingEntry.setAccount(altAccount);
                laborWorkingEntry.setAccountNumber(altAccount.getAccountNumber());
                laborWorkingEntry.setChartOfAccountsCode(altAccount.getChartOfAccountsCode());
                Message err = handleExpiredClosedAccount(altAccount, laborOriginEntry, laborWorkingEntry,
                        universityRunDate);
                if (err == null) {
                    err = messageBuilderService.buildMessageWithPlaceHolder(
                            LaborKeyConstants.MESSAGE_FRINGES_MOVED_TO,
                            Message.TYPE_WARNING,
                            new Object[]{altAccount.getAccountNumber()}
                    );
                }
                return err;
            }

            // no alt acct, use suspense acct if active
            final boolean suspenseAccountLogicInd = parameterService.getParameterValueAsBoolean(LaborScrubberStep.class,
                    LaborParameterConstants.SUSPENSE_ACCOUNT_IND
            );
            if (suspenseAccountLogicInd) {
                return useSuspenseAccount(laborWorkingEntry);
            }

            return messageBuilderService.buildMessage(
                    LaborKeyConstants.ERROR_NON_FRINGE_ACCOUNT_ALTERNATIVE_NOT_FOUND,
                    Message.TYPE_FATAL
            );
        }

        return handleExpiredClosedAccount(account, laborOriginEntry, laborWorkingEntry, universityRunDate);
    }

    /**
     * This method changes account to suspenseAccount
     */
    protected Message useSuspenseAccount(final LaborOriginEntry workingEntry) {
        final String suspenseAccountNumber = parameterService.getParameterValueAsString(LaborScrubberStep.class,
                LaborParameterConstants.SUSPENSE_ACCOUNT);
        final String suspenseCOAcode = parameterService.getParameterValueAsString(LaborScrubberStep.class,
                LaborParameterConstants.SUSPENSE_CHART);
        final String suspenseSubAccountNumber = parameterService.getParameterValueAsString(LaborScrubberStep.class,
                LaborParameterConstants.SUSPENSE_SUB_ACCOUNT);

        final Account account = accountService.getByPrimaryId(suspenseCOAcode, suspenseAccountNumber);

        if (ObjectUtils.isNull(account)) {
            return messageBuilderService.buildMessage(
                    LaborKeyConstants.ERROR_INVALID_SUSPENSE_ACCOUNT,
                    Message.TYPE_FATAL
            );
        }

        workingEntry.setAccount(account);
        workingEntry.setAccountNumber(suspenseAccountNumber);
        workingEntry.setChartOfAccountsCode(suspenseCOAcode);
        workingEntry.setSubAccountNumber(suspenseSubAccountNumber);

        return messageBuilderService.buildMessageWithPlaceHolder(
                LaborKeyConstants.MESSAGE_SUSPENSE_ACCOUNT_APPLIED,
                Message.TYPE_WARNING,
                new Object[]{suspenseCOAcode, suspenseAccountNumber, suspenseSubAccountNumber}
        );
    }

    /**
     * Validates the sub account of the origin entry
     *
     * @param originEntry  the origin entry being scrubbed
     * @param workingEntry the scrubbed version of the origin entry
     * @return a Message if an error was encountered, otherwise null
     */
    protected Message validateSubAccount(
            final LaborOriginEntry originEntry, final LaborOriginEntry workingEntry,
            final LaborAccountingCycleCachingService laborAccountingCycleCachingService) {
        LOG.debug("validateSubAccount() started");

        // when continuationAccount used, the subAccountNumber should be changed to dashes and skip validation
        // subAccount process
        if (continuationAccountIndicator) {
            workingEntry.setSubAccountNumber(KFSConstants.getDashSubAccountNumber());
            return null;
        }

        // If the sub account number is empty, set it to dashes.
        // Otherwise set the workingEntry sub account number to the sub account number of the input origin entry.
        if (StringUtils.isNotBlank(originEntry.getSubAccountNumber())) {
            // sub account IS specified
            if (!KFSConstants.getDashSubAccountNumber().equals(originEntry.getSubAccountNumber())) {
                final SubAccount originEntrySubAccount = laborAccountingCycleCachingService.getSubAccount(
                        originEntry.getChartOfAccountsCode(), originEntry.getAccountNumber(),
                        originEntry.getSubAccountNumber());
                if (originEntrySubAccount == null) {
                    // sub account is not valid
                    return messageBuilderService.buildMessage(
                            KFSKeyConstants.ERROR_SUB_ACCOUNT_NOT_FOUND,
                            originEntry.getChartOfAccountsCode() + "-" + originEntry.getAccountNumber() + "-"
                            + originEntry.getSubAccountNumber(),
                            Message.TYPE_FATAL
                    );
                } else {
                    // sub account IS valid
                    if (originEntrySubAccount.isActive()) {
                        // sub account IS active
                        workingEntry.setSubAccountNumber(originEntry.getSubAccountNumber());
                    } else {
                        // sub account IS NOT active
                        if (parameterService.getParameterValueAsString(KfsParameterConstants.GENERAL_LEDGER_BATCH.class,
                                KFSConstants.SystemGroupParameterNames.GL_ANNUAL_CLOSING_DOC_TYPE).equals(
                                        originEntry.getFinancialDocumentTypeCode())) {
                            // document IS annual closing
                            workingEntry.setSubAccountNumber(originEntry.getSubAccountNumber());
                        } else {
                            // document is NOT annual closing
                            return messageBuilderService.buildMessage(
                                    KFSKeyConstants.ERROR_SUB_ACCOUNT_NOT_ACTIVE,
                                    originEntry.getChartOfAccountsCode() + "-" + originEntry.getAccountNumber() + "-"
                                    + originEntry.getSubAccountNumber(),
                                    Message.TYPE_FATAL
                            );
                        }
                    }
                }
            } else {
                // the sub account is dashes
                workingEntry.setSubAccountNumber(KFSConstants.getDashSubAccountNumber());
            }
        } else {
            // No sub account is specified.
            workingEntry.setSubAccountNumber(KFSConstants.getDashSubAccountNumber());
        }

        return null;
    }

    @Override
    public boolean isAccountExpired(final Account account, final UniversityDate universityRunDate) {
        return scrubberValidator.isAccountExpired(account, universityRunDate);
    }

    @Override
    public void validateForInquiry(final GeneralLedgerPendingEntry entry) {
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setConfigurationService(final ConfigurationService service) {
        kualiConfigurationService = service;
    }

    public void setAccountService(final AccountService as) {
        accountService = as;
    }

    public void setPersistenceService(final PersistenceService ps) {
        persistenceService = ps;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setBalanceTypService(final BalanceTypeService balanceTypService) {
        this.balanceTypService = balanceTypService;
    }

    public void setScrubberValidator(final ScrubberValidator sv) {
        scrubberValidator = sv;
    }

    public void setPersistenceStructureService(final PersistenceStructureService persistenceStructureService) {
        this.persistenceStructureService = persistenceStructureService;
    }

    public void setOptionsService(final OptionsService optionsService) {
        this.optionsService = optionsService;
    }

    public void setMessageBuilderService(final MessageBuilderService messageBuilderService) {
        this.messageBuilderService = messageBuilderService;
    }
}
